
package edu.uthscsa.ric.volume;

import java.io.File;
import java.nio.ByteBuffer;


/**
 * A class that implements WritableHeader can be registered with Mango to extend the types of image header formats that can be written. WritableHeader extends
 * ReadableHeader, hence any class that implements this interface must be able to both read and write the header.
 */
public interface WritableHeader extends ReadableHeader {

	/**
	 * Returns true if the format supports compressed data.
	 * 
	 * @return true if the format supports compressed data
	 */
	boolean canCompress();



	/**
	 * Returns a formatted image file (string) based on a header file (i.e., a header type might have a convention for naming the image file relative to header
	 * file name). It is possible that the header and image data are stored in the same file and so this method may simply return the file parameter.
	 * 
	 * @param headerFile the header file
	 * @return the formatted image file or null if the image file should not be written by the application
	 */
	File formatImageFile(File headerFile);



	/**
	 * Returns an array of byte counts that supported for a datatype.
	 * 
	 * @param type the datatype (see ImageType for possible values)
	 * @return an array of supported byte counts or null if none
	 */
	int[] getSupportedBytesForType(int type);



	/**
	 * Returns an array of valid data orientations in the format "XYZ+--" where the first character represents the fastest moving dimension and that third
	 * character represents the slowest moving dimension. The fourth through the sixth characters represent the sense of the dimensions. If this method returns
	 * null or an empty array, then it is assumed all orientations are supported.
	 * 
	 * @return an array of supported orientation strings or null if all orientations are supported
	 */
	String[] getSupportedOrientations();



	/**
	 * Returns an array of datatypes that are supported for a number of bytes.
	 * 
	 * @param bytes the number of bytes (possible values will be 1, 2, or 4)
	 * @return an array of supported datatypes or null if none
	 */
	int[] getSupportedTypesForBytes(int bytes);



	/**
	 * Returns true if this image format supports writing in big endian byte order.
	 * 
	 * @return true if this image format supports writing in big endian byte order
	 */
	boolean isSupportedBigEndian();



	/**
	 * Returns true if this image format supports a data scale intercept.
	 * 
	 * @return true if this image format supports a data scale intercept
	 */
	boolean isSupportedDataScaleIntercept();



	/**
	 * Returns true if this image format supports a data scale slope.
	 * 
	 * @return true if this image format supports a data scale slope
	 */
	boolean isSupportedDataScaleSlope();



	/**
	 * Returns true if this image format supports writing in little endian byte order.
	 * 
	 * @return true if this image format supports writing in little endian byte order
	 */
	boolean isSupportedLittleEndian();



	/**
	 * Returns true if this image format supports multiple data scale values, such as a data scale per slice or per volume.
	 * 
	 * @return true if this image format supports multiple data scale values
	 */
	boolean isSupportedMultipleDataScales();



	/**
	 * If for some reason the application should not attempt to write the image (i.e., the image is not stored as contiguous data), then this method should
	 * return true. Otherwise, it should return false
	 * 
	 * @return true if the plugin will write the image file, false if the application should write the image file
	 */
	boolean willWriteImage();



	/**
	 * Writes the header. Formats that precatenate the header data to the image file should return a byte array containing the header data instead of writing
	 * directly to the file.
	 * 
	 * @param id image dimensions
	 * @param vd voxel dimensions
	 * @param it image type
	 * @param ir image range
	 * @param imdes image description
	 * @param compress user selected compression option (if supported, plugin should write gzip compressed header file to disk)
	 * @param origin the origin voxel location relative to the native orientation of the data
	 * @param orientation the orientation (e.g., XYZ+--)
	 * @param transforms an array of image transforms (image or world) or null if there is none
	 * @param headerFile the file that this header will be written to
	 * @param imageFile image file (if same as header file, data should be returned as byte array)
	 * @param imageBuffers the image buffer (contains float data)
	 * @param tempDir directory to store temporary files, if necessary
	 * @return data that will be precatenated to the image file (e.g., the header) or null if there is none
	 * @throws Exception exception
	 */
	byte[] writeHeader(ImageDimensions id, VoxelDimensions vd, ImageType it, ImageRange ir, ImageDescription imdes, boolean compress, Coordinate origin,
			String orientation, ImageTransform[] transforms, File headerFile, File imageFile, ByteBuffer[] imageBuffers, File tempDir) throws Exception; // NOPMD



	/**
	 * Writes a series of headers. Formats that precatenate the header data to the image file should return a byte array containing the header data instead of
	 * writing directly to the file.
	 * 
	 * @param id image dimensions
	 * @param vd voxel dimensions
	 * @param it image type
	 * @param ir image range
	 * @param imdes image description
	 * @param compress user selected compression option (if supported, plugin should write gzip compressed header file to disk)
	 * @param origin the origin voxel location relative to the native orientation of the data
	 * @param orientation the orientation (e.g., XYZ+--)
	 * @param transforms an array of image transforms (image or world) or null if there is none
	 * @param headerFile the file that this header will be written to
	 * @param imageFile image file (if same as header file, data should be returned as byte array)
	 * @param imageBuffers the image buffer (contains float data)
	 * @param tempDir directory to store temporary files, if necessary
	 * @return data that will be precatenated to the image file (e.g., the header) or null if there is none
	 * @throws Exception exception
	 */
	byte[][] writeSeriesHeaders(ImageDimensions id, VoxelDimensions vd, ImageType it, ImageRange ir, ImageDescription imdes, boolean compress,
			Coordinate origin, String orientation, ImageTransform[] transforms, File headerFile, File imageFile, ByteBuffer[] imageBuffers, File tempDir)
			throws Exception; // NOPMD
}
