
package edu.uthscsa.ric.volume;

import edu.uthscsa.ric.utilities.MathUtilities;


/**
 * VoxelDimensions contains information related to voxel size in column/row/slice and x/y/z orientations, as well as temporal resolution and units.
 */
public class VoxelDimensions implements Cloneable {

	private double TE1; //NOPMD
	private double TE2; //NOPMD
	private double TR; //NOPMD
	private double colSize;
	private double rowSize;
	private double sliceThickness;
	private double xSize;
	private double ySize;
	private double zSize;
	private boolean flip;
	private float[] variableSpacing;
	private float[] variableTiming;
	private int spatialUnit;
	private int temporalUnit;

	public static final String[] UNIT_STRING = new String[49];
	public static final short UNITS_HZ = 32;
	public static final short UNITS_METER = 1;
	public static final short UNITS_MICRON = 3;
	public static final short UNITS_MM = 2;
	public static final short UNITS_MSEC = 16;
	public static final short UNITS_PPM = 40;
	public static final short UNITS_RADS = 48;
	public static final short UNITS_SEC = 8;
	public static final short UNITS_UNKNOWN = 0;
	public static final short UNITS_USEC = 24;

	static {
		UNIT_STRING[UNITS_UNKNOWN] = "Unknown Unit";
		UNIT_STRING[UNITS_METER] = "Meters";
		UNIT_STRING[UNITS_MM] = "Millimeters";
		UNIT_STRING[UNITS_MICRON] = "Microns";
		UNIT_STRING[UNITS_SEC] = "Seconds";
		UNIT_STRING[UNITS_MSEC] = "Milliseconds";
		UNIT_STRING[UNITS_USEC] = "Microseconds";
		UNIT_STRING[UNITS_HZ] = "Hertz";
		UNIT_STRING[UNITS_PPM] = "Parts-per-million";
		UNIT_STRING[UNITS_RADS] = "Radians-per-second";
	}



	/**
	 * The only class constructor.
	 * 
	 * @param colSize size of a voxel's column dimension
	 * @param rowSize size of a voxel's row dimension
	 * @param sliceThickness size of a voxel's slice dimension
	 * @param TR temporal resolution
	 */
	public VoxelDimensions(final double colSize, final double rowSize, final double sliceThickness, final double TR) { // NOPMD
		this.colSize = colSize;
		this.rowSize = rowSize;
		this.sliceThickness = sliceThickness;
		this.TR = TR;
	}



	/**
	 * Replaces any stored negative voxel sizes with their absolute value.
	 */
	public void clearNegativeVoxelSizes() {
		this.colSize = Math.abs(this.colSize);
		this.rowSize = Math.abs(this.rowSize);
		this.sliceThickness = Math.abs(this.sliceThickness);
	}



	/**
	 * Returns a copy of this object.
	 * 
	 * @return cloned object
	 * @throws CloneNotSupportedException clone not supported exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}



	/**
	 * Tests equality. Ignores TR, TE1, TE2.
	 * 
	 * @param obj object to compare
	 * @return true if the objects are equal, false otherwise
	 * @deprecated
	 */
	@Deprecated
	public boolean equals(final VoxelDimensions obj) { // NOPMD
		return equals((Object) obj);
	}



	/**
	 * Tests equality. Ignores TR, TE1, TE2.
	 * 
	 * @param obj object to compare
	 * @return true if the objects are equal, false otherwise
	 */
	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		final VoxelDimensions other = (VoxelDimensions) obj;
		if (!MathUtilities.essentiallyEqual(this.colSize, other.colSize)) {
			return false;
		}
		if (!MathUtilities.essentiallyEqual(this.rowSize, other.rowSize)) {
			return false;
		}
		if (!MathUtilities.essentiallyEqual(this.sliceThickness, other.sliceThickness)) {
			return false;
		}
		if (!MathUtilities.essentiallyEqual(this.xSize, other.xSize)) {
			return false;
		}
		if (!MathUtilities.essentiallyEqual(this.ySize, other.ySize)) {
			return false;
		}

		return (MathUtilities.essentiallyEqual(this.zSize, other.zSize));
	}



	/**
	 * Returns the area of a voxel in the axial plane.
	 * 
	 * @return area of a voxel in the axial plane
	 */
	public double getAxialArea() {
		return this.xSize * this.ySize;
	}



	/**
	 * Returns the size of a voxel's column dimension.
	 * 
	 * @param abs a negative value will be converted to positive, if true
	 * @return size of a voxel's column dimension
	 */
	public double getColSize(final boolean abs) {
		if (abs) {
			return Math.abs(this.colSize);
		}

		return this.colSize;
	}



	/**
	 * Returns the area of a voxel in the coronal plane.
	 * 
	 * @return area of a voxel in the coronal plane
	 */
	public double getCoronalArea() {
		return this.xSize * this.zSize;
	}



	/**
	 * Returns the size of a voxel's row dimension.
	 * 
	 * @param abs a negative value will be converted to positive, if true
	 * @return size of a voxel's row dimension
	 */
	public double getRowSize(final boolean abs) {
		if (abs) {
			return Math.abs(this.rowSize);
		}

		return this.rowSize;
	}



	/**
	 * Returns the area of a voxel in the sagittal plane.
	 * 
	 * @return area of a voxel in the sagittal plane
	 */
	public double getSagittalArea() {
		return this.ySize * this.zSize;
	}



	/**
	 * Returns the size of a voxel's slice dimension.
	 * 
	 * @param abs negative value will be converted to positive, if true
	 * @return size of a voxel's slice dimension
	 */
	public double getSliceThickness(final boolean abs) {
		if (abs) {
			return Math.abs(this.sliceThickness);
		}

		return this.sliceThickness;
	}



	/**
	 * Returns the spatial unit code.
	 * 
	 * @return spatial unit code
	 */
	public int getSpatialUnit() {
		return this.spatialUnit;
	}



	/**
	 * Returns a multiplier to convert units to mm.
	 * 
	 * @return a multiplier to convert units to mm
	 */
	public double getSpatialUnitMultiplier() {
		if (this.spatialUnit == UNITS_METER) {
			return .001;
		} else if (this.spatialUnit == UNITS_MICRON) {
			return 1000;
		} else {
			return 1;
		}
	}



	/**
	 * Returns the TE1.
	 * 
	 * @return size of TE1
	 */
	public double getTE1() {
		return this.TE1;
	}



	/**
	 * Returns the TE2.
	 * 
	 * @return size of TE2
	 */
	public double getTE2() {
		return this.TE2;
	}



	/**
	 * Returns the temporal unit code.
	 * 
	 * @return temporal unit code
	 */
	public int getTemporalUnit() {
		return this.temporalUnit;
	}



	/**
	 * Returns a multiplier to convert units to seconds.
	 * 
	 * @return a multiplier to convert units to seconds
	 */
	public double getTemporalUnitMultiplier() {
		if (this.temporalUnit == UNITS_MSEC) {
			return .001;
		} else if (this.temporalUnit == UNITS_USEC) {
			return .000001;
		} else {
			return 1;
		}
	}



	/**
	 * Returns the TR.
	 * 
	 * @return size of TR
	 */
	public double getTR() {
		return this.TR;
	}



	/**
	 * Returns an array of variable slice spacings. If this array is null, then it is assumed the slice spacings are constant across the volume and equal to
	 * sliceThickness.
	 * 
	 * @return the array
	 */
	public float[] getVariableSpacing() {
		return this.variableSpacing;
	}



	/**
	 * Returns an array of variable volume durations. If this array is null for series data, then it is assumed the volume durations are constant across the
	 * series and equals to TR.
	 * 
	 * @return the array
	 */
	public float[] getVariableTiming() {
		return this.variableTiming;
	}



	/**
	 * Returns the volume of a voxel.
	 * 
	 * @return volume of a voxel.
	 */
	public double getVolume() {
		return getRowSize(true) * getColSize(true) * getSliceThickness(true);
	}



	/**
	 * Returns the size of a voxel along the X dimension.
	 * 
	 * @return size of voxel along X dimension
	 */
	public double getXSize() {
		return this.xSize;
	}



	/**
	 * Returns the size of a voxel along the Y dimension.
	 * 
	 * @return size of voxel along Y dimension
	 */
	public double getYSize() {
		return this.ySize;
	}



	/**
	 * Returns the size of a voxel along the Z dimension.
	 * 
	 * @return size of voxel along Z dimension
	 */
	public double getZSize() {
		return this.zSize;
	}



	/**
	 * Compute hash code.
	 * 
	 * @return the hash code
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		long temp;
		temp = Double.doubleToLongBits(this.colSize);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(this.rowSize);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(this.sliceThickness);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(this.xSize);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(this.ySize);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(this.zSize);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		return result;
	}



	/**
	 * Use to verify that column, row, and slice sizes and TR have valid values.
	 * 
	 * @return true if valid
	 */
	public boolean isValid() {
		return isValid(null);
	}



	/**
	 * Use to verify that column, row, and slice sizes and TR have valid values.
	 * 
	 * @param buf stores any error messages
	 * @return true if valid
	 */
	public boolean isValid(final StringBuffer buf) {
		boolean valid = true;

		if (!(this.colSize != 0)) {
			if (buf != null) {
				buf.append("Column Size=" + this.colSize + "\n");
			}

			valid = false;
		}

		if (!(this.rowSize != 0)) {
			if (buf != null) {
				buf.append("Row Size=" + this.rowSize + "\n");
			}

			valid = false;
		}

		if (!(this.sliceThickness != 0)) {
			if (buf != null) {
				buf.append("Slice Thickness=" + this.sliceThickness + "\n");
			}

			valid = false;
		}

		if (!(this.TR >= 0)) {
			if (buf != null) {
				buf.append("TR=" + this.TR + "\n");
			}

			valid = false;
		}

		return valid;
	}



	/**
	 * Sets the negative state of voxel column size.
	 * 
	 * @param bool if true, column size will be set as negative; positive otherwise
	 */
	public void setColNegativeState(final boolean bool) {
		if (bool) {
			this.colSize = Math.abs(this.colSize) * -1;
		} else {
			this.colSize = Math.abs(this.colSize);
		}
	}



	/**
	 * Sets the size of a voxel along the column dimension.
	 * 
	 * @param val size of a voxel along the column dimension
	 */
	public void setColSize(final double val) {
		this.colSize = val;
	}



	/**
	 * Sets the negative state of voxel row size.
	 * 
	 * @param bool if true, row size will be set as negative; positive otherwise
	 */
	public void setRowNegativeState(final boolean bool) {
		if (bool) {
			this.rowSize = Math.abs(this.rowSize) * -1;
		} else {
			this.rowSize = Math.abs(this.rowSize);
		}
	}



	/**
	 * Sets the size of a voxel along the row dimension.
	 * 
	 * @param val size of a voxel along the row dimension
	 */
	public void setRowSize(final double val) {
		this.rowSize = val;
	}



	/**
	 * Sets the negative state of voxel slice size.
	 * 
	 * @param bool if true, slice size will be set as negative; positive otherwise
	 */
	public void setSliceNegativeState(final boolean bool) {
		if (bool) {
			this.sliceThickness = Math.abs(this.sliceThickness) * -1;
		} else {
			this.sliceThickness = Math.abs(this.sliceThickness);
		}
	}



	/**
	 * Sets the size of a voxel along the slice dimension.
	 * 
	 * @param val size of a voxel along the slice dimension
	 */
	public void setSliceThickness(final double val) {
		this.sliceThickness = val;
	}



	/**
	 * Sets the spatial unit of the voxel.
	 * 
	 * @param unit spatial unit code
	 */
	public void setSpatialUnit(final int unit) {
		this.spatialUnit = unit;
	}



	/**
	 * Sets the spatial unit of the voxel.
	 * 
	 * @param unit spatial unit name
	 */
	public void setSpatialUnit(final String unit) {
		loop: for (int ctr = 0; ctr < UNIT_STRING.length; ctr++) {
			if (UNIT_STRING[ctr] != null) {
				if (UNIT_STRING[ctr].equals(unit)) {
					this.spatialUnit = ctr;
					break loop;
				}
			}
		}
	}



	/**
	 * Sets the TE1.
	 * 
	 * @param num size of TE1
	 */
	public void setTE1(final double num) {
		this.TE1 = num;
	}



	/**
	 * Sets the TE2.
	 * 
	 * @param num size of TE2
	 */
	public void setTE2(final double num) {
		this.TE2 = num;
	}



	/**
	 * Sets the temporal unit of the voxel.
	 * 
	 * @param unit temporal unit code
	 */
	public void setTemporalUnit(final int unit) {
		this.temporalUnit = unit;
	}



	/**
	 * Sets the temporal unit of the voxel.
	 * 
	 * @param unit temporal unit name
	 */
	public void setTemporalUnit(final String unit) {
		loop: for (int ctr = 0; ctr < UNIT_STRING.length; ctr++) {
			if (UNIT_STRING[ctr] != null) {
				if (UNIT_STRING[ctr].equals(unit)) {
					this.temporalUnit = ctr;
					break loop;
				}
			}
		}
	}



	/**
	 * Sets the TR.
	 * 
	 * @param num size of TR
	 */
	public void setTR(final double num) {
		this.TR = num;
	}



	/**
	 * Set the values of this VoxelDimensions to that of another.
	 * 
	 * @param vd replacement VoxelDimensions object
	 */
	public void setValues(final VoxelDimensions vd) {
		setValues(vd, true);
	}



	/**
	 * Set the values of this VoxelDimensions to that of another.
	 * 
	 * @param vd replacement VoxelDimensions object
	 * @param updateXYZ xyz values will be replaced, if true
	 */
	public void setValues(final VoxelDimensions vd, final boolean updateXYZ) {
		this.rowSize = vd.rowSize;
		this.colSize = vd.colSize;
		this.sliceThickness = vd.sliceThickness;
		this.TR = vd.TR;
		this.TE1 = vd.TE1;
		this.TE2 = vd.TE2;
		this.spatialUnit = vd.spatialUnit;
		this.temporalUnit = vd.temporalUnit;

		if (updateXYZ) {
			this.xSize = vd.xSize;
			this.ySize = vd.ySize;
			this.zSize = vd.zSize;
		}
	}



	/**
	 * Sets an array of variable slice spacings.
	 * 
	 * @param variableSpacing the array
	 */
	public void setVariableSpacing(final float[] variableSpacing) {
		this.variableSpacing = variableSpacing;
	}



	/**
	 * Sets an array of variable volume durations.
	 * 
	 * @param variableTiming the array
	 */
	public void setVariableTiming(final float[] variableTiming) {
		this.variableTiming = variableTiming;
	}



	/**
	 * Sets the size of a voxel along the X dimension.
	 * 
	 * @param num size of voxel along the X dimension
	 */
	public void setXSize(final double num) {
		this.xSize = num;
	}



	/**
	 * Sets the size of a voxel along the Y dimension.
	 * 
	 * @param num size of voxel along the Y dimension
	 */
	public void setYSize(final double num) {
		this.ySize = num;
	}



	/**
	 * Sets the size of a voxel along the Z dimension.
	 * 
	 * @param num size of voxel along the Z dimension
	 */
	public void setZSize(final double num) {
		this.zSize = num;
	}



	/**
	 * Returns a string representation of this object.
	 * 
	 * @return a string representation of this object
	 */
	@Override
	public String toString() {
		return "VoxelDimensions [xSize=" + this.xSize + ", ySize=" + this.ySize + ", zSize=" + this.zSize + ", rowSize=" + this.rowSize + ", colSize="
				+ this.colSize + ", sliceThickness=" + this.sliceThickness + ", TR=" + this.TR + ", TE1=" + this.TE1 + ", TE2=" + this.TE2 + ", spatialUnit="
				+ this.spatialUnit + ", temporalUnit=" + this.temporalUnit + "]";
	}



	public boolean isFlip() {
		return this.flip;
	}



	public void setFlip(final boolean flip) {
		this.flip = flip;
	}
}
