
package edu.uthscsa.ric.volume;

import java.io.File;
import java.net.URI;
import java.nio.ByteBuffer;

import edu.uthscsa.ric.mango.MangoPluginType;


/**
 * A class that implements ReadableHeader can be registered with Mango to extend the types of image header formats that can be read.
 */
public interface ReadableHeader extends MangoPluginType {

	int ORIENTATION_CERTAINTY_HIGH = 2;
	int ORIENTATION_CERTAINTY_LOW = 0;
	int ORIENTATION_CERTAINTY_MID = 1;



	/**
	 * Return null to have the application handle reading the image data. To handle reading the data yourself (say if the data is not contiguous in the file),
	 * return a ByteBuffer which wraps a float array, one float value per voxel
	 * 
	 * @deprecated
	 * @return the image buffer or null
	 */
	@Deprecated
	ByteBuffer getImage();



	/**
	 * Return null to have the application handle reading the image data or return an array of ByteBuffers. Each ByteBuffer must be a FloatBuffer and represent
	 * a 3D volume. Multiple buffers represent a series.
	 * 
	 * @return the image buffer array or null
	 */
	ByteBuffer[] getImageBuffers();



	/**
	 * Returns the image description.
	 * 
	 * @return the image description
	 */
	ImageDescription getImageDescription();



	/**
	 * Returns the image dimensions.
	 * 
	 * @return the image dimensions
	 */
	ImageDimensions getImageDimensions();



	/**
	 * Returns the image file (could be the same as the header file).
	 * 
	 * @return the image file
	 */
	URI getImageFile();



	/**
	 * Returns the image range.
	 * 
	 * @return the image range
	 */
	ImageRange getImageRange();



	/**
	 * Returns the image type.
	 * 
	 * @return the image type
	 */
	ImageType getImageType();



	/**
	 * Returns the orientation (e.g., XYZ+--).
	 * 
	 * @return the orientation
	 */
	String getOrientation();



	/**
	 * Returns the orientation certainty (0=Unknown/Default, 1=Incomplete/Suspicious, 2=Complete/Sufficient).
	 * 
	 * @return the orientation certainty
	 */
	int getOrientationCertainty();



	/**
	 * Returns the origin voxel location relative to the native orientation of the data.
	 * 
	 * @return the origin
	 */
	Coordinate getOrigin();



	/**
	 * Returns the preferred file extension of this format.
	 * 
	 * @return the preferred file extension
	 */
	String getPreferredFileExtension();



	/**
	 * Return null to have the application handle reading the image preview. To create a preview image yourself, return a float array containing one float value
	 * per voxel. The float array length should be equal to the number of voxels in a representative slice.
	 * 
	 * @return the preview image
	 */
	float[] getPreviewImage();



	/**
	 * Returns an array of any transforms stored in the header.
	 * 
	 * @return an array of any transforms stored in the header (or null if none found)
	 */
	ImageTransform[] getTransforms();



	/**
	 * Returns the voxel dimensions.
	 * 
	 * @return the voxel dimensions
	 */
	VoxelDimensions getVoxelDimensions();



	/**
	 * Used to determine if a file is of this format (e.g., read magic number, check file size, check file extension, etc.).
	 * 
	 * @param file the file to read
	 * @return returns true if this file can be read by this format, false otherwise
	 */
	boolean isThisFormat(File file);



	/**
	 * Reads the file. The application expects that the image data is contiguous. If it is not, it can be processed into a contiguous file and stored in the
	 * provided temporary directory or getImage() can return an image buffer.
	 * 
	 * @param file the file to read
	 * @param tempDir directory to store temporary files
	 * @exception InvalidHeaderException An exception that should be thrown if the header is invalid or cannot be read successfully.
	 * @exception Exception Anything else that might have gone wrong.
	 */
	void readHeader(File file, File tempDir) throws InvalidHeaderException, Exception; // NOPMD



	/**
	 * Reads the URI. The application expects that the image data is contiguous.
	 * 
	 * @param uri the file to read
	 * @param tempDir directory to store temporary files (only useful for local files)
	 * @exception InvalidHeaderException An exception that should be thrown if the header is invalid or cannot be read successfully.
	 * @exception Exception Anything else that might have gone wrong.
	 */
	void readHeader(URI uri, File tempDir) throws InvalidHeaderException, Exception; // NOPMD



	/**
	 * Returns a formatted text representation of the image header (such as would be displayed in the 'Image Info' window).
	 * 
	 * @return image header as string
	 */
	@Override
	String toString();



	/**
	 * Returns a formatted text representation of the image header associated with the specified slice.
	 * 
	 * @param slice the slice number
	 * @param seriesPoint the series point
	 * @return image header as string
	 */
	String getSliceMetadata(int slice, int seriesPoint);



	/**
	 * Returns a bitmap of annotation data associated with the specified slice.
	 * 
	 * @param slice the slice number
	 * @param seriesPoint the series point
	 * @return a bitmap
	 */
	short[] getSliceAnnotation(int slice, int seriesPoint);
}
