
package edu.uthscsa.ric.volume;

/**
 * ImageType stores information related to data type and data mask.
 */
public class ImageType implements Cloneable {

	private ImageType nativeType;
	private boolean ascii;
	private boolean isRGBBySample;
	private boolean isRGBMode;
	private boolean isRGBPalette;
	private boolean littleEndian;
	private boolean readOnly;
	private int bitsStored;
	private int byteType;
	private int compression;
	private int numBytes;

	public static final int BYTE_TYPE_BINARY = 1;
	public static final int BYTE_TYPE_COMPLEX = 5;
	public static final int BYTE_TYPE_FLOAT = 4;
	public static final int BYTE_TYPE_INTEGER = 2;
	public static final int BYTE_TYPE_INTEGER_UNSIGNED = 3;
	public static final int BYTE_TYPE_RGB = 6;
	public static final int BYTE_TYPE_UNKNOWN = 0;

	public static final int COMPRESSION_TYPE_NONE = 0;
	public static final int COMPRESSION_TYPE_GZIP = 1;
	public static final int COMPRESSION_TYPE_DICOM = 2;
	public static final int COMPRESSION_TYPE_DICOM_JPEG_BASELINE = COMPRESSION_TYPE_DICOM | 4;
	public static final int COMPRESSION_TYPE_DICOM_JPEG_LOSSLESS = COMPRESSION_TYPE_DICOM | 8;
	public static final int COMPRESSION_TYPE_DICOM_JPEG2000 = COMPRESSION_TYPE_DICOM | 16;
	public static final int COMPRESSION_TYPE_DICOM_RLE = COMPRESSION_TYPE_DICOM | 32;



	/**
	 * Converts datatype from unsigned integer to signed integer. If not unsigned integer, nothing is changed.
	 * 
	 * @param it the ImageType object
	 * @return the same ImageType object
	 */
	public static ImageType convertToSigned(final ImageType it) {
		if (it.byteType == BYTE_TYPE_INTEGER_UNSIGNED) {
			it.byteType = BYTE_TYPE_INTEGER;
		}

		return it;
	}



	/**
	 * Returns the byte type name associated with a byte type code.
	 * 
	 * @param aByteType the byte type code
	 * @return byte type name
	 */
	public static String getByteTypeString(final int aByteType) {
		if (aByteType == BYTE_TYPE_INTEGER) {
			return "Signed Integer";
		} else if (aByteType == BYTE_TYPE_INTEGER_UNSIGNED) {
			return "Unsigned Integer";
		} else if (aByteType == BYTE_TYPE_FLOAT) {
			return "Float";
		} else if (aByteType == BYTE_TYPE_COMPLEX) {
			return "Complex";
		} else if (aByteType == BYTE_TYPE_RGB) {
			return "RGB";
		} else if (aByteType == BYTE_TYPE_BINARY) {
			return "Binary";
		} else if (aByteType == BYTE_TYPE_UNKNOWN) {
			return "Unknown";
		} else {
			return null;
		}
	}



	/**
	 * Returns an ImageType which is a 4-byte float, 32 bits stored, and big endian.
	 * 
	 * @return an ImageType of a work buffer
	 */
	public static ImageType getWorkBufferImageType() {
		return new ImageType(4, BYTE_TYPE_FLOAT, 32, false);
	}



	private static String convertByteTypeToString(final int bt) {
		if (bt == BYTE_TYPE_INTEGER) {
			return "Signed Integer";
		} else if (bt == BYTE_TYPE_INTEGER_UNSIGNED) {
			return "Unsigned Integer";
		} else if (bt == BYTE_TYPE_FLOAT) {
			return "Float";
		} else if (bt == BYTE_TYPE_COMPLEX) {
			return "Complex";
		} else if (bt == BYTE_TYPE_RGB) {
			return "RGB";
		} else if (bt == BYTE_TYPE_BINARY) {
			return "Binary";
		} else if (bt == BYTE_TYPE_UNKNOWN) {
			return "Unknown";
		} else {
			return null;
		}
	}



	/**
	 * The only class constructor.
	 * 
	 * @param numBytes number of bytes per voxel
	 * @param byteType byte type code
	 * @param bitsStored number of bits stored per voxel
	 * @param littleEndian byte order of the data
	 */
	public ImageType(final int numBytes, final int byteType, final int bitsStored, final boolean littleEndian) {
		this.numBytes = numBytes;
		this.byteType = byteType;
		this.bitsStored = bitsStored;
		this.littleEndian = littleEndian;
	}



	/**
	 * Returns a copy of this object.
	 * 
	 * @return cloned object
	 * @throws CloneNotSupportedException clone not supported exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}



	/**
	 * Returns a bit mask based on the number of bytes and bits stored per voxel.
	 * 
	 * @return bit mask
	 */
	public long createBitMask() {
		long mask = 0xFFFFFFFFFFFFFFFFL;
		mask >>>= (((8 - this.numBytes) * 8) + ((this.numBytes * 8) - this.bitsStored));

		if (this.byteType == BYTE_TYPE_INTEGER_UNSIGNED) {
			if (this.numBytes == 1) {
				mask &= 0x00000000000000FFL;
			} else if (this.numBytes == 2) {
				mask &= 0x000000000000FFFFL;
			} else if (this.numBytes == 4) {
				mask &= 0x00000000FFFFFFFFL;
			} else if (this.numBytes == 8) {
				mask = 0xFFFFFFFFFFFFFFFFL;
			}
		} else {
			mask = 0xFFFFFFFFFFFFFFFFL;
		}

		return mask;
	}



	/**
	 * Tests equality.
	 * 
	 * @param obj object to compare
	 * @return true if the objects are equals, false otherwise
	 */
	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		final ImageType other = (ImageType) obj;
		if (this.bitsStored != other.bitsStored) {
			return false;
		}
		if (this.byteType != other.byteType) {
			return false;
		}
		if (this.littleEndian != other.littleEndian) {
			return false;
		}

		return (this.numBytes == other.numBytes);
	}



	/**
	 * Returns the number of bits stored.
	 * 
	 * @return number of bits stored
	 */
	public int getBitsStored() {
		return this.bitsStored;
	}



	/**
	 * Returns the byte type code.
	 * 
	 * @return byte type code
	 */
	public int getByteType() {
		return this.byteType;
	}



	/**
	 * Returns the byte type name.
	 * 
	 * @return byte type name
	 */
	public String getByteTypeString() {
		return convertByteTypeToString(this.byteType);
	}



	public String getCompressionTypeString() {
		if (this.compression == COMPRESSION_TYPE_NONE) {
			return "None";
		} else if (this.compression == COMPRESSION_TYPE_GZIP) {
			return "GZIP";
		} else if (this.compression == COMPRESSION_TYPE_DICOM_JPEG_BASELINE) {
			return "JPEG (Baseline)";
		} else if (this.compression == COMPRESSION_TYPE_DICOM_JPEG_LOSSLESS) {
			return "JPEG (Lossless)";
		} else if (this.compression == COMPRESSION_TYPE_DICOM_JPEG2000) {
			return "JP2";
		} else if (this.compression == COMPRESSION_TYPE_DICOM_RLE) {
			return "Run Length Encoding";
		} else {
			return "Unknown";
		}
	}



	/**
	 * If image data was loaded as a different type than what is specified in the header, store the native type here.
	 * 
	 * @return the native image type
	 */
	public ImageType getNativeType() {
		return this.nativeType;
	}



	/**
	 * Returns the number of bytes per voxel.
	 * 
	 * @return number of bytes per voxel
	 */
	public int getNumBytesPerVoxel() {
		return this.numBytes;
	}



	/**
	 * Get the maximum value of this type.
	 * 
	 * @param forceAsUnsigned force method to consider type as unsigned
	 * @return max value
	 */
	public double getTypeMax(final boolean forceAsUnsigned) {
		double typeMax = 0;

		if ((this.numBytes == 1) && ((this.byteType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || forceAsUnsigned)) {
			typeMax = Byte.MAX_VALUE + Math.abs(Byte.MIN_VALUE);
		} else if ((this.numBytes == 1) && (this.byteType == ImageType.BYTE_TYPE_INTEGER)) {
			typeMax = Byte.MAX_VALUE;
		} else if ((this.numBytes == 2) && ((this.byteType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || forceAsUnsigned)) {
			typeMax = Short.MAX_VALUE + Math.abs(Short.MIN_VALUE);
		} else if ((this.numBytes == 2) && (this.byteType == ImageType.BYTE_TYPE_INTEGER)) {
			typeMax = Short.MAX_VALUE;
		} else if ((this.numBytes == 4) && ((this.byteType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || forceAsUnsigned)) {
			typeMax = Integer.MAX_VALUE;
		} else if ((this.numBytes == 4) && (this.byteType == ImageType.BYTE_TYPE_INTEGER)) {
			typeMax = Integer.MAX_VALUE;
		} else if ((this.numBytes == 4) && (this.byteType == ImageType.BYTE_TYPE_FLOAT)) {
			typeMax = Float.MAX_VALUE;
		}

		return typeMax;
	}



	/**
	 * Get the minimum value of this type.
	 * 
	 * @param forceAsUnsigned force method to consider type as unsigned
	 * @return min value
	 */
	public double getTypeMin(final boolean forceAsUnsigned) {
		return getTypeMin(forceAsUnsigned, false);
	}



	/**
	 * Get the minimum value of this type.
	 * 
	 * @param forceAsUnsigned force method to consider type as unsigned
	 * @param forceAsSigned force method to consider type as signed
	 * @return min value
	 */
	public double getTypeMin(final boolean forceAsUnsigned, final boolean forceAsSigned) {
		double typeMin = 0;

		if (forceAsUnsigned) {
			typeMin = 0;
		} else if ((this.numBytes == 1) && (forceAsSigned || (this.byteType == ImageType.BYTE_TYPE_INTEGER))) {
			typeMin = Byte.MIN_VALUE;
		} else if ((this.numBytes == 1) && (this.byteType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED)) {
			typeMin = 0;
		} else if ((this.numBytes == 2) && (forceAsSigned || (this.byteType == ImageType.BYTE_TYPE_INTEGER))) {
			typeMin = Short.MIN_VALUE;
		} else if ((this.numBytes == 2) && (this.byteType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED)) {
			typeMin = 0;
		} else if ((this.numBytes == 4) && (forceAsSigned || (this.byteType == ImageType.BYTE_TYPE_INTEGER))) {
			typeMin = Integer.MIN_VALUE;
		} else if ((this.numBytes == 4) && (this.byteType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED)) {
			typeMin = 0;
		} else if ((this.numBytes == 4) && (this.byteType == ImageType.BYTE_TYPE_FLOAT)) {
			typeMin = Float.MIN_VALUE;
		}

		return typeMin;
	}



	/**
	 * Computes hash code.
	 * 
	 * @return the hash code
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = (prime * result) + this.bitsStored;
		result = (prime * result) + this.byteType;
		result = (prime * result) + (this.littleEndian ? 1231 : 1237);
		result = (prime * result) + this.numBytes;
		return result;
	}



	/**
	 * Returns true if ascii state flag is set.
	 * 
	 * @return true if ascii state flag is set
	 */
	public boolean isAscii() {
		return this.ascii;
	}



	/**
	 * Returns true if the byte type is integer.
	 * 
	 * @return true if the byte type is integer
	 */
	public boolean isInteger() {
		return (this.byteType == BYTE_TYPE_INTEGER) || (this.byteType == BYTE_TYPE_INTEGER_UNSIGNED);
	}



	/**
	 * Returns the byte order state.
	 * 
	 * @return true if byte order is little endian; false otherwise
	 */
	public boolean isLittleEndian() {
		return this.littleEndian;
	}



	public boolean isRGB() {
		return (this.byteType == BYTE_TYPE_RGB);
	}



	/**
	 * Returns true if RGB data is stored by sample, rather than location.
	 * 
	 * @return true if RGB data is stored by sample
	 */
	public boolean isRGBBySample() {
		return this.isRGBBySample;
	}



	/**
	 * Returns true if the RGB data is to interpreted as color rather than intensity.
	 * 
	 * @return true if the RGB data is to interpreted as color
	 */
	public boolean isRGBMode() {
		return this.isRGBMode;
	}



	/**
	 * Returns true if this data uses a RGB palette.
	 * 
	 * @return true if this data uses a RGB palette
	 */
	public boolean isRGBPalette() {
		return this.isRGBPalette;
	}



	/**
	 * Use to verify that values are valid.
	 * 
	 * @return true if valid
	 */
	public boolean isValid() {
		return (((this.numBytes == 1) || (this.numBytes == 2) || (this.numBytes == 4) || (this.numBytes == 8))
				&& ((this.byteType == BYTE_TYPE_INTEGER) || (this.byteType == BYTE_TYPE_INTEGER_UNSIGNED) || (this.byteType == BYTE_TYPE_FLOAT))
				&& (this.bitsStored > 0) && (this.bitsStored <= (this.numBytes * 8)));
	}



	/**
	 * Returns true if this ImageType is valid, false otherwise. Any error messages will be stored in the specified StringBuffer.
	 * 
	 * @param buf StringBuffer to hold messages
	 * @return true if the ImageType is valid, false otherwise
	 */
	public boolean isValid(final StringBuffer buf) {
		boolean valid = true;

		if (!(((this.numBytes == 1) || (this.numBytes == 2) || (this.numBytes == 4) || (this.numBytes == 8)
				|| ((this.numBytes == 3) && (this.byteType == BYTE_TYPE_RGB))))) {
			if (buf != null) {
				buf.append("Num Bytes=" + this.numBytes + "\n");
			}

			valid = false;
		}

		if (!((this.byteType == BYTE_TYPE_INTEGER) || (this.byteType == BYTE_TYPE_INTEGER_UNSIGNED) || (this.byteType == BYTE_TYPE_FLOAT)
				|| (this.byteType == BYTE_TYPE_RGB))) {
			if (buf != null) {
				buf.append("Byte Type=" + getByteTypeString(this.byteType) + "\n");
			}

			valid = false;
		}

		if (!(this.bitsStored > 0) && (this.bitsStored <= (this.numBytes * 8))) {
			if (buf != null) {
				buf.append("Bits Stored=" + this.bitsStored + "\n");
			}

			valid = false;
		}

		return valid;
	}



	/**
	 * Set the ASCII state flag.
	 * 
	 * @param bool true if data is in ASCII form, false otherwise
	 */
	public void setAscii(final boolean bool) {
		this.ascii = bool;
	}



	/**
	 * Set the native image type.
	 * 
	 * @param nativeType the native image type
	 */
	public void setNativeType(final ImageType nativeType) {
		this.nativeType = nativeType;
	}



	/**
	 * Sets whether RGB data is grouped by location or sample.
	 * 
	 * @param isRGBBySample true if RGB data is grouped by sample, false otherwise
	 */
	public void setRGBBySample(final boolean isRGBBySample) {
		this.isRGBBySample = isRGBBySample;
	}



	/**
	 * Set to true to interpret this RGB data as color rather than intensity
	 * 
	 * @param isRGBMode true to interpret RGB data as color
	 */
	public void setRGBMode(final boolean isRGBMode) {
		this.isRGBMode = isRGBMode;
	}



	/**
	 * Sets to true if this data uses a RGB palette.
	 * 
	 * @param isRGBPalette true if this data uses a RGB palette, false otherwise
	 */
	public void setRGBPalette(final boolean isRGBPalette) {
		this.isRGBPalette = isRGBPalette;
	}



	/**
	 * Set the values of this ImageType to that of another.
	 * 
	 * @param it replacement ImageType object
	 */
	public void setValues(final ImageType it) {
		this.numBytes = it.numBytes;
		this.byteType = it.byteType;
		this.bitsStored = it.bitsStored;
		this.littleEndian = it.littleEndian;

	}



	/**
	 * Returns a string representation of this object.
	 * 
	 * @return a string representation of this object
	 */
	@Override
	public String toString() {
		return "ImageType [numBytes=" + this.numBytes + ", byteType=" + this.byteType + ", bitsStored=" + this.bitsStored + ", littleEndian="
				+ this.littleEndian + ", ascii=" + this.ascii + "]";
	}



	/**
	 * Returns true if this data was compressed.
	 * 
	 * @return true if this data was compressed
	 */
	public boolean isCompressed() {
		return (this.compression != COMPRESSION_TYPE_NONE);
	}



	/**
	 * Returns the compression type.
	 * 
	 * @return the compression type
	 */
	public int getCompressionType() {
		return this.compression;
	}



	/**
	 * Sets the compression type.
	 * 
	 * @param compression the compression type
	 */
	public void setCompressionType(final int compression) {
		this.compression = compression;
	}



	public boolean isReadOnly() {
		return this.readOnly;
	}



	public void setReadOnly(final boolean readOnly) {
		this.readOnly = readOnly;
	}
}
