
package edu.uthscsa.ric.volume;

import java.util.Arrays;


/**
 * ImageRange stores information related to the value range of the image and value scaling.
 */
public class ImageRange implements Cloneable {

	private double displayMax;
	private double displayMin;
	private double imageMax;
	private double imageMin;
	private float[] intercepts;
	private float[] slopes;
	private long imageMaxOffset;
	private long imageMinOffset;

	public static final double DEFAULT_DATA_SCALE_INTERCEPT = 0;
	public static final double DEFAULT_DATA_SCALE_SLOPE = 1.0;



	/**
	 * Checks if all data scale intercepts are default.
	 * 
	 * @param someIntercepts the data scale intercepts to check
	 * @return true if all data scale intercepts are default, false otherwise
	 */
	public static boolean hasDefaultDataScaleIntercept(final float someIntercepts[]) {
		if (someIntercepts == null) {
			return false;
		}

		for (final float someIntercept : someIntercepts) {
			if ((someIntercept != someIntercepts[0]) || (someIntercept != 0.0)) {
				return false;
			}
		}

		return true;
	}



	/**
	 * Checks if all data scale slopes are default.
	 * 
	 * @param someSlopes the data scale slopes to check
	 * @return true if all data scale slopes are default, false otherwise
	 */
	public static boolean hasDefaultDataScaleSlope(final float[] someSlopes) {
		if (someSlopes == null) {
			return false;
		}

		for (final float someSlope : someSlopes) {
			if ((someSlope != someSlopes[0]) || (someSlope != 1.0)) {
				return false;
			}
		}

		return true;
	}



	/**
	 * The only class constructor.
	 * 
	 * @param iMin the image min
	 * @param iMax the image max
	 * @param dMin the display min
	 * @param dMax the display max
	 */
	public ImageRange(final double iMin, final double iMax, final double dMin, final double dMax) {
		this.imageMin = iMin;
		this.imageMax = iMax;
		this.displayMin = dMin;
		this.displayMax = dMax;
	}



	/**
	 * Returns a copy of this object.
	 * 
	 * @return cloned object
	 * @throws CloneNotSupportedException clone not supported exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		final Object obj = super.clone();
		final ImageRange ir = (ImageRange) obj;
		ir.slopes = this.slopes.clone();
		ir.intercepts = this.intercepts.clone();

		return ir;
	}



	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		final ImageRange other = (ImageRange) obj;
		if (Double.doubleToLongBits(this.displayMax) != Double.doubleToLongBits(other.displayMax)) {
			return false;
		}
		if (Double.doubleToLongBits(this.displayMin) != Double.doubleToLongBits(other.displayMin)) {
			return false;
		}
		if (Double.doubleToLongBits(this.imageMax) != Double.doubleToLongBits(other.imageMax)) {
			return false;
		}
		if (Double.doubleToLongBits(this.imageMin) != Double.doubleToLongBits(other.imageMin)) {
			return false;
		}
		if (!Arrays.equals(this.intercepts, other.intercepts)) {
			return false;
		}

		return (Arrays.equals(this.slopes, other.slopes));
	}



	/**
	 * Returns the data scale intercepts.
	 * 
	 * @return the data scale intercepts
	 */
	public float[] getDataScaleIntercepts() {
		return this.intercepts;
	}



	/**
	 * Returns the data scale slopes.
	 * 
	 * @return the data scale slopes
	 */
	public float[] getDataScaleSlopes() {
		return this.slopes;
	}



	/**
	 * Returns the display max.
	 * 
	 * @return the display max
	 */
	public double getDisplayMax() {
		return this.displayMax;
	}



	/**
	 * Returns the display min.
	 * 
	 * @return the display min
	 */
	public double getDisplayMin() {
		return this.displayMin;
	}



	/**
	 * Returns the image max.
	 * 
	 * @return the image max
	 */
	public double getImageMax() {
		return this.imageMax;
	}



	/**
	 * Returns the offset of the maximum voxel.
	 * 
	 * @return the offset of the maximum voxel
	 */
	public long getImageMaxOffset() {
		return this.imageMaxOffset;
	}



	/**
	 * Returns the image min.
	 * 
	 * @return the image min
	 */
	public double getImageMin() {
		return this.imageMin;
	}



	/**
	 * Returns the offset of the minimum voxel.
	 * 
	 * @return the offset of the minimum voxel
	 */
	public long getImageMinOffset() {
		return this.imageMinOffset;
	}



	/**
	 * Checks if all data scale intercepts are default.
	 * 
	 * @return true if all data scale intercepts are default, false otherwise
	 */
	public boolean hasDefaultDataScaleIntercept() {
		return hasDefaultDataScaleIntercept(this.intercepts);
	}



	/**
	 * Checks if all data scale slopes are default.
	 * 
	 * @return true if all data scale slopes are default, false otherwise
	 */
	public boolean hasDefaultDataScaleSlope() {
		return hasDefaultDataScaleSlope(this.slopes);
	}



	/**
	 * Checks if all data scale intercepts are the same.
	 * 
	 * @return true if all data scale intercepts are the same, false otherwise
	 */
	public boolean hasGlobalDataScaleIntercept() {
		if (this.intercepts == null) {
			return false;
		}

		for (final float intercept : this.intercepts) {
			if (intercept != this.intercepts[0]) {
				return false;
			}
		}

		return true;
	}



	/**
	 * Checks if all data scale slopes are the same.
	 * 
	 * @return true if all data scale slopes are the same, false otherwise
	 */
	public boolean hasGlobalDataScaleSlope() {
		if (this.slopes == null) {
			return false;
		}

		for (final float slope : this.slopes) {
			if (slope != this.slopes[0]) {
				return false;
			}
		}

		return true;
	}



	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		long temp;
		temp = Double.doubleToLongBits(this.displayMax);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(this.displayMin);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(this.imageMax);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(this.imageMin);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		result = (prime * result) + Arrays.hashCode(this.intercepts);
		result = (prime * result) + Arrays.hashCode(this.slopes);
		return result;
	}



	/**
	 * Sets the data scale intercepts.
	 * 
	 * @param val the data scale intercepts
	 */
	public void setDataScaleIntercepts(final float[] val) {
		this.intercepts = val;
	}



	/**
	 * Sets the data scale slopes.
	 * 
	 * @param val the data scale slopes
	 */
	public void setDataScaleSlopes(final float[] val) {
		this.slopes = val;
	}



	/**
	 * Sets the display max.
	 * 
	 * @param val the display max
	 */
	public void setDisplayMax(final double val) {
		this.displayMax = val;
	}



	/**
	 * Sets the display min.
	 * 
	 * @param val the display min
	 */
	public void setDisplayMin(final double val) {
		this.displayMin = val;
	}



	/**
	 * Sets global data scale intercept.
	 * 
	 * @param num the number of slices
	 * @param val the data scale intercept
	 */
	public void setGlobalDataScaleIntercept(final int num, final float val) {
		this.intercepts = new float[num];

		for (int ctr = 0; ctr < num; ctr++) {
			this.intercepts[ctr] = val;
		}
	}



	/**
	 * Sets global data scale slope.
	 * 
	 * @param num the number of slices
	 * @param val the data scale slope
	 */
	public void setGlobalDataScaleSlope(final int num, final float val) {
		this.slopes = new float[num];

		for (int ctr = 0; ctr < num; ctr++) {
			this.slopes[ctr] = val;
		}
	}



	/**
	 * Sets the image max.
	 * 
	 * @param val the image max
	 */
	public void setImageMax(final double val) {
		this.imageMax = val;
	}



	/**
	 * Sets the offset of the maximum voxel.
	 * 
	 * @param imageMaxOffset offset
	 */
	public void setImageMaxOffset(final long imageMaxOffset) {
		this.imageMaxOffset = imageMaxOffset;
	}



	/**
	 * Sets the image min.
	 * 
	 * @param val the image min
	 */
	public void setImageMin(final double val) {
		this.imageMin = val;
	}



	/**
	 * Sets the offset of the minimum voxel.
	 * 
	 * @param imageMinOffset offset
	 */
	public void setImageMinOffset(final long imageMinOffset) {
		this.imageMinOffset = imageMinOffset;
	}



	/**
	 * Returns a string representation of this object.
	 * 
	 * @return a string represnetation of this object
	 */
	@Override
	public String toString() {
		return "ImageRange [imageMin=" + this.imageMin + ", imageMax=" + this.imageMax + ", displayMin=" + this.displayMin + ", displayMax=" + this.displayMax
				+ ", slopes=" + Arrays.toString(this.slopes) + ", intercepts=" + Arrays.toString(this.intercepts) + ", imageMinOffset=" + this.imageMinOffset
				+ ", imageMaxOffset=" + this.imageMaxOffset + "]";
	}
}
