
package edu.uthscsa.ric.volume;

import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;


/**
 * ImageDescription contains textual information describing the image.
 */
public class ImageDescription implements Cloneable {

	private Date studyDate;
	private Map<Long, String> dicomItems;
	private String description;
	private String imageTypeDescription;
	private String scanDate;
	private String scanID;
	private String scanName;
	private String scanTime;
	private String title;
	private URI file;
	private URI[] associatedFiles;
	private boolean edited;
	public final SimpleDateFormat DATETIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	public final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");
	public final SimpleDateFormat TIME_FORMAT = new SimpleDateFormat("HH:mm:ss");



	/**
	 * The default class constructor.
	 */
	public ImageDescription() {
		this.scanID = this.scanName = this.scanTime = this.scanDate = this.description = "";
	}



	/**
	 * Class constructor. Arguments include common textual information found in image headers.
	 * 
	 * @param id the scan ID
	 * @param name the name associated with the scan (e.g., patient name)
	 * @param time the scan time
	 * @param date the scan date
	 * @param des any further description
	 * @deprecated
	 */
	@Deprecated
	public ImageDescription(final String id, final String name, final String time, final String date, final String des) {
		this.scanID = id;
		this.scanName = name;
		this.scanTime = time;
		this.scanDate = date;
		this.description = des;
	}



	/**
	 * Class constructor. Arguments include common textual information found in image headers.
	 * 
	 * @param studyID the study ID
	 * @param studyName the study name
	 * @param studyDate the scan date
	 * @param studyDes the study description
	 */
	public ImageDescription(final String studyID, final String studyName, final Date studyDate, final String studyDes) {
		this.scanID = studyID;
		this.scanName = studyName;
		this.studyDate = studyDate;
		this.description = studyDes;
	}



	/**
	 * Clears this data.
	 */
	public void anonymize() {
		this.dicomItems = null;
		this.scanID = this.scanName = this.scanTime = this.scanDate = this.description = "";
	}



	/**
	 * Returns a copy of this object.
	 * 
	 * @return clone object
	 * @throws CloneNotSupportedException clone not supported exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}



	/**
	 * Returns both ID and name.
	 * 
	 * @return ID and name
	 */
	public String getAllID() {
		return this.scanID + " " + this.scanName;
	}



	/**
	 * Returns both time and date.
	 * 
	 * @return time and date
	 * @deprecated
	 */
	@Deprecated
	public String getAllTime() {
		return this.scanDate + " " + this.scanTime;
	}



	/**
	 * Returns an array of files associated with this image (e.g., a multifile image format).
	 * 
	 * @return an array of files
	 */
	public URI[] getAssociatedFiles() {
		return this.associatedFiles;
	}



	/**
	 * Returns the scan date.
	 * 
	 * @return the scan date
	 * @deprecated
	 */
	@Deprecated
	public String getDate() {
		return this.scanDate;
	}



	/**
	 * Returns the description.
	 * 
	 * @return the description
	 */
	public String getDescription() {
		return this.description;
	}



	/**
	 * Returns a map of DICOM tag id to string value (see setDicomItems() for more explanation).
	 * 
	 * @return a map of dicom tags to data
	 */
	public Map<Long, String> getDicomItems() {
		return this.dicomItems;
	}



	/**
	 * Returns the URI of this image.
	 * 
	 * @return the URI
	 */
	public URI getFile() {
		return this.file;
	}



	/**
	 * Returns the scan ID.
	 * 
	 * @return the scan ID
	 */
	public String getID() {
		return this.scanID;
	}



	/**
	 * Returns the scan name.
	 * 
	 * @return the scan name
	 */
	public String getName() {
		return this.scanName;
	}



	/**
	 * Returns the scan time.
	 * 
	 * @return the scan time
	 * @deprecated
	 */
	@Deprecated
	public String getTime() {
		return this.scanTime;
	}



	/**
	 * Returns a nice title for this image.
	 * 
	 * @return a nice title for this image
	 */
	public String getTitle() {
		return this.title;
	}



	/**
	 * Returns true if this object has been edited since construction.
	 * 
	 * @return true if edited, false otherwise
	 */
	public boolean hasBeenEdited() {
		return this.edited;
	}



	/**
	 * Sets an array of files associated with this image.
	 * 
	 * @param associatedFiles an array of file
	 */
	public void setAssociatedFiles(final URI[] associatedFiles) {
		this.associatedFiles = associatedFiles;
	}



	/**
	 * Sets the scan date.
	 * 
	 * @param date the scan date
	 * @deprecated
	 */
	@Deprecated
	public void setDate(final String date) {
		this.scanDate = date;
		this.edited = true;
	}



	/**
	 * Sets the description.
	 * 
	 * @param des the description
	 */
	public void setDescription(final String des) {
		this.description = des;
		this.edited = true;
	}



	/**
	 * Sets a map of DICOM tag id to string value. This map in intended to serve as a general way to store metadata information among different image format
	 * header types. The long map key is a concatenation of two integers corresponding to the DICOM tag group and element values. The value is a string
	 * representation of the DICOM tag value.
	 * 
	 * @param dicomItems the DICOM map
	 */
	public void setDicomItems(final Map<Long, String> dicomItems) {
		this.dicomItems = dicomItems;
	}



	/**
	 * Sets the URI of this image.
	 * 
	 * @param file the URI
	 */
	public void setFile(final URI file) {
		this.file = file;
	}



	/**
	 * Sets the scan ID.
	 * 
	 * @param id the scan ID
	 */
	public void setID(final String id) {
		this.scanID = id;
		this.edited = true;
	}



	/**
	 * Sets the scan name.
	 * 
	 * @param name the scan name
	 */
	public void setName(final String name) {
		this.scanName = name;
		this.edited = true;
	}



	/**
	 * Sets the scan time.
	 * 
	 * @param time the scan time
	 * @deprecated
	 */
	@Deprecated
	public void setTime(final String time) {
		this.scanTime = time;
		this.edited = true;
	}



	/**
	 * Sets a nice title for this image.
	 * 
	 * @param title
	 */
	public void setTitle(final String title) {
		this.title = title;
	}



	/**
	 * Returns a string representation of this object.
	 * 
	 * @return a string representation of this object
	 */
	@Override
	public String toString() {
		return "ImageDescription [dicomItems=" + this.dicomItems + ", scanID=" + this.scanID + ", scanName=" + this.scanName + ", scanTime=" + this.scanTime
				+ ", scanDate=" + this.scanDate + ", description=" + this.description + ", edited=" + this.edited + "]";
	}



	/**
	 * Returns a description of the image type.
	 * 
	 * @return a description of the image type
	 */
	public String getImageTypeDescription() {
		return this.imageTypeDescription;
	}



	/**
	 * Sets the image type description.
	 * 
	 * @param imageTypeDescription the image type description
	 */
	public void setImageTypeDescription(final String imageTypeDescription) {
		this.imageTypeDescription = imageTypeDescription;
	}



	/**
	 * Sets the study date.
	 * 
	 * @param studyDate the study date
	 */
	public void setStudyDate(final Date studyDate) {
		this.studyDate = studyDate;
	}



	/**
	 * Returns the study date.
	 * 
	 * @return the study date
	 */
	public Date getStudyDate() {
		return this.studyDate;
	}



	public String getStudyDateString() {
		if (this.studyDate != null) {
			return this.DATE_FORMAT.format(this.studyDate);
		} else if (this.scanDate != null) {
			return this.scanDate;
		} else {
			return "";
		}
	}



	public String getStudyTimeString() {
		if (this.studyDate != null) {
			return this.TIME_FORMAT.format(this.studyDate);
		} else if (this.scanTime != null) {
			return this.scanTime;
		} else {
			return "";
		}
	}



	public String getStudyDateTimeString() {
		if (this.studyDate != null) {
			return this.DATETIME_FORMAT.format(this.studyDate);
		} else {
			return getAllTime();
		}
	}
}
