
package edu.uthscsa.ric.volume;

/**
 * Stores image bounds.
 * 
 */
public class ImageBounds {

	private boolean valuesSet;
	private int tMax;
	private int tMin;
	private int xMax;
	private int xMin;
	private int yMax;
	private int yMin;
	private int zMax;
	private int zMin;



	/**
	 * Default constructor.
	 */
	public ImageBounds() {}



	/**
	 * Copy constructor.
	 * 
	 * @param bounds the ImageBounds to copy
	 */
	public ImageBounds(final ImageBounds bounds) {
		setValues(bounds.xMin, bounds.xMax, bounds.yMin, bounds.yMax, bounds.zMin, bounds.zMax, bounds.tMin, bounds.tMax);
	}



	public ImageBounds(final int minT, final int maxT) {
		setRangeT(minT, maxT);
	}



	/**
	 * Constructor.
	 * 
	 * @param minX the minimum x index
	 * @param maxX the maximum x index
	 * @param minY the minimum y index
	 * @param maxY the maximum y index
	 * @param minZ the minimum z index
	 * @param maxZ the maximum z index
	 */
	public ImageBounds(final int minX, final int maxX, final int minY, final int maxY, final int minZ, final int maxZ) {
		setValues(minX, maxX, minY, maxY, minZ, maxZ);
	}



	/**
	 * Returns T upper bound.
	 * 
	 * @return T upper bound
	 */
	public int getMaxT() {
		return this.tMax;
	}



	/**
	 * Returns X upper bound.
	 * 
	 * @return X upper bound
	 */
	public int getMaxX() {
		return this.xMax;
	}



	/**
	 * Returns Y upper bound.
	 * 
	 * @return Y upper bound
	 */
	public int getMaxY() {
		return this.yMax;
	}



	/**
	 * Returns Z upper bound.
	 * 
	 * @return Z upper bound
	 */
	public int getMaxZ() {
		return this.zMax;
	}



	/**
	 * Returns T lower bound.
	 * 
	 * @return T lower bound
	 */
	public int getMinT() {
		return this.tMin;
	}



	/**
	 * Returns X lower bound.
	 * 
	 * @return X lower bound
	 */
	public int getMinX() {
		return this.xMin;
	}



	/**
	 * Returns Y lower bound.
	 * 
	 * @return Y lower bound
	 */
	public int getMinY() {
		return this.yMin;
	}



	/**
	 * Returns Z lower bound.
	 * 
	 * @return Z lower bound
	 */
	public int getMinZ() {
		return this.zMin;
	}



	/**
	 * Returns true of the specified location is within bounds.
	 * 
	 * @param xLoc X location
	 * @param yLoc Y location
	 * @param zLoc Z location
	 * @return true of the specified location is within bounds
	 */
	public boolean inBounds(final int xLoc, final int yLoc, final int zLoc) {
		return ((xLoc >= this.xMin) && (xLoc <= this.xMax) && (yLoc >= this.yMin) && (yLoc <= this.yMax) && (zLoc >= this.zMin) && (zLoc <= this.zMax));
	}



	/**
	 * Returns true if the upper and lower bounds of X, Y, or Z are the same.
	 * 
	 * @return true if the upper and lower bounds of X, Y, or Z are the same
	 */
	public boolean isSingleSlice() {
		return ((this.xMin == this.xMax) || (this.yMin == this.yMax) || (this.zMin == this.zMax));
	}



	/**
	 * Set series bounds.
	 * 
	 * @param minT the minimum series index
	 * @param maxT the maximum series index
	 */
	public void setRangeT(final int minT, final int maxT) {
		this.tMin = minT;
		this.tMax = maxT;
	}



	/**
	 * Set X bounds.
	 * 
	 * @param minX the minimum x index
	 * @param maxX the maximum x index
	 */
	public void setRangeX(final int minX, final int maxX) {
		this.xMin = minX;
		this.xMax = maxX;
	}



	/**
	 * Set Y bounds.
	 * 
	 * @param minY the minimum y index
	 * @param maxY the minimum y index
	 */
	public void setRangeY(final int minY, final int maxY) {
		this.yMin = minY;
		this.yMax = maxY;
	}



	/**
	 * Set Z bounds.
	 * 
	 * @param minZ the minimum z index
	 * @param maxZ the maximum z index
	 */
	public void setRangeZ(final int minZ, final int maxZ) {
		this.zMin = minZ;
		this.zMax = maxZ;
	}



	/**
	 * Set XYZ bounds.
	 * 
	 * @param minX the minimum x index
	 * @param maxX the maximum x index
	 * @param minY the minimum y index
	 * @param maxY the minimum y index
	 * @param minZ the minimum z index
	 * @param maxZ the maximum z index
	 */
	public void setValues(final int minX, final int maxX, final int minY, final int maxY, final int minZ, final int maxZ) {
		setRangeX(minX, maxX);
		setRangeY(minY, maxY);
		setRangeZ(minZ, maxZ);
		this.valuesSet = true;
	}



	/**
	 * Set XYZT bounds.
	 * 
	 * @param minX the minimum x index
	 * @param maxX the maximum x index
	 * @param minY the minimum y index
	 * @param maxY the minimum y index
	 * @param minZ the minimum z index
	 * @param maxZ the maximum z index
	 * @param minT the minimum series index
	 * @param maxT the maximum series index
	 */
	public void setValues(final int minX, final int maxX, final int minY, final int maxY, final int minZ, final int maxZ, final int minT, final int maxT) {
		setValues(minX, maxX, minY, maxY, minZ, maxZ);
		setRangeT(minT, maxT);
	}



	/**
	 * Returns a string representation of this object.
	 * 
	 * @return a string representation of this object
	 */
	@Override
	public String toString() {
		return ("[" + this.xMin + ", " + this.xMax + "] [" + this.yMin + ", " + this.yMax + "] [" + this.zMin + ", " + this.zMax + "] [" + this.tMin + ", "
				+ this.tMax + "]");
	}



	/**
	 * Returns true if setValues() has been called.
	 * 
	 * @return true if setValues() has been called.
	 */
	public boolean valuesHaveBeenSet() {
		return this.valuesSet;
	}
}
