
package edu.uthscsa.ric.visualization.surface;

import java.awt.image.BufferedImage;

import javax.swing.JFrame;
import javax.vecmath.Point3d;

import com.jogamp.opengl.awt.GLCanvas;

import edu.uthscsa.ric.mango.BasicController;
import edu.uthscsa.ric.visualization.surface.primitives.Surface;
import edu.uthscsa.ric.volume.Coordinate;


public interface SurfaceController {

	/**
	 * Registers this plugin instance so that its display method will be called.
	 * 
	 * @param plugin the plugin
	 */
	void addToDisplay(SurfacePlugin plugin);



	/**
	 * Returns a BufferedImage of the specified surface.
	 * 
	 * @param surface the surface to capture or null to capture all
	 * @param showLinesAndPoints true to include added lines and points in capture, false otherwise
	 * @return the capture as a BufferedImage
	 */
	BufferedImage captureImage(Surface surface, boolean showLinesAndPoints);



	/**
	 * Convert viewer coordinate to surface coordinate.
	 * 
	 * @param coor the viewer coordinate
	 * @param isWorld true if the coordinate is in world space
	 * @return the surface point
	 */
	Point3d convertViewerCoordinateToSurfacePoint(Coordinate coor, boolean isWorld);



	/**
	 * Returns the canvas instance.
	 * 
	 * @return the canvas
	 */
	GLCanvas getCanvas();



	/**
	 * Returns the lighting values (ambient, diffuse, specular, directionX, directionY, directionZ).
	 * 
	 * @return the lighting values
	 */
	float[] getLightingValues();



	/**
	 * Returns the material values (red, green, blue, shininess, emissivity, transparency).
	 * 
	 * @param surface the surface from which the material values should be returned
	 * @return the material values
	 */
	float[] getMaterialValues(Surface surface);



	/**
	 * Returns a list of all surfaces being rendered (starts with the base surface).
	 * 
	 * @return a list of all surfaces
	 */
	Surface[] getSurfaces();



	/**
	 * Returns the surface viewer frame.
	 * 
	 * @return the surface viewer frame
	 */
	JFrame getSurfaceViewerFrame();



	/**
	 * Returns the threshold used to build the surface.
	 * 
	 * @return the threshold used to build the surface
	 */
	float getThreshold();



	/**
	 * Returns the associated viewer controller
	 * 
	 * @return the associated vidwer controller
	 */
	BasicController getViewerController();



	/**
	 * Returns a 4x4 of the view transform.
	 * 
	 * @return the view transform
	 */
	double[] getViewTransform();



	/**
	 * Unregisters this plugin instance. After this call, the plugin's display method will no longer be called.
	 * 
	 * @param plugin the plugin to remove
	 */
	void removeFromDisplay(SurfacePlugin plugin);



	/**
	 * Sets the lighting values.
	 * 
	 * @param ambient the ambient lighting value
	 * @param diffuse the diffuse lighting value
	 * @param specular the specular lighting value
	 * @param direction x, y, z
	 */
	void setLightingValues(float ambient, float diffuse, float specular, float[] direction);



	/**
	 * Sets the material values.
	 * 
	 * @param surface the surface to apply the changes
	 * @param color red, green, blue
	 * @param shininess the shininess material value
	 * @param emissivity the emissivity material value
	 * @param transparency the transparency material value
	 */
	void setMaterialValues(Surface surface, float[] color, float shininess, float emissivity, float transparency);



	/**
	 * Sets the view transform.
	 * 
	 * @param xform a 4x4 transform
	 */
	void setViewTransform(double[] xform);



	/**
	 * Redraws the surface. This is the same as calling getCanvas().display().
	 */
	void updateSurface();
}
