
package edu.uthscsa.ric.utilities;

import java.io.File;
import java.io.IOException;

import org.apache.log4j.Logger;


/**
 * A class to manage logging in the application and plugins.
 * 
 */
public final class AppLogger {

	private static ConsoleLogger console;
	private static File logFile;
	private static Logger logger; // NOPMD

	public static final String LOG_FILENAME = "mango.log";



	private AppLogger() {}



	public static void reset() {
		console = null;
		logFile = null;
		logger = null;
	}



	/**
	 * Creates the file logger. Usually called by the main method.
	 * 
	 * @param logDir the log directory
	 * @return true if the logger was created, or false if it already existed
	 */
	public static boolean createFileLogger(final File logDir) {
		if (logger == null) {
			boolean exists = logDir.exists();

			if (!exists) {
				exists = logDir.mkdirs();
			}

			if (exists) {
				try {
					logFile = new File(logDir.getCanonicalPath(), LOG_FILENAME);
					System.setProperty("log.dir", logDir.getCanonicalPath());
					System.setProperty("log.name", LOG_FILENAME);
				} catch (final IOException ex) {
					ex.printStackTrace(); // NOPMD
				}

				logger = Logger.getLogger(AppLogger.class);

				return true;
			}
		}

		return false;
	}



	/**
	 * Log debug.
	 * 
	 * @param message the debug message
	 */
	public static void debug(final String message) {
		if (logger != null) {
			if (logger.isDebugEnabled()) {
				logger.debug(message);
			}
		} else {
			System.out.println("DEBUG: " + message); // NOPMD
		}
	}



	/**
	 * Log error.
	 * 
	 * @param message the error message
	 */
	public static void error(final String message) {
		if (console != null) {
			console.println("Error: " + message);
		}

		if (logger != null) {
			logger.error(message);
		} else {
			System.err.println(message); // NOPMD
		}
	}



	/**
	 * Log error.
	 * 
	 * @param throwable the caught exception
	 */
	public static void error(final Throwable throwable) {
		if (console != null) {
			console.println("Error: " + findFirstRelevantStackTraceElement(throwable) + " [" + throwable.getMessage() + "]");
		}

		if (logger != null) {
			logger.error(findFirstRelevantStackTraceElement(throwable), throwable);
		} else {
			System.err.println(findFirstRelevantStackTraceElement(throwable) + " [" + throwable.getMessage() + "]"); // NOPMD
			throwable.printStackTrace(System.err);
		}
	}



	/**
	 * Log error (without stacktrace).
	 * 
	 * @param throwable the caught exception
	 */
	public static void errorMinimal(final Throwable throwable) {
		if (logger != null) {
			logger.error(findFirstRelevantStackTraceElement(throwable) + " [" + throwable.getMessage() + "]");
		} else {
			System.err.println(findFirstRelevantStackTraceElement(throwable) + " [" + throwable.getMessage() + "]"); // NOPMD
		}
	}



	/**
	 * Returns the current ConsoleLogger.
	 * 
	 * @return the current ConsoleLogger
	 */
	public static ConsoleLogger getConsoleLogger() {
		return console;
	}



	/**
	 * Returns the log file.
	 * 
	 * @return the log file
	 */
	public static File getLogFile() {
		return logFile;
	}



	/**
	 * Log info.
	 * 
	 * @param message the info message
	 */
	public static void info(final String message) {
		if (console != null) {
			console.println(message);
		}

		if (logger != null) {
			logger.info(message);
		} else {
			System.out.println(message); // NOPMD
		}
	}



	/**
	 * Log info.
	 * 
	 * @param throwable the caught exception
	 */
	public static void info(final Throwable throwable) {
		if (console != null) {
			console.println(findFirstRelevantStackTraceElement(throwable) + " [" + throwable.getMessage() + "]");
		}

		if (logger != null) {
			logger.info(findFirstRelevantStackTraceElement(throwable) + " [" + throwable.getMessage() + "]");
		} else {
			System.out.println("INFO: " + throwable.getStackTrace()[0] + " [" + throwable.getMessage() + "]"); // NOPMD
		}
	}



	/**
	 * Returns true if debug is enabled.
	 * 
	 * @return true if debug is enabled
	 */
	public static boolean isDebugEnabled() {
		return (logger != null) && logger.isDebugEnabled();
	}



	/**
	 * Set the ConsoleLogger to use.
	 * 
	 * @param console the ConsoleLogger to use
	 */
	public static void setConsole(final ConsoleLogger console) {
		AppLogger.console = console;
	}



	/**
	 * Log warning.
	 * 
	 * @param message the warning message
	 */
	public static void warn(final String message) {
		if (console != null) {
			console.println("Warning: " + message);
		}

		if (logger != null) {
			logger.warn(message);
		} else {
			System.err.println(message); // NOPMD
		}
	}



	/**
	 * Log warning.
	 * 
	 * @param throwable the caught exception
	 */
	public static void warn(final Throwable throwable) {
		if (console != null) {
			console.println("Warning: " + findFirstRelevantStackTraceElement(throwable) + " [" + throwable.getMessage() + "]");
		}

		if (logger != null) {
			logger.warn(findFirstRelevantStackTraceElement(throwable), throwable);
		} else {
			System.err.println(findFirstRelevantStackTraceElement(throwable) + " [" + throwable.getMessage() + "]"); // NOPMD
			throwable.printStackTrace(System.err);
		}
	}



	private static String findFirstRelevantStackTraceElement(final Throwable throwable) {
		final StackTraceElement[] stacktrace = throwable.getStackTrace();
		if (stacktrace != null) {
			for (int ctr = 0; ctr < stacktrace.length; ctr++) {
				final String str = throwable.getStackTrace()[ctr].toString();
				if (str.indexOf("edu.uthscsa") != -1) {
					return str;
				}
			}

			return throwable.getStackTrace()[0].toString();
		}

		return "(No stacktrace available)";
	}
}
