
package edu.uthscsa.ric.roi;

import java.awt.Color;
import java.awt.Shape;
import java.awt.Stroke;
import java.nio.ByteBuffer;
import java.util.List;

import edu.uthscsa.ric.mango.viewerslice.VolumeManager;
import edu.uthscsa.ric.roi.mask.ROIMaskBuffer;
import edu.uthscsa.ric.volume.Coordinate;
import edu.uthscsa.ric.volume.LabelManager;


/**
 * ROIData can be used to access and edit ROI mask, line, and point data.
 * 
 */
public interface ROIData extends ColorManager {

	@Deprecated
	byte BLUE_MASK = 4;
	@Deprecated
	byte CYAN_MASK = 16;
	@Deprecated
	byte GREEN_MASK = 2;
	@Deprecated
	byte MAGENTA_MASK = 8;
	@Deprecated
	byte ORANGE_MASK = 64;
	@Deprecated
	byte RED_MASK = 1;
	@Deprecated
	byte SLATE_MASK = (byte) 128;
	@Deprecated
	byte YELLOW_MASK = 32;
	@Deprecated
	int BLUE_INDEX = 2;
	@Deprecated
	int CYAN_INDEX = 4;
	@Deprecated
	int GREEN_INDEX = 1;
	@Deprecated
	int MAGENTA_INDEX = 3;
	@Deprecated
	int ORANGE_INDEX = 6;
	@Deprecated
	int RED_INDEX = 0;
	@Deprecated
	int SLATE_INDEX = 7;
	@Deprecated
	int YELLOW_INDEX = 5;
	@Deprecated
	long MASL_BLUE = 4;

	int INDEX_BLUE = 2;
	int INDEX_GREEN = 1;
	int INDEX_RED = 0;
	int SLICE_DIRECTION_AXIAL = VolumeManager.SLICE_DIRECTION_AXIAL;
	int SLICE_DIRECTION_CORONAL = VolumeManager.SLICE_DIRECTION_CORONAL;
	int SLICE_DIRECTION_SAGITTAL = VolumeManager.SLICE_DIRECTION_SAGITTAL;
	long MASK_BLUE = 4;
	long MASK_GREEN = 2;
	long MASK_RED = 1;



	/**
	 * Adds an ROI value at a specified location. The call addROIValue(0, 0, 0, MASK_RED) is equivalent to replaceROIValue(0, 0, 0, getROIValue(0, 0, 0) |
	 * MASK_RED).
	 * 
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @param tLoc the series index
	 * @param mask the value to add
	 */
	void addMaskValue(int xLoc, int yLoc, int zLoc, int tLoc, long mask);



	/**
	 * Adds an ROI value at a specified location. The call addROIValue(0, 0, 0, MASK_RED) is equivalent to replaceROIValue(0, 0, 0, getROIValue(0, 0, 0) |
	 * MASK_RED).
	 * 
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @param mask the value to add
	 */
	void addMaskValue(int xLoc, int yLoc, int zLoc, long mask);



	/**
	 * Adds a point.
	 * 
	 * @param point the point to add
	 * @param color the color index (e.g., RED_INDEX, GREEN_INDEX, etc)
	 */
	void addPoint(Coordinate point, int color);



	/**
	 * Adds an ROI value at a specified XYZ index coordinate. An ROI value is a one-byte bit map representing the state of eight possible ROIs for a voxel. The
	 * call addROIValue(0, 0, 0, RED_MASK) is equivalent to replaceROIValue(0, 0, 0, getROIValue(0, 0, 0) | RED_MASK).
	 * 
	 * @deprecated
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @param roiVal the value to add
	 */
	@Deprecated
	void addROIMaskValue(int xLoc, int yLoc, int zLoc, byte roiVal);



	/**
	 * Adds an ROI value at a specified XYZ index coordinate. An ROI value is a one-byte bit map representing the state of eight possible ROIs for a voxel. The
	 * call addROIValue(0, 0, 0, RED_MASK) is equivalent to replaceROIValue(0, 0, 0, getROIValue(0, 0, 0) | RED_MASK).
	 * 
	 * @deprecated
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @param tLoc the series index
	 * @param roiVal the value to add
	 */
	@Deprecated
	void addROIMaskValue(int xLoc, int yLoc, int zLoc, int tLoc, byte roiVal);



	/**
	 * Add a shape of a specified color within a slice.
	 * 
	 * @param sliceDirection SLICE_DIRECTION_AXIAL, SLICE_DIRECTION_CORONAL, or SLICE_DIRECTION_SAGITTAL
	 * @param slice the slice index
	 * @param color the color index (e.g., RED_INDEX, GREEN_INDEX, etc)
	 * @param shape the shape to add
	 * @return the added shape
	 */
	Shape addShape(int sliceDirection, int slice, int color, Shape shape);



	/**
	 * Adds a point with special properties that can only be removed programmatically.
	 * 
	 * @param point the point to add
	 * @param color the color value
	 * @param editable true to make this point editable by the user
	 */
	void addSpecialPoint(Coordinate point, Color color, boolean editable);



	/**
	 * Add a shape with special properties that can only be remove programmatically.
	 * 
	 * @param sliceDirection SLICE_DIRECTION_AXIAL, SLICE_DIRECTION_CORONAL, or SLICE_DIRECTION_SAGITTAL
	 * @param slice the slice index (only relevant when fixed is false)
	 * @param color the color value
	 * @param shape the shape to add
	 * @param stroke the shape stroke
	 * @param editable true to make this shape editable by the user
	 * @param fixed true to make this shape fixed across all slices (i.e., not associated with a particular slice)
	 * @return the added shape
	 */
	Shape addSpecialShape(int sliceDirection, int slice, Color color, Shape shape, Stroke stroke, boolean editable, boolean fixed);



	/**
	 * Creates a bit mask for a ROI color.
	 * 
	 * @param colorIndex the ROI color index
	 * @return a bit mask
	 */
	long createMask(int colorIndex);



	/**
	 * Returns a list of all ROIs.
	 * 
	 * @return a list of all ROIs
	 */
	List<ROI> getAllROIs();



	/**
	 * If there exists selected mask, it returns getSelectedMask(), otherwise it returns getUsedMaks().
	 * 
	 * @return the available mask
	 */
	long getAvailableMask();



	/**
	 * Returns the ROI mask LabelManager.
	 * 
	 * @return the ROI mask LabelManager
	 */
	LabelManager getLabelManager();



	/**
	 * Gets the ROI value at a specified location.
	 * 
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @return the ROI value
	 */
	long getMaskValue(int xLoc, int yLoc, int zLoc);



	/**
	 * Gets the ROI value at a specified location.
	 * 
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @param tLoc the series index
	 * @return the ROI value
	 */
	long getMaskValue(int xLoc, int yLoc, int zLoc, int tLoc);



	/**
	 * Gets all points of a specified color.
	 * 
	 * @param color the color index (e.g., RED_INDEX, GREEN_INDEX, etc)
	 * @return an array of all points of the specified color
	 */
	Coordinate[] getPoints(int color);



	/**
	 * Returns the color of this ROI.
	 * 
	 * @param colorIndex the color index
	 * @return the Color
	 */
	Color getROIColor(int colorIndex);



	/**
	 * Returns the ROI data buffer. The ROI buffer is one greater in each dimension than the image dimensions. ROI buffer data is always stored in XYZ+-- data
	 * orientation.
	 * 
	 * @return the ROIBuffer
	 */
	ROIMaskBuffer getROIDataBuffer();



	/**
	 * Returns the label of this ROI.
	 * 
	 * @param colorIndex the color index
	 * @return the label
	 */
	String getROILabel(int colorIndex);



	/**
	 * Returns the ROI work buffer, for use with advanced methods. The ROI buffer is one greater in each dimension than the image dimensions. ROI buffer data is
	 * always stored in XYZ+-- data orientation.
	 * 
	 * @deprecated
	 * @return the ROI work buffer
	 */
	@Deprecated
	ByteBuffer getROIMaskBuffer();



	/**
	 * Gets an ROI value at a specified XYZ index coordinate. An ROI value is a one-byte bit map representing the state of eight possible ROIs for a voxel.
	 * 
	 * @deprecated
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @return the ROI value at the specified index coordinate
	 */
	@Deprecated
	byte getROIMaskValue(int xLoc, int yLoc, int zLoc);



	/**
	 * Gets an ROI value at a specified XYZ index coordinate. An ROI value is a one-byte bit map representing the state of eight possible ROIs for a voxel.
	 * 
	 * @deprecated
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @param tLoc the series index
	 * @return the ROI value at the specified index coordinate
	 */
	@Deprecated
	byte getROIMaskValue(int xLoc, int yLoc, int zLoc, int tLoc);



	/**
	 * Returns the "selected" state of the ROI mask.
	 * 
	 * @deprecated
	 * @return a byte mask
	 */
	@Deprecated
	byte getROISelectedMask();



	/**
	 * Returns the "used" state of the ROI mask.
	 * 
	 * @deprecated
	 * @return a byte mask
	 */
	@Deprecated
	byte getROIUsedMask();



	/**
	 * Returns a list of selected ROI lines.
	 * 
	 * @return a list of selected ROI lines
	 */
	List<Line> getSelectedLines();



	/**
	 * Returns a bit mask of selected ROI colors.
	 * 
	 * @return a bit mask
	 */
	long getSelectedMask();



	/**
	 * Returns a list of selected ROI masks.
	 * 
	 * @return a list of selected ROI masks
	 */
	List<ROI> getSelectedMasks();



	/**
	 * Returns a list of selected ROI points.
	 * 
	 * @return a list of selected ROI points
	 */
	List<Point> getSelectedPoints();



	/**
	 * Returns a list of selected ROIs.
	 * 
	 * @return a list of selected ROIs
	 */
	List<ROI> getSelectedROIs();



	/**
	 * Gets all shapes of a specified color within a slice. For example, getShapes(0, 0, 0) will return all shapes in the first axial slice matching color value
	 * 0.
	 * 
	 * @param sliceDirection SLICE_DIRECTION_AXIAL, SLICE_DIRECTION_CORONAL, or SLICE_DIRECTION_SAGITTAL
	 * @param slice the slice index
	 * @param color the color index (e.g., RED_INDEX, GREEN_INDEX, etc)
	 * @return an array of Shapes or null if none are found
	 */
	Shape[] getShapes(int sliceDirection, int slice, int color);



	/**
	 * Gets all special points of a specified color.
	 * 
	 * @param color the color value
	 * @return an array of all points of the specified color
	 */
	Coordinate[] getSpecialPoints(Color color);



	/**
	 * Gets all special shapes of a specified color within a slice. For example, getShapes(0, 0, 0) will return all shapes in the first axial slice matching the
	 * Color.
	 * 
	 * @param sliceDirection SLICE_DIRECTION_AXIAL, SLICE_DIRECTION_CORONAL, or SLICE_DIRECTION_SAGITTAL
	 * @param slice the slice index
	 * @param color the color value
	 * @return an array of Shapes or null if none are found
	 */
	Shape[] getSpecialShapes(int sliceDirection, int slice, Color color);



	/**
	 * Returns a list of used ROI lines.
	 * 
	 * @return a list of used ROI lines
	 */
	List<Line> getUsedLines();



	/**
	 * Returns a bit mask of used ROI colors.
	 * 
	 * @return a bit mask
	 */
	long getUsedMask();



	/**
	 * Returns a list of used ROIs.
	 * 
	 * @return a list of used ROIs
	 */
	List<ROI> getUsedMasks();



	/**
	 * Returns a list of used ROI points.
	 * 
	 * @return a list of used ROI points
	 */
	List<Point> getUsedPoints();



	/**
	 * Test if there is a selected ROI mask in the specified bounds.
	 * 
	 * @param xMin the X minimum
	 * @param xMax the X maximum
	 * @param yMin the Y minimum
	 * @param yMax the Y maximum
	 * @param zMin the Z minimum
	 * @param zMax the Z maximum
	 * @return true if a selected ROI mask is found in the specified bounds, false otherwise
	 */
	boolean hasSelectedMaskInRange(final int xMin, final int xMax, final int yMin, final int yMax, final int zMin, final int zMax);



	/**
	 * Initializes one or more ROI colors. For example, initROIMask(ROIData.MASK_RED) or initROIMask(createMask(11)). This should be called before any calls to
	 * replaceROIMaskValue() or addROIMaskValue().
	 * 
	 * @param mask the bit mask of all the ROI colors to initialize
	 */
	void initMask(long mask);



	/**
	 * Initializes one or more ROI colors. For example, initROIMask(ROIData.MASK_RED). This should be called before any calls to replaceROIMaskValue() or
	 * addROIMaskValue().
	 * 
	 * @deprecated
	 * @param mask the bit mask of all the ROI colors to initialize
	 */
	@Deprecated
	void initROIMask(byte mask);



	/**
	 * Test if there is a selected ROI mask at the specified offset.
	 * 
	 * @param offset the offset
	 * @return true if a selected ROI mask is found at the specified offset, false otherwise
	 */
	boolean isInsideSelectedMaskAtOffset(final int offset);



	/**
	 * Removes a point.
	 * 
	 * @param point the point to remove
	 * @param color the color index (e.g., RED_INDEX, GREEN_INDEX, etc)
	 */
	void removePoint(Coordinate point, int color);



	/**
	 * Remove a shape.
	 * 
	 * @param sliceDirection SLICE_DIRECTION_AXIAL, SLICE_DIRECTION_CORONAL, or SLICE_DIRECTION_SAGITTAL
	 * @param shape the shape to remove
	 */
	void removeShape(int sliceDirection, Shape shape);



	/**
	 * Removes a special point.
	 * 
	 * @param point the point to remove
	 * @param color the color value
	 */
	void removeSpecialPoint(Coordinate point, Color color);



	/**
	 * Remove a special shape.
	 * 
	 * @param sliceDirection SLICE_DIRECTION_AXIAL, SLICE_DIRECTION_CORONAL, or SLICE_DIRECTION_SAGITTAL
	 * @param shape the shape to remove
	 */
	void removeSpecialShape(int sliceDirection, Shape shape);



	/**
	 * Replace a ROI value at a specified location.
	 * 
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @param tLoc the series index
	 * @param mask the value to write
	 */
	void replaceMaskValue(int xLoc, int yLoc, int zLoc, int tLoc, long mask);



	/**
	 * Replace a ROI value at a specified location.
	 * 
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @param mask the value to write
	 */
	void replaceMaskValue(int xLoc, int yLoc, int zLoc, long mask);



	/**
	 * Replace an ROI value at a specified XYZ index coordinate. An ROI value is a one-byte bit map representing the state of eight possible ROIs for a voxel.
	 * 
	 * @deprecated
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @param roiVal the value to write
	 */
	@Deprecated
	void replaceROIMaskValue(int xLoc, int yLoc, int zLoc, byte roiVal);



	/**
	 * Replace an ROI value at a specified XYZ index coordinate. An ROI value is a one-byte bit map representing the state of eight possible ROIs for a voxel.
	 * 
	 * @deprecated
	 * @param xLoc the X index
	 * @param yLoc the Y index
	 * @param zLoc the Z index
	 * @param tLoc the series index
	 * @param roiVal the value to write
	 */
	@Deprecated
	void replaceROIMaskValue(int xLoc, int yLoc, int zLoc, int tLoc, byte roiVal);



	/**
	 * Call this following an ROI operation (e.g., after iterating through the entire volume making changes).
	 */
	void roiMaskOperationCompleted();



	/**
	 * Call this following an ROI operation (e.g., after iterating through the entire volume making changes).
	 * 
	 * @param des message to log
	 */
	void roiMaskOperationCompleted(String des);



	/**
	 * Sets the highlight state of a special point.
	 * 
	 * @param point the point to change
	 * @param color the color value
	 * @param highlight true, to set as highlighted
	 */
	void setPointHighlightState(Coordinate point, Color color, boolean highlight);



	/**
	 * Sets the highlight state of a special shape.
	 * 
	 * @param sliceDirection SLICE_DIRECTION_AXIAL, SLICE_DIRECTION_CORONAL, or SLICE_DIRECTION_SAGITTAL
	 * @param shape the shape to change
	 * @param highlight true, to set as highlighted
	 */
	void setShapeHighlightState(int sliceDirection, Shape shape, boolean highlight);



	/**
	 * True to use a ROI series, false otherwise.
	 * 
	 * @param useROISeries true to use a ROI series
	 */
	void setUseROISeries(boolean useROISeries);
}
