
package edu.uthscsa.ric.mango.viewersurface;

import java.util.List;

import javax.swing.JFrame;

import edu.uthscsa.ric.cli.ParameterNameAnnotation;
import edu.uthscsa.ric.visualization.surface.primitives.Shape;
import edu.uthscsa.ric.visualization.surface.primitives.Surface;
import edu.uthscsa.ric.volume.Analysis;
import edu.uthscsa.ric.volume.ImageVolume;


/**
 * Manages surface data.
 * 
 * @author michaelmartinez
 *
 */
public interface SurfaceManager {

	int RENDER_TYPE_POINTS = 2;
	int RENDER_TYPE_TRIANGLES = 0;
	int RENDER_TYPE_WIREFRAME = 1;
	int VIDEO_FORMAT_AVI_DIB = 2;
	int VIDEO_FORMAT_AVI_MJPG = 0;
	int VIDEO_FORMAT_AVI_PNG = 1;
	int VIDEO_FORMAT_QUICKTIME_ANIMATION = 5;
	int VIDEO_FORMAT_QUICKTIME_JPEG = 3;
	int VIDEO_FORMAT_QUICKTIME_PNG = 4;
	int VIDEO_FORMAT_QUICKTIME_RAW = 6;
	int VIDEO_QUALITY_BEST = 0;
	int VIDEO_QUALITY_FASTEST = 2;
	int VIDEO_QUALITY_GOOD = 1;



	/**
	 * Adds a surface shape from a logical.
	 * 
	 * @param overlays the logical
	 * @param name the name of the surface
	 * @param rgb the color of the surface as an RGB color int
	 * @param roiThreshold the build threshold (0 to 1)
	 * @param shrinkWrap true to use shrinkwrap, false otherwise
	 * @param maximum true if the threshold represents a maximum, false otherwise
	 * @param imageSmoothing true to use image smoothing, false otherwise
	 * @param width FWHM of the image smoothing filter
	 * @param kernelSize kernel size of the image smoothing filter
	 * @param resX the output X resolution
	 * @param resY the output Y resolution
	 * @param resZ the output Z resolution
	 * @param surfaceSmoothing true to use surface smoothing, false otherwise
	 * @param pointError the point error threshold
	 * @param featureAngle the feature angle threshold
	 * @param iterations number of iterations of surface smoothing
	 * @return the new Surface
	 */
	@ParameterNameAnnotation(names = { "overlays", "name", "rgb", "logicalThreshold", "shrinkWrap", "maximum", "imageSmoothing", "width", "kernelSize", "resX",
			"resY", "resZ", "surfaceSmoothing", "pointError", "featureAngle", "iterations" })
	Surface addShapeFromLogical(List<ImageVolume> overlays, String name, int rgb, double roiThreshold, boolean shrinkWrap, boolean maximum,
			boolean imageSmoothing, double width, int kernelSize, double resX, double resY, double resZ, boolean surfaceSmoothing, double pointError,
			double featureAngle, int iterations);



	/**
	 * Adds a surface shape from an overlay.
	 * 
	 * @param overlay the overlay
	 * @param name the name of the surface
	 * @param rgb the color of the surface as an RGB color int
	 * @param threshold the build threshold
	 * @param shrinkWrap true to use shrinkwrap, false otherwise
	 * @param maximum true if the threshold represents a maximum, false otherwise
	 * @param imageSmoothing true to use image smoothing, false otherwise
	 * @param width FWHM of the image smoothing filter
	 * @param kernelSize kernel size of the image smoothing filter
	 * @param resX the output X resolution
	 * @param resY the output Y resolution
	 * @param resZ the output Z resolution
	 * @param surfaceSmoothing true to use surface smoothing, false otherwise
	 * @param pointError the point error threshold
	 * @param featureAngle the feature angle threshold
	 * @param iterations number of iterations of surface smoothing
	 * @return the new Surface
	 */
	@ParameterNameAnnotation(names = { "overlay", "name", "rgb", "threshold", "shrinkWrap", "maximum", "imageSmoothing", "width", "kernelSize", "resX", "resY",
			"resZ", "surfaceSmoothing", "pointError", "featureAngle", "iterations" })
	Surface addShapeFromOverlay(ImageVolume overlay, String name, int rgb, double threshold, boolean shrinkWrap, boolean maximum, boolean imageSmoothing,
			double width, int kernelSize, double resX, double resY, double resZ, boolean surfaceSmoothing, double pointError, double featureAngle,
			int iterations);



	/**
	 * Adds a surface shape from a ROI.
	 * 
	 * @param roiColorIndex the ROI color index
	 * @param name the name of the surface
	 * @param rgb the color of the surface as an RGB color int
	 * @param roiThreshold the build threshold (0 to 1)
	 * @param shrinkWrap true to use shrinkwrap, false otherwise
	 * @param maximum true if the threshold represents a maximum, false otherwise
	 * @param imageSmoothing true to use image smoothing, false otherwise
	 * @param width FWHM of the image smoothing filter
	 * @param kernelSize kernel size of the image smoothing filter
	 * @param resX the output X resolution
	 * @param resY the output Y resolution
	 * @param resZ the output Z resolution
	 * @param surfaceSmoothing true to use surface smoothing, false otherwise
	 * @param pointError the point error threshold
	 * @param featureAngle the feature angle threshold
	 * @param iterations number of iterations of surface smoothing
	 * @return the new Surface
	 */
	@ParameterNameAnnotation(names = { "roiColorIndex", "name", "rgb", "roiThreshold", "shrinkWrap", "maximum", "imageSmoothing", "width", "kernelSize",
			"resX", "resY", "resZ", "surfaceSmoothing", "pointError", "featureAngle", "iterations" })
	Surface addShapeFromROI(int roiColorIndex, String name, int rgb, double roiThreshold, boolean shrinkWrap, boolean maximum, boolean imageSmoothing,
			double width, int kernelSize, double resX, double resY, double resZ, boolean surfaceSmoothing, double pointError, double featureAngle,
			int iterations);



	/**
	 * Adds a surface shape from file.
	 * 
	 * @param openLocation the file to load
	 * @return a list of Surfaces found in the file
	 */
	@ParameterNameAnnotation(names = { "openLocation" })
	List<Surface> addShapesFromFile(final String openLocation);



	/**
	 * Save a snapshot of the surface viewer to the specified location.
	 * 
	 * @param saveLocation the location to save the snapshot file
	 */
	@ParameterNameAnnotation(names = { "saveLocation" })
	void captureSnapshot(String saveLocation);



	/**
	 * Starts saving a captured video to the specified file location.
	 * 
	 * @param saveLocation the location to save the video file
	 * @param videoFormat the video format code (see constants)
	 * @param videoQuality the video quality code (see constants)
	 * @param fps the frames per second
	 * @param animate true to automatically animate the projection, false otherwise
	 */
	@ParameterNameAnnotation(names = { "saveLocation", "videoFormat", "videoQuality", "fps", "record/animate" })
	void captureVideoStart(String saveLocation, int videoFormat, int videoQuality, int fps, boolean animate);



	/**
	 * Stops the video capture.
	 */
	void captureVideoStop();



	/**
	 * Exports the surface data to the specified file location.
	 * 
	 * @param saveLocation the location to save the surface file
	 * @param surfaceFormat the name of the surface plugin format
	 * @param surfaces specify a single surface to save or "null" to save all surfaces
	 * @param options options specific to this surface format
	 */
	@ParameterNameAnnotation(names = { "saveLocation", "surfaceFormat", "surfaces", "options" })
	void exportTo(final String saveLocation, String surfaceFormat, Surface surfaces, String options);



	/**
	 * Returns a list of surfaces added to the base surface.
	 * 
	 * @return a list of surfaces added to the base surface
	 */
	List<Surface> getAddedShapes();



	/**
	 * Returns a list of all points and lines.
	 * 
	 * @return a list of all points and lines
	 */
	List<Shape> getAllPointsAndLines();



	/**
	 * Returns the base surface
	 * 
	 * @return the base surface
	 */
	Surface getBaseSurface();



	/**
	 * Returns the parent window of the surface view.
	 * 
	 * @return the parent window of the surface view
	 */
	JFrame getFrame();



	/**
	 * Returns a list of lines.
	 * 
	 * @return a list of lines
	 */
	List<Shape> getLines();



	/**
	 * Returns a list of points.
	 * 
	 * @return a list of points
	 */
	List<Shape> getPoints();



	/**
	 * Returns an array of loaded surfaces.
	 * 
	 * @return an array of loaded surfaces
	 */
	Surface[] getSurfaces();



	/**
	 * Returns the threshold used to build the base surface.
	 * 
	 * @return the threshold used to build the base surface
	 */
	float getThreshold();



	/**
	 * Returns the view transform (4x4).
	 * 
	 * @return the view transform
	 */
	double[] getViewTransform();



	/**
	 * Move the current coordinate to the specified line or point shape.
	 * 
	 * @param shape the line or point shape
	 */
	@ParameterNameAnnotation(names = { "shape" })
	void goToShape(Shape shape);



	/**
	 * Hide/show all the added surface shapes, lines, and points.
	 * 
	 * @param hidden true to hide, false to show
	 */
	@ParameterNameAnnotation(names = { "hidden" })
	void hideAll(boolean hidden);



	/**
	 * Adds a direct line between two shapes.
	 * 
	 * @param fromShape the first shape
	 * @param toShape the second shape
	 */
	@ParameterNameAnnotation(names = { "fromShape", "toShape" })
	void lineToShape(Shape fromShape, Shape toShape);



	/**
	 * Measure all added surface shapes, lines, and points.
	 * 
	 * @return a list of Analyses
	 */
	List<Analysis> measureAll();



	/**
	 * Pauses execution of a running script.
	 */
	void pauseScriptForUserInput();



	/**
	 * Remove all the added surface shapes, lines, and points.
	 */
	void removeAll();



	/**
	 * Remove a line or point shape.
	 * 
	 * @param shape the line or point to remove
	 */
	@ParameterNameAnnotation(names = { "shape" })
	void removeShape(Shape shape);



	/**
	 * Remove an added surface shape.
	 * 
	 * @param surface the surface to remove
	 */
	@ParameterNameAnnotation(names = { "surface" })
	void removeSurface(Surface surface);



	/**
	 * Runs a surface plugin.
	 * 
	 * @param pluginName the plugin name
	 * @param args plugin arguments
	 * @return an optional return object
	 */
	@ParameterNameAnnotation(names = { "pluginName", "args" })
	Object runPlugin(String pluginName, String[] args);



	/**
	 * Runs surface area stats on the specified surface.
	 * 
	 * @param surface the specified surface
	 * @return a list of two Analyses, the visible part of the surface and the cut-away part (if applicable)
	 */
	@ParameterNameAnnotation(names = { "surface" })
	List<Analysis> runSurfaceAreaStats(Surface surface);



	/**
	 * Runs surface volume stats on the specified surface.
	 * 
	 * @param surface the specified surface
	 * @return a list of two Analysis, the visible part of the surface ant the cut-away part (if applicable)
	 */
	@ParameterNameAnnotation(names = { "surface" })
	List<Analysis> runSurfaceVolumeStats(Surface surface);



	/**
	 * Saves the surface data in the Mango surface format.
	 * 
	 * @param saveLocation the location to save the surface file
	 * @param surfaces specify a single surface to save or "null" to save all surfaces
	 * @param includeLinesAndPoints true to save lines and points, false otherwise
	 */
	@ParameterNameAnnotation(names = { "saveLocation", "surfaces", "includeLinesAndPoints" })
	void saveAs(final String saveLocation, Surface surfaces, boolean includeLinesAndPoints);



	/**
	 * Selects menu options.
	 * 
	 * @param name the menu item name
	 */
	@ParameterNameAnnotation(names = { "name" })
	void selectMenuOption(final String name);



	/**
	 * Sets the surface background
	 * 
	 * @param rgb an RGB color int
	 */
	@ParameterNameAnnotation(names = { "rgb" })
	void setBackground(int rgb);



	/**
	 * Sets the lighting parameters.
	 * 
	 * @param ambient the ambient light parameter (0 to 1)
	 * @param diffuse the diffuse light parameter (0 to 1)
	 * @param specular the specular light parameter (0 to 1)
	 * @param horizontal the horizontal light position (-1 to 1)
	 * @param vertical the vertical light position (-1 to 1)
	 */
	@ParameterNameAnnotation(names = { "ambient", "diffuse", "specular", "horizontal", "vertical" })
	void setLighting(double ambient, double diffuse, double specular, double horizontal, double vertical);



	/**
	 * Sets the material parameters.
	 * 
	 * @param surface the surface
	 * @param rgb the surface color as an RGB color int
	 * @param showOverlayColors true to show overlay colors on the surface, false otherwise
	 * @param showOnlyThisOverlayColor true to show only the overlay associated with this surface, false otherwise
	 * @param flatShading true to use flat shading, false otherwise
	 * @param renderType the render type (see constants)
	 * @param transparency the transparency parameter (0 to 1)
	 * @param emmisivity the emmisivity parameter (0 to 1)
	 * @param shininess the shininess parameter (0 to 1)
	 * @param searchDistance the distance (mm) to search above and below the surface when building overlay colors
	 */
	@ParameterNameAnnotation(names = { "surface", "rgb", "showOverlayColors", "showOnlyThisOverlayColor", "flatShading", "renderType", "transparency",
			"emmisivity", "shininess", "searchDistance" })
	void setMaterial(Surface surface, int rgb, boolean showOverlayColors, boolean showOnlyThisOverlayColor, boolean flatShading, int renderType,
			double transparency, double emmisivity, double shininess, double searchDistance);



	/**
	 * Sets menu items (typically checkboxes).
	 * 
	 * @param name the menu item name
	 * @param value the menu item value
	 */
	@ParameterNameAnnotation(names = { "name", "value" })
	void setMenuOption(String name, boolean value);



	/**
	 * Adds a line along the surface bewteen two shapes.
	 * 
	 * @param fromPoint the first point
	 * @param toPoint the second point
	 */
	@ParameterNameAnnotation(names = { "fromPoint", "toPoint" })
	void surfaceToShape(Shape fromPoint, Shape toPoint);



	/**
	 * Redraws the surface view.
	 */
	void updateSurface();
}
