
package edu.uthscsa.ric.mango;

import java.awt.Dimension;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.File;
import java.nio.ByteBuffer;

import javax.swing.JButton;
import javax.swing.JFrame;

import edu.uthscsa.ric.mango.components.MangoColorSelectedListener;
import edu.uthscsa.ric.volume.ImageDimensions;
import edu.uthscsa.ric.volume.ImageRange;
import edu.uthscsa.ric.volume.VoxelDimensions;


/**
 * ViewerController extends BasicController and adds methods for multi-window environments.
 */
public interface ViewerController extends BasicController {

	/**
	 * Adds new image.
	 * 
	 * @param id the ImageDimensions
	 * @param vd the VoxelDimensions
	 * @param ir the ImageRange
	 * @param buffers an array of FloatBuffers, one per timepoint
	 * @param orientation the orientation string
	 * @param name the name of the new image
	 * @return the new viewer (then using getMangoData() in the new ViewerController will provide access to image data)
	 */
	ViewerController addNewImage(ImageDimensions id, VoxelDimensions vd, ImageRange ir, ByteBuffer[] buffers, String orientation, String name);



	/**
	 * Adds new image.
	 * 
	 * @param id the ImageDimensions
	 * @param vd the VoxelDimensions
	 * @param orientation the orientation string
	 * @param name the name of the new image
	 * @return the new viewer (then using getMangoData() in the new ViewerController will provide access to image data)
	 */
	ViewerController addNewImage(ImageDimensions id, VoxelDimensions vd, String orientation, String name);



	/**
	 * Associates a window generated by a plugin with the viewer. An associated window will appear in the Toolbox window menu and will close automatically when
	 * the viewer closes.
	 * 
	 * @param des the name of the window
	 * @param win the window
	 */
	void associatePluginWindow(String des, JFrame win);



	/**
	 * Closes a viewer-associated window.
	 * 
	 * @param des the name of the window (as it would appear in the ToolBox window menu)
	 */
	void closeAssociatedWindow(String des);



	/**
	 * Create a ROI color picker button. The button icon is a square swatch of the currently selected ROI color. When clicked the button will display a ROI
	 * color picker widget.
	 * 
	 * @param listener callback after user selects color
	 * @param showUsedOnly true to only show used ROI colors in picker, false to show all colors
	 * @return the button
	 */
	JButton createColorPickerButton(MangoColorSelectedListener listener, boolean showUsedOnly);



	/**
	 * Returns location that will center window in current display.
	 * 
	 * @param size the size of window
	 * @return location
	 */
	Point findCenterRelativeToScreen(Dimension size);



	/**
	 * Returns location that will center window relative to viewer.
	 * 
	 * @param size the size of window
	 * @return location
	 */
	Point findCenterRelativeToViewer(Dimension size);



	/**
	 * Returns bounds of display that viewer is currently within.
	 * 
	 * @return bounds
	 */
	Rectangle findCurrentDisplayBounds();



	/**
	 * Returns insets of display that viewer is currently within.
	 * 
	 * @return insets
	 */
	Insets findCurrentDisplayInsets();



	/**
	 * Returns dimensions of display that viewer is currently within.
	 * 
	 * @return dimensions
	 */
	Dimension findCurrentDisplaySize();



	/**
	 * Returns an appropriate location for a new window. Takes into consideration viewer location, new window size, and display bounds.
	 * 
	 * @param size new window size
	 * @return new location
	 */
	Point findGoodPluginWindowLocation(Dimension size);



	/**
	 * Returns all currently open viewers.
	 * 
	 * @param includeSelf include this ViewerController in the list
	 * @return an array of all currently open viewers
	 */
	ViewerController[] getAllViewerControllers(boolean includeSelf);



	/**
	 * Returns an atlas found by name.
	 * 
	 * @param atlasName name of the atlas
	 * @return the atlas, or null if none were found by that name
	 */
	Atlas getAtlas(String atlasName);



	/**
	 * Returns viewer frame.
	 * 
	 * @return viewer frame
	 */
	JFrame getFrame();



	/**
	 * Returns a directory to store library files.
	 * 
	 * @return the library directory
	 */
	File getLibDir();



	/**
	 * Returns a directory to store temporary data. These data files will be deleted when the application quits.
	 * 
	 * @return the temporary directory
	 */
	File getTempDir();



	/**
	 * Write a message to the Mango console.
	 * 
	 * @param str message to log
	 */
	void logToConsole(String str);



	/**
	 * Sets the location of the viewer on the screen.
	 * 
	 * @param aPoint the location of the viewer
	 */
	void setViewerLocation(Point aPoint);



	/**
	 * Sets the width (pixels) of the viewer.
	 * 
	 * @param width number of pixels
	 */
	void setViewerSize(int width);



	/**
	 * Activates the viewer window.
	 */
	void toFront();



	/**
	 * Updates the image and ROI slice data in all viewers.
	 */
	void updateAllViewers();
}
