
package edu.uthscsa.ric.mango;

import java.awt.Color;
import java.awt.Shape;
import java.awt.Stroke;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

import edu.uthscsa.ric.mango.viewerslice.VolumeManager;
import edu.uthscsa.ric.roi.Line;
import edu.uthscsa.ric.roi.Point;
import edu.uthscsa.ric.roi.PointsManager;
import edu.uthscsa.ric.roi.PointsSpecialManager;
import edu.uthscsa.ric.roi.ROI;
import edu.uthscsa.ric.roi.ROIData;
import edu.uthscsa.ric.roi.ROIMaskManager;
import edu.uthscsa.ric.roi.ShapesManager;
import edu.uthscsa.ric.roi.ShapesSpecialManager;
import edu.uthscsa.ric.roi.mask.ROIMaskBuffer;
import edu.uthscsa.ric.volume.Coordinate;
import edu.uthscsa.ric.volume.LabelManager;


public class MangoROI implements ROIData {

	private final PointsManager points;
	private final PointsSpecialManager pointsSpecial;
	private final ROIMaskManager roiMask;
	private final ShapesManager axialShapes;
	private final ShapesManager coronalShapes;
	private final ShapesManager sagittalShapes;
	private final ShapesSpecialManager axialShapesP;
	private final ShapesSpecialManager coronalShapesP;
	private final ShapesSpecialManager sagittalShapesP;

	public static final int SLICE_DIRECTION_AXIAL = VolumeManager.SLICE_DIRECTION_AXIAL;
	public static final int SLICE_DIRECTION_CORONAL = VolumeManager.SLICE_DIRECTION_CORONAL;
	public static final int SLICE_DIRECTION_SAGITTAL = VolumeManager.SLICE_DIRECTION_SAGITTAL;



	public MangoROI(final ROIMaskManager rm, final ShapesManager[] sm, final ShapesSpecialManager[] ssm, final PointsManager pm, final PointsSpecialManager psm) {
		this.roiMask = rm;
		this.axialShapes = sm[0];
		this.coronalShapes = sm[1];
		this.sagittalShapes = sm[2];
		this.points = pm;

		if (ssm != null) {
			this.axialShapesP = ssm[0];
			this.coronalShapesP = ssm[1];
			this.sagittalShapesP = ssm[2];
		} else {
			this.axialShapesP = null;
			this.coronalShapesP = null;
			this.sagittalShapesP = null;
		}

		this.pointsSpecial = psm;
	}



	public void addImmutablePoint(final Coordinate point, final Color color) {
		this.pointsSpecial.addImmutablePoint(point, color);
	}



	public Shape addImmutableShape(final int sliceDirection, final int slice, final Shape shape, final Color color, final Stroke stroke) {
		if (sliceDirection == SLICE_DIRECTION_AXIAL) {
			return this.axialShapesP.addImmutableShape(shape, slice, color, stroke);
		} else if (sliceDirection == SLICE_DIRECTION_CORONAL) {
			return this.coronalShapesP.addImmutableShape(shape, slice, color, stroke);
		} else if (sliceDirection == SLICE_DIRECTION_SAGITTAL) {
			return this.sagittalShapesP.addImmutableShape(shape, slice, color, stroke);
		}

		return null;
	}



	public Shape addImmutableStaticShape(final int sliceDirection, final Shape shape, final Color color, final Stroke stroke) {
		if (sliceDirection == SLICE_DIRECTION_AXIAL) {
			return this.axialShapesP.addStaticShape(shape, color, stroke);
		} else if (sliceDirection == SLICE_DIRECTION_CORONAL) {
			return this.coronalShapesP.addStaticShape(shape, color, stroke);
		} else if (sliceDirection == SLICE_DIRECTION_SAGITTAL) {
			return this.sagittalShapesP.addStaticShape(shape, color, stroke);
		}

		return null;
	}



	@Override
	public void addMaskValue(final int xLoc, final int yLoc, final int zLoc, final int tLoc, final long mask) {
		this.roiMask.addMaskValue(xLoc, yLoc, zLoc, tLoc, mask);
	}



	@Override
	public void addMaskValue(final int xLoc, final int yLoc, final int zLoc, final long mask) {
		this.roiMask.addMaskValue(xLoc, yLoc, zLoc, mask);
	}



	@Override
	public void addPoint(final Coordinate point, final int color) {
		this.points.addPoint(point, color);
	}



	@Override
	@Deprecated
	public void addROIMaskValue(final int xLoc, final int yLoc, final int zLoc, final byte roiVal) {
		this.roiMask.addROIMaskValue(xLoc, yLoc, zLoc, roiVal);
	}



	@Override
	@Deprecated
	public void addROIMaskValue(final int xLoc, final int yLoc, final int zLoc, final int tLoc, final byte roiVal) {
		this.roiMask.addROIMaskValue(xLoc, yLoc, zLoc, tLoc, roiVal);
	}



	@Override
	public Shape addShape(final int sliceDirection, final int slice, final int color, final Shape shape) {
		if (sliceDirection == SLICE_DIRECTION_AXIAL) {
			return this.axialShapes.addShape(shape, slice, color);
		} else if (sliceDirection == SLICE_DIRECTION_CORONAL) {
			return this.coronalShapes.addShape(shape, slice, color);
		} else if (sliceDirection == SLICE_DIRECTION_SAGITTAL) {
			return this.sagittalShapes.addShape(shape, slice, color);
		}

		return null;
	}



	public void addSpecialPoint(final Coordinate point, final Color color) {
		this.pointsSpecial.addPoint(point, color);
	}



	@Override
	public void addSpecialPoint(final Coordinate point, final Color color, final boolean editable) {
		if (editable) {
			addSpecialPoint(point, color);
		} else {
			addImmutablePoint(point, color);
		}
	}



	public Shape addSpecialShape(final int sliceDirection, final int slice, final Color color, final Shape shape, final Stroke stroke) {
		if (sliceDirection == SLICE_DIRECTION_AXIAL) {
			return this.axialShapesP.addShape(shape, slice, color, stroke);
		} else if (sliceDirection == SLICE_DIRECTION_CORONAL) {
			return this.coronalShapesP.addShape(shape, slice, color, stroke);
		} else if (sliceDirection == SLICE_DIRECTION_SAGITTAL) {
			return this.sagittalShapesP.addShape(shape, slice, color, stroke);
		}

		return null;
	}



	@Override
	public Shape addSpecialShape(final int sliceDirection, final int slice, final Color color, final Shape shape, final Stroke stroke, final boolean editable,
			final boolean fixed) {
		if (editable) {
			return addSpecialShape(sliceDirection, slice, color, shape, stroke);
		} else {
			if (fixed) {
				return addImmutableStaticShape(sliceDirection, shape, color, stroke);
			} else {
				return addImmutableShape(sliceDirection, slice, shape, color, stroke);
			}
		}
	}



	@Override
	public long createMask(final int colorIndex) {
		return (1L << colorIndex);
	}



	@Override
	public List<ROI> getAllROIs() {
		final List<ROI> rois = new ArrayList<ROI>();
		rois.addAll(getUsedPoints());
		rois.addAll(getUsedLines());
		rois.addAll(getUsedMasks());
		return rois;
	}



	@Override
	public long getAvailableMask() {
		return this.roiMask.getAvailableMask();
	}



	@Override
	public LabelManager getLabelManager() {
		return this.roiMask;
	}



	@Override
	public long getMaskValue(final int xLoc, final int yLoc, final int zLoc) {
		return this.roiMask.getMaskValue(xLoc, yLoc, zLoc);
	}



	@Override
	public long getMaskValue(final int xLoc, final int yLoc, final int zLoc, final int tLoc) {
		return this.roiMask.getMaskValue(xLoc, yLoc, zLoc, tLoc);
	}



	@Override
	public int getMaximumColors() {
		return this.roiMask.getROIDataBuffer().getMaximumColors();
	}



	@Override
	public Coordinate[] getPoints(final int color) {
		return this.points.getPoints(color);
	}



	@Override
	public Color getROIColor(final int colorIndex) {
		return this.roiMask.getROIColor(colorIndex);
	}



	@Override
	public ROIMaskBuffer getROIDataBuffer() {
		return this.roiMask.getROIDataBuffer();
	}



	@Override
	public String getROILabel(final int colorIndex) {
		return this.roiMask.getROILabel(colorIndex);
	}



	@Override
	@Deprecated
	public ByteBuffer getROIMaskBuffer() {
		return this.roiMask.getROIMaskBuffer();
	}



	@Override
	@Deprecated
	public byte getROIMaskValue(final int xLoc, final int yLoc, final int zLoc) {
		return this.roiMask.getROIMaskValue(xLoc, yLoc, zLoc);
	}



	@Override
	@Deprecated
	public byte getROIMaskValue(final int xLoc, final int yLoc, final int zLoc, final int tLoc) {
		return this.roiMask.getROIMaskValue(xLoc, yLoc, zLoc, tLoc);
	}



	@Override
	@Deprecated
	public byte getROISelectedMask() {
		return this.roiMask.getSelectedROIs();
	}



	@Override
	@Deprecated
	public byte getROIUsedMask() {
		return this.roiMask.getUsedROIs();
	}



	@Override
	public List<Line> getSelectedLines() {
		final List<Line> lines = new ArrayList<Line>();

		lines.addAll(this.axialShapes.getSelectedLines());
		lines.addAll(this.coronalShapes.getSelectedLines());
		lines.addAll(this.sagittalShapes.getSelectedLines());

		return lines;
	}



	@Override
	public long getSelectedMask() {
		return this.roiMask.getSelectedMask();
	}



	@Override
	public List<ROI> getSelectedMasks() {
		return this.roiMask.getSelectedMasks();
	}



	@Override
	public List<Point> getSelectedPoints() {
		return this.points.getSelectedPoints();
	}



	@Override
	public List<ROI> getSelectedROIs() {
		final List<ROI> rois = new ArrayList<ROI>();
		rois.addAll(getSelectedPoints());
		rois.addAll(getSelectedLines());
		rois.addAll(getSelectedMasks());
		return rois;
	}



	@Override
	public Shape[] getShapes(final int sliceDirection, final int slice, final int color) {
		if (sliceDirection == SLICE_DIRECTION_AXIAL) {
			return this.axialShapes.getShapes(slice, color);
		} else if (sliceDirection == SLICE_DIRECTION_CORONAL) {
			return this.coronalShapes.getShapes(slice, color);
		} else if (sliceDirection == SLICE_DIRECTION_SAGITTAL) {
			return this.sagittalShapes.getShapes(slice, color);
		} else {
			return null;
		}
	}



	@Override
	public Coordinate[] getSpecialPoints(final Color color) {
		return this.pointsSpecial.getPoints(color);
	}



	@Override
	public Shape[] getSpecialShapes(final int sliceDirection, final int slice, final Color color) {
		if (sliceDirection == SLICE_DIRECTION_AXIAL) {
			return this.axialShapesP.getShapes(slice, color);
		} else if (sliceDirection == SLICE_DIRECTION_CORONAL) {
			return this.coronalShapesP.getShapes(slice, color);
		} else if (sliceDirection == SLICE_DIRECTION_SAGITTAL) {
			return this.sagittalShapesP.getShapes(slice, color);
		} else {
			return null;
		}
	}



	@Override
	public List<Line> getUsedLines() {
		final List<Line> lines = new ArrayList<Line>();

		lines.addAll(this.axialShapes.getUsedLines());
		lines.addAll(this.coronalShapes.getUsedLines());
		lines.addAll(this.sagittalShapes.getUsedLines());

		return lines;
	}



	@Override
	public long getUsedMask() {
		return this.roiMask.getUsedMask();
	}



	@Override
	public List<ROI> getUsedMasks() {
		return this.roiMask.getUsedMasks();
	}



	@Override
	public List<Point> getUsedPoints() {
		return this.points.getUsedPoints();
	}



	@Override
	public boolean hasSelectedMaskInRange(final int xMin, final int xMax, final int yMin, final int yMax, final int zMin, final int zMax) {
		return this.roiMask.hasSelectedInRange(xMin, xMax, yMin, yMax, zMin, zMax);
	}



	@Override
	public void initMask(final long mask) {
		this.roiMask.initMask(mask);
	}



	@Override
	@Deprecated
	public void initROIMask(final byte mask) {
		this.roiMask.initROIMask(mask);
	}



	@Override
	public boolean isInsideSelectedMaskAtOffset(final int offset) {
		return this.roiMask.isInsideSelectedROIs(offset);
	}



	@Override
	public boolean isSelected(final int color) {
		return (this.roiMask.getSelectedMask() & (1 << color)) != 0;
	}



	@Override
	public boolean isUsed(final int color) {
		return (this.roiMask.getUsedMask() & (1 << color)) != 0;
	}



	@Override
	public void removePoint(final Coordinate point, final int color) {
		this.points.removePoint(point, color);
	}



	@Override
	public void removeShape(final int sliceDirection, final Shape shape) {
		if (sliceDirection == SLICE_DIRECTION_AXIAL) {
			this.axialShapes.removeShape(shape);
		} else if (sliceDirection == SLICE_DIRECTION_CORONAL) {
			this.coronalShapes.removeShape(shape);
		} else if (sliceDirection == SLICE_DIRECTION_SAGITTAL) {
			this.sagittalShapes.removeShape(shape);
		}
	}



	@Override
	public void removeSpecialPoint(final Coordinate point, final Color color) {
		this.pointsSpecial.removePoint(point, color);
	}



	@Override
	public void removeSpecialShape(final int sliceDirection, final Shape shape) {
		if (sliceDirection == SLICE_DIRECTION_AXIAL) {
			this.axialShapesP.removeShape(shape);
		} else if (sliceDirection == SLICE_DIRECTION_CORONAL) {
			this.coronalShapesP.removeShape(shape);
		} else if (sliceDirection == SLICE_DIRECTION_SAGITTAL) {
			this.sagittalShapesP.removeShape(shape);
		}
	}



	@Override
	public void replaceMaskValue(final int xLoc, final int yLoc, final int zLoc, final int tLoc, final long mask) {
		this.roiMask.replaceMaskValue(xLoc, yLoc, zLoc, tLoc, mask);
	}



	@Override
	public void replaceMaskValue(final int xLoc, final int yLoc, final int zLoc, final long mask) {
		this.roiMask.replaceMaskValue(xLoc, yLoc, zLoc, mask);
	}



	@Override
	@Deprecated
	public void replaceROIMaskValue(final int xLoc, final int yLoc, final int zLoc, final byte roiVal) {
		this.roiMask.replaceROIMaskValue(xLoc, yLoc, zLoc, roiVal);
	}



	@Override
	@Deprecated
	public void replaceROIMaskValue(final int xLoc, final int yLoc, final int zLoc, final int tLoc, final byte roiVal) {
		this.roiMask.replaceROIMaskValue(xLoc, yLoc, zLoc, tLoc, roiVal);
	}



	@Override
	public void roiMaskOperationCompleted() {
		this.roiMask.roiMaskOperationCompleted();
	}



	@Override
	public void roiMaskOperationCompleted(final String des) {
		this.roiMask.roiMaskOperationCompleted();
	}



	@Override
	public void setPointHighlightState(final Coordinate point, final Color color, final boolean highlight) {
		this.pointsSpecial.setPointHighlightState(point, color, highlight);
	}



	@Override
	public void setShapeHighlightState(final int sliceDirection, final Shape shape, final boolean highlight) {
		if (sliceDirection == SLICE_DIRECTION_AXIAL) {
			this.axialShapesP.setShapeHighlightState(shape, highlight);
		} else if (sliceDirection == SLICE_DIRECTION_CORONAL) {
			this.coronalShapesP.setShapeHighlightState(shape, highlight);
		} else if (sliceDirection == SLICE_DIRECTION_SAGITTAL) {
			this.sagittalShapesP.setShapeHighlightState(shape, highlight);
		}
	}



	@Override
	public void setUseROISeries(final boolean useROISeries) {
		this.roiMask.setUseROISeries(useROISeries);
	}
}
