
package edu.uthscsa.ric.mango;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.KeyListener;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.File;
import java.net.URI;

import edu.uthscsa.ric.mango.viewerslice.VolumeManager;
import edu.uthscsa.ric.volume.Coordinate;
import edu.uthscsa.ric.volume.ImageVolume;


/**
 * BasicController serves as an interface between MangoPlugin and the viewer window and toolbox. It provides methods to gather information about viewer state,
 * update the viewer, as well as methods to fetch MangoData and other ViewerController objects.
 */
public interface BasicController {

	int SLICE_DIRECTION_AXIAL = VolumeManager.SLICE_DIRECTION_AXIAL;
	int SLICE_DIRECTION_CORONAL = VolumeManager.SLICE_DIRECTION_CORONAL;
	int SLICE_DIRECTION_SAGITTAL = VolumeManager.SLICE_DIRECTION_SAGITTAL;



	/**
	 * Adds a key listener to the viewer.
	 * 
	 * @param kl the key listener to add
	 */
	void addKeyListener(KeyListener kl);



	/**
	 * Adds a mouse listener to a screen slice.
	 * 
	 * @param ml the mouse listener to add
	 * @param sliceDir the slice direction
	 */
	void addMouseListener(MouseListener ml, int sliceDir);



	/**
	 * Adds a mouse motion listener to a screen slice.
	 * 
	 * @param ml the mouse motion listener to add
	 * @param sliceDir the slice direction
	 */
	void addMouseMotionListener(MouseMotionListener ml, int sliceDir);



	/**
	 * Add an overlay to the viewer.
	 * 
	 * @param uri header file
	 * @param imageVolume the volume to overlay
	 * @param colorTable name of color table (if null, color table is automatically selected)
	 * @param transparency transparency
	 */
	void addOverlay(URI uri, ImageVolume imageVolume, String colorTable, double transparency);



	/**
	 * Adds a slice listener, which will be notified anytime a slice's position has changed.
	 * 
	 * @param sl the SliceListener to add
	 * @param sliceDirection the slice direction
	 */
	void addSliceListener(SliceListener sl, int sliceDirection);



	/**
	 * Adds a transform to the viewer.
	 * 
	 * @deprecated
	 * @param type the ImageTransform type
	 * @param mat new transformation matrix as a 4x4 two-dimensional array
	 * @param name a name for the transform
	 */
	@Deprecated
	void addTransform(int type, double[][] mat, String name);



	/**
	 * Build a surface using a default set of parameters.
	 */
	void buildSurface();



	/**
	 * Converts in place a coordinate from index to world space.
	 * 
	 * @param coor the coordinate
	 */
	void convertIndexToWorldCoordinate(Coordinate coor);



	/**
	 * Converts in place a coordinate from world to index space.
	 * 
	 * @param coor the coordinate
	 */
	void convertWorldToIndexCoordinate(Coordinate coor);



	/**
	 * Used to programmatically click a viewer menu item.
	 * 
	 * @param name menu item name
	 */
	void doClickMenuItem(String name);



	/**
	 * Perform "magic wand" preserve operation on main slice. Removes unconnected ROIs, leaving ROI at location unchanged.
	 * 
	 * @param point point in current slice to perform operation
	 */
	void doMagicWandPreserve(Point point);



	/**
	 * Perform "magic wand" remove operation on main slice. Removes ROI at location, leaving unconnected ROIs unchanged.
	 * 
	 * @param point point in current slice to perform operation
	 */
	void doMagicWandRemove(Point point);



	/**
	 * Returns current atlas.
	 * 
	 * @return the atlas, or null if none were found by that name
	 */
	Atlas getAtlas();



	/**
	 * Returns the currently selected coordinate type.
	 * 
	 * @return the currently selected coordinate type
	 */
	int getCoordinateType();



	/**
	 * Returns the current coordinate. The returned values reflect the currently selected coordinate type.
	 * 
	 * @param coor (optional) this Coordinate's values will be replaced with that of the current coordinate
	 * @return a new Coordinate with the values of the current coordinate, or if the param is not null, then coor
	 */
	Coordinate getCurrentCoordinate(Coordinate coor);



	/**
	 * Returns the current coordinate index.
	 * 
	 * @param coor (optional) this Coordinate's values will be replaced with that of the current coordinate
	 * @return a new Coordinate with the values of the current coordinate, or if the param is not null, then coor
	 */
	Coordinate getCurrentIndex(Coordinate coor);



	/**
	 * Returns the index value of the currently selected overlay.
	 * 
	 * @return index value of the currently selected overlay
	 */
	int getCurrentOverlayIndex();



	/**
	 * Returns the current ROI index. [Use getROIColor() instead.]
	 * 
	 * @return the current ROI index
	 * @deprecated
	 */
	@Deprecated
	int getCurrentROI();



	/**
	 * Returns the currently selected timepoint.
	 * 
	 * @return the current timepoint
	 */
	int getCurrentTimepoint();



	/**
	 * Returns the index value of the currently selected transform.
	 * 
	 * @deprecated
	 * @return index value of the currently selected transform
	 */
	@Deprecated
	int getCurrentTransformIndex();



	/**
	 * Returns the image origin.
	 * 
	 * @return the image origin
	 */
	Coordinate getImageOrigin();



	/**
	 * Returns the name of the image.
	 * 
	 * @return the name of the image
	 */
	String getImageTitle();



	/**
	 * Returns a file for the loaded base image of this viewer.
	 * 
	 * @return the loaded file
	 */
	File getLoadedFile();



	/**
	 * Returns a rendering of the main slice view. If operating in a new thread, you may want to call setStopPaintingState() and setNavigationDisabledState()
	 * first. If several slices are needed, successive calls can use getMainImage(BufferedImage bi, int slice).
	 * 
	 * @return the rendered image
	 */
	BufferedImage getMainImage();



	/**
	 * Returns a rendering of the main slice view. If operating in a new thread, you may want to call setStopPaintingState() and setNavigationDisabledState()
	 * first. If several slices are needed, successive calls can use getMainImage(BufferedImage bi, int slice, boolean exclude).
	 * 
	 * @param exclude exclude non-image information (e.g., ROIs)
	 * @return the rendered image
	 */
	BufferedImage getMainImage(boolean exclude);



	/**
	 * Returns a rendering of the main slice view.
	 * 
	 * @param bi the image in which to store the image (typical to call after calling the no-arg method first)
	 * @param slice the slice to render
	 * @return the rendered image
	 */
	BufferedImage getMainImage(BufferedImage bi, int slice);



	/**
	 * Returns a rendering of the main slice view.
	 * 
	 * @param bi the image in which to store the image (typical to call after calling the no-arg method first)
	 * @param slice the slice to render
	 * @param exclude exclude non-image information (e.g., ROIs)
	 * @return the rendered image
	 */
	BufferedImage getMainImage(BufferedImage bi, int slice, boolean exclude);



	/**
	 * Returns a rendering of the main slice view.
	 * 
	 * @param bi the image in which to store the image (typical to call after calling the no-arg method first)
	 * @param slice the slice to render
	 * @param timeIndex the timepoint to render (note, you must manually set the timepoint back to the correct position after this method is called)
	 * @param exclude exclude non-image information (e.g., ROIs)
	 * @return the rendered image
	 */
	BufferedImage getMainImage(BufferedImage bi, int slice, int timeIndex, boolean exclude);



	/**
	 * Returns the slice direction of the main slice view.
	 * 
	 * @return slice direction of the main slice view
	 */
	int getMainSliceDirection();



	/**
	 * Returns the slice position of the main slice view.
	 * 
	 * @return slice position of the main slice view
	 */
	int getMainSlicePosition();



	/**
	 * This can be used to update the MangoData object.
	 * 
	 * @return an updated MangoData object
	 */
	MangoData getMangoData();



	/**
	 * Returns the Mango version.
	 * 
	 * @return version
	 */
	String getMangoVersion();



	/**
	 * Returns the current ROI color index.
	 * 
	 * @return the current ROI color index
	 */
	int getROIColor();



	/**
	 * Returns the current ROI tool code. (See http://rii.uthscsa.edu/mango/webmango_control.html for a list of codes.)
	 * 
	 * @return the current ROI tool code
	 */
	int getROITool();



	/**
	 * Returns the screen transform (can be used to convert mouse coordinates to image coordinates).
	 * 
	 * @param dir the slice direction
	 * @return the screen transform
	 */
	AffineTransform getScreenTransform(int dir);



	/**
	 * Returns the location of the viewer on the screen.
	 * 
	 * @return the location of the viewer
	 */
	Point getViewerLocation();



	/**
	 * Returns the viewer dimensions in pixels.
	 * 
	 * @return the viewer dimensions
	 */
	Dimension getViewerSize();



	/**
	 * Imports an image as an ROI. Any non-zero voxel in the loaded image sets the ROI to the current color at that location. If in world mode, the ROI is
	 * positioned relative to the origin, otherwise it is based on millimeter distance from the LAS corner.
	 * 
	 * @param uri the image file to load
	 * @param forceWorldSpace true if it is necessary that the image be added in world space
	 * @param isMangoROI true if this file represents a MangoROI
	 */
	void importROI(URI uri, boolean forceWorldSpace, boolean isMangoROI);



	/**
	 * Returns true if viewer is in the process of closing.
	 * 
	 * @return true if viewer is closing
	 */
	boolean isClosing();



	/**
	 * Returns true if viewer is doing an ROI or image operation, analyzing statistics, saving, or if navigation is disabled.
	 * 
	 * @return true if viewer is in working state
	 */
	boolean isInWorkingState();



	/**
	 * Returns the VOI mode state. In VOI mode, an ROI is considered as a volume. In ROI mode, an ROI is considered as an area.
	 * 
	 * @return the VOI mode state
	 */
	boolean isMultiSliceMode();



	/**
	 * Returns true if radiological mode is being used.
	 * 
	 * @return true if radiological mode
	 */
	boolean isRadiologicalMode();



	/**
	 * Returns the blue lookup value for the default "Spectrum" lookup table.
	 * 
	 * @param index the index to lookup (0 - 255).
	 * @return the blue lookup value
	 */
	int lookupDefaultBlue(int index);



	/**
	 * Returns the green lookup value for the default "Spectrum" lookup table.
	 * 
	 * @param index the index to lookup (0 - 255).
	 * @return the green lookup value
	 */
	int lookupDefaultGreen(int index);



	/**
	 * Returns the red lookup value for the default "Spectrum" lookup table.
	 * 
	 * @param index the index to lookup (0 - 255).
	 * @return the red lookup value
	 */
	int lookupDefaultRed(int index);



	/**
	 * Create a new ImageVolume object. Useful if plugin is loading volume resources. See ImageVolume.readFiles().
	 * 
	 * @return new ImageVolume object
	 */
	ImageVolume makeNewVolume();



	/**
	 * Makes a static slice overlay for painting by a plugin.
	 * 
	 * @param sliceDirection slice direction of the main slice view
	 * @return the overlay data as an ARGB BufferedImage
	 */
	BufferedImage makePluginOverlay(int sliceDirection);



	/**
	 * Creates a progress meter.
	 * 
	 * @return the progress meter
	 */
	ProgressMeter makeProgressMeter();



	/**
	 * Creates a progress meter.
	 * 
	 * @deprecated
	 * @param des the text to be displayed on the progress meter
	 * @return the progress meter
	 * @see #makeProgressMeter()
	 */
	@Deprecated
	ProgressMeter makeProgressMeter(String des);



	/**
	 * Notifies the viewer that plugin operation has been completed.
	 */
	void pluginOperationCompleted();



	/**
	 * Removes a key listener from the viewer.
	 * 
	 * @param kl the key listener to remove
	 */
	void removeKeyListener(KeyListener kl);



	/**
	 * Removes a mouse listener from a screen slice.
	 * 
	 * @param ml the mouse listener to remove
	 * @param sliceDir the slice direction
	 */
	void removeMouseListener(MouseListener ml, int sliceDir);



	/**
	 * Removes a mouse motion listener from a screen slice.
	 * 
	 * @param ml the mouse motion listener to remove
	 * @param sliceDir the slice direction
	 */
	void removeMouseMotionListener(MouseMotionListener ml, int sliceDir);



	/**
	 * Removes all plugin overlays.
	 */
	void removePluginOverlays();



	/**
	 * Removes a slice listener.
	 * 
	 * @param sl the SliceListener to remove
	 * @param sliceDirection the slice direction
	 */
	void removeSliceListener(SliceListener sl, int sliceDirection);



	/**
	 * Removes the top overlay in the stack.
	 */
	void removeTopOverlay();



	/**
	 * Repaints the viewer.
	 */
	void repaint();



	/**
	 * Set the coordinate type.
	 * 
	 * @param type the coordinate type
	 */
	void setCoordinateType(int type);



	/**
	 * Sets the current coordinate. The values of coor are considered relative to the currently selected coordinate space.
	 * 
	 * @param coor the Coordinate to set the current coordinate values
	 */
	void setCurrentCoordinate(Coordinate coor);



	/**
	 * Sets the current index.
	 * 
	 * @param coor the Coordinate to set the current coordinate values
	 */
	void setCurrentIndex(Coordinate coor);



	/**
	 * Sets the current timepoint.
	 * 
	 * @param timepoint current timepoint
	 */
	void setCurrentTimepoint(int timepoint);



	/**
	 * Sets the current timepoint.
	 * 
	 * @param timepoint current timepoint
	 * @param volume if multiple overlay timeseries are loaded, all other overlays will be aligned in time to this volume
	 */
	void setCurrentTimepoint(int timepoint, ImageVolume volume);



	/**
	 * Sets the image origin.
	 * 
	 * @deprecated
	 * @param coor the new image origin
	 */
	@Deprecated
	void setImageOrigin(Coordinate coor);



	/**
	 * Sets the slice direction of the main slice view.
	 * 
	 * @param sliceDirection slice direction of the main slice view
	 */
	void setMainSliceDirection(int sliceDirection);



	/**
	 * Sets the main slice view position.
	 * 
	 * @param slicePos main slice view position
	 */
	void setMainSlicePosition(int slicePos);



	/**
	 * Disables navigation, puts viewer into "working state".
	 * 
	 * @param bool true to enable navigation disabled state
	 */
	void setNavigationDisabledState(boolean bool);



	/**
	 * Sets the origin.
	 * 
	 * @param coor the new origin
	 */
	void setOriginTo(Coordinate coor);



	/**
	 * Sets the current ROI color.
	 * 
	 * @param color index (0=red, 1=green, etc.)
	 */
	void setROIColor(int color);



	/**
	 * Sets the current ROI tool code:
	 * 
	 * <pre>
	 * Navigation:
	 * 	 0 = navigation
	 * 
	 * Painting tools:
	 *   1 = ellipse edit
	 *   2 = ellipse erase
	 *   3 = ellipse paint
	 *   4 = rectangle edit
	 *   5 = rectangle erase
	 *   6 = rectangle paint
	 *   
	 * Drawing tools:
	 *   7 = draw rectangle
	 *   8 = draw ellipse
	 *   9 = trace region
	 *   10 = trace line
	 *   11 = add point
	 *   
	 * Magic wand tools:
	 *   12 = add region
	 *   13 = preserve region
	 *   14 = remove region
	 *   15 = dilate/erode region
	 *   16 = copy regiom
	 * </pre>
	 * 
	 * @param tool code
	 */
	void setROITool(int tool);



	/**
	 * Used to toggle the display of the lower slice views crosshairs.
	 * 
	 * @param show if true, the lower slice views crosshairs will be displayed
	 */
	void setShowLowerCrosshairs(boolean show);



	/**
	 * Used to toggle the display of the main slice view crosshairs.
	 * 
	 * @param show if true, the main slice view crosshairs will be displayed
	 */
	void setShowMainCrosshairs(boolean show);



	/**
	 * Disables viewer painting.
	 * 
	 * @param bool true to enable "stop painting" state
	 */
	void setStopPaintingState(boolean bool);



	/**
	 * Scans the base image for its minimum and maximum values and updates the viewer with this range. The same as calling
	 * udpateRange(getMangoData().getBaseImage().getScreenImage()).
	 */
	void updateRange();



	/**
	 * Scans a ScreenImage for its minimum and maximum values. If this ScreenImage is currently selected on the toolbox image stack, then its values will be
	 * updated.
	 * 
	 * @param si the ScreenImage to update
	 */
	void updateRange(ScreenImage si);



	/**
	 * Updates the toolbox.
	 */
	void updateToolBox();



	/**
	 * Updates the image and ROI slice data.
	 */
	void updateViewer();



	/**
	 * Toggles on/off the image transform.
	 * 
	 * @param use true to turn on the transform, false otherwise.
	 */
	void useTransform(boolean use);
}
