
package edu.uthscsa.ric.volume.formats.dicom.compression;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import edu.uthscsa.ric.utilities.AppLogger;


public class RLEDecoder {

	private CompressedPart part;
	private InputStream input;
	private ByteBuffer output;
	private ByteBuffer header;
	private int bytesRead;
	private int segElemRead;
	private int numSegments;
	private int[] segmentOffsets;
	private byte[] rawData;
	private boolean littleEndian;
	private int segmentIndex;
	private int numElements;
	private int size;
	public static final int HEADER_SIZE = 64;



	public ByteBuffer decode(final CompressedPart part, final boolean littleEndian, final int numElements) throws IOException {
		this.part = part;
		this.littleEndian = littleEndian;
		this.numElements = numElements;

		prepareInputStream();
		readHeader();
		prepareOutputStream();

		for (int ctr = 0; ctr < numSegments; ctr++) {
			readNextSegment();
		}

		return (ByteBuffer) output.rewind();
	}



	private void readNextSegment() throws IOException {
		prepareInputStream();

		long totalBytesSkipped = input.skip(segmentOffsets[segmentIndex]);

		while (totalBytesSkipped < segmentOffsets[segmentIndex]) {
			totalBytesSkipped += input.skip(segmentOffsets[segmentIndex] - totalBytesSkipped);
		}

		segElemRead = 0;

		try {
			while (hasValidInput()) {
				final byte code = read();

				if ((code >= 0) && (code < 128)) {
					readLiteral(code);
				} else if ((code <= -1) && (code > -128)) {
					readEncoded(code);
				} else if (code == -128) {
					// do nothing
				}
			}
		} catch (final IOException ex) {
			AppLogger.error(ex);
		}

		segmentIndex++;
	}



	public int getNumSegments() {
		return numSegments;
	}



	private void readHeader() throws IOException {
		header = ByteBuffer.allocate(HEADER_SIZE);

		if (littleEndian) {
			header.order(ByteOrder.LITTLE_ENDIAN);
		}

		final byte[] headerArray = header.array();
		int totalBytesRead = input.read(headerArray);

		while (totalBytesRead < HEADER_SIZE) {
			totalBytesRead += input.read(headerArray, totalBytesRead, HEADER_SIZE - totalBytesRead);
		}

		header.rewind();
		numSegments = header.getInt();
		size = numElements * numSegments;
		segmentOffsets = new int[numSegments];

		for (int ctr = 0; ctr < numSegments; ctr++) {
			segmentOffsets[ctr] = header.getInt();
		}
	}



	private void prepareOutputStream() {
		output = ByteBuffer.allocate(size);

		if (littleEndian) {
			output.order(ByteOrder.LITTLE_ENDIAN);
		}
	}



	private void prepareInputStream() throws IOException {
		if (rawData == null) {
			rawData = CompressedPart.concatenateCompressedInternalParts(part, part.getUri());
		}

		input = new ByteArrayInputStream(rawData);
	}



	private void put(final byte val) throws IOException {
		if (output.remaining() == 0) {
			throw new IOException("RLE buffer overflow!");
		}

		bytesRead++;
		segElemRead++;
		output.put(val);
	}



	private boolean hasValidInput() throws IOException {
		return ((input.available() > 0) && (bytesRead < size) && (segElemRead < numElements));
	}



	private byte read() throws IOException {
		if (!hasValidInput()) {
			throw new IOException("RLE buffer underflow! " + bytesRead);
		}

		return ((byte) input.read());
	}



	private void readEncoded(final byte code) throws IOException {
		final int runLength = (1 - code);
		final byte encoded = read();

		for (int ctr = 0; ctr < runLength; ctr++) {
			put(encoded);
		}
	}



	private void readLiteral(final byte code) throws IOException {
		final int length = (code + 1);

		if (hasValidInput()) {
			for (int ctr = 0; ctr < length; ctr++) {
				put(read());
			}
		} else {
			AppLogger.warn("Insufficient RLE data!");
		}
	}



	public int getBytesRead() {
		return bytesRead;
	}
}
