
package edu.uthscsa.ric.volume.formats.dicom;

import icc.ICCProfileException;

import java.awt.image.DataBuffer;
import java.awt.image.Raster;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import colorspace.ColorSpaceException;
import edu.uthscsa.ric.utilities.AppLogger;
import edu.uthscsa.ric.utilities.CollectionUtilities;
import edu.uthscsa.ric.utilities.CompressionUtilities;
import edu.uthscsa.ric.utilities.DateUtilities;
import edu.uthscsa.ric.utilities.FileUtilities;
import edu.uthscsa.ric.volume.Coordinate;
import edu.uthscsa.ric.volume.EditableHeader;
import edu.uthscsa.ric.volume.ImageDescription;
import edu.uthscsa.ric.volume.ImageDimensions;
import edu.uthscsa.ric.volume.ImageRange;
import edu.uthscsa.ric.volume.ImageTransform;
import edu.uthscsa.ric.volume.ImageType;
import edu.uthscsa.ric.volume.InvalidHeaderException;
import edu.uthscsa.ric.volume.ReadableHeader;
import edu.uthscsa.ric.volume.VolumeIOException;
import edu.uthscsa.ric.volume.VoxelDimensions;
import edu.uthscsa.ric.volume.formats.dicom.compression.CompressedPart;
import edu.uthscsa.ric.volume.formats.dicom.compression.J2000Decoder;
import edu.uthscsa.ric.volume.formats.dicom.compression.RLEDecoder;
import edu.uthscsa.ric.volume.formats.jpeg.JPEGLosslessDecoder;


public class Series implements EditableHeader, ReadableHeader {

	private ByteBuffer[] buffers;
	private List<DICOM> dicoms;
	private URI selectedURI;
	private boolean isImplicitTimeseries;
	private boolean isMosaic;
	private boolean isMultiFrame;
	private boolean isMultiFrameTimeseries;
	private boolean isMultiFrameVolume;
	private boolean mosaic;
	private int mosaicCols;
	private int mosaicHeight;
	private int mosaicRows;
	private int mosaicWidth;
	private int numSlices;
	private int numberOfFrames;
	private int numberOfFramesInFile;

	private final List<CompressedPart> compressedParts = new ArrayList<CompressedPart>();
	private final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyyMMdd");
	private final SimpleDateFormat DATE_FORMAT2 = new SimpleDateFormat("yyyy.MM.dd");

	private static final List<String> SUPPORTED_TRASFER_SYNTAX = new ArrayList<String>();
	private static final byte[] MAGIC_NUMBER = { 0x44, 0x49, 0x43, 0x4D };
	private static boolean skipFindAssociated;

	public static final DICOMFileFilter DICOM_FILE_FILTER = new DICOMFileFilter(true);
	public static final List<String> COMMON_EXTENSIONS = CollectionUtilities.immutable("dcm", "img", "dic");
	public static final String NAME = "DICOM";
	public static final String PREFERRED_EXTENSION = ".dcm";
	public static final int MAGIC_NUMBER_POSITION = 128;
	public static final int BUFFER_SIZE = 2048;
	public static final String DEFAULT_ORIENTATION = "XYZ+--";

	private static final List<Tag> ANONYMIZABLE_TAGS = Collections.unmodifiableList(Arrays.asList(Dictionary.TAG_0008_INSTITUTION_NAME,
			Dictionary.TAG_0008_INSTITUTION_ADDRESS, Dictionary.TAG_0008_REFERRING_PHYSICIAN_NAME, Dictionary.TAG_0010_PATIENT_NAME,
			Dictionary.TAG_0010_PATIENT_ID, Dictionary.TAG_0010_PATIENT_BIRTH_DATE, Dictionary.TAG_0010_PATIENT_OTHER_ID,
			Dictionary.TAG_0010_PATIENT_OTHER_NAMES, Dictionary.TAG_0010_PATIENT_BIRTH_NAME, Dictionary.TAG_0010_PATIENT_ADDRESS));

	static {
		// normal
		SUPPORTED_TRASFER_SYNTAX.add(Constants.TRANSFER_SYNTAX_ID_EXPLICIT_BIG);
		SUPPORTED_TRASFER_SYNTAX.add(Constants.TRANSFER_SYNTAX_ID_IMPLICIT);
		SUPPORTED_TRASFER_SYNTAX.add(Constants.TRANSFER_SYNTAX_ID_EXPLICIT_LITTLE);

		// compressed
		SUPPORTED_TRASFER_SYNTAX.add(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_RLE);
		SUPPORTED_TRASFER_SYNTAX.add(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_8BIT);
		SUPPORTED_TRASFER_SYNTAX.add(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_LOSSLESS);
		SUPPORTED_TRASFER_SYNTAX.add(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_LOSSLESS_SEL1);
		SUPPORTED_TRASFER_SYNTAX.add(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_2000_LOSSLESS);
		SUPPORTED_TRASFER_SYNTAX.add(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_2000);
	}



	private static int getMosaicOffset(final int mosaicCols, final int mosaicColWidth, final int mosaicRowHeight, final int mosaicWidth, final int xLocVal,
			final int yLocVal, final int zLocVal) {
		int xLoc = xLocVal;
		int yLoc = yLocVal;
		final int zLoc = zLocVal;

		xLoc = ((zLoc % mosaicCols) * mosaicColWidth) + xLoc;
		yLoc = (((zLoc / mosaicCols) * mosaicRowHeight) + yLoc) * mosaicWidth;

		return (xLoc + yLoc);
	}



	public void buildDCM(final DICOM selectedDICOM, final List<DICOM> dicoms) throws InvalidHeaderException, IOException {
		DICOM[] orderedDICOMs = null;
		int buildCount = 0;
		int originalSize = 0;

		do {
			this.dicoms = dicoms;
			selectedURI = selectedDICOM.getImageFile();

			if (!isSupportedTransferSyntax()) {
				throw new InvalidHeaderException("Unsupported transfer syntax!");
			}

			isMosaic = selectedDICOM.isMosaic();

			if (isMosaic) {
				final int[] matSize = selectedDICOM.getAcquisitionMatrix();
				setMosaic(isMosaic);
				setMosaicCols(selectedDICOM.getCols() / matSize[1]);
				setMosaicRows(selectedDICOM.getRows() / matSize[0]);
				setMosaicWidth(selectedDICOM.getCols());
				setMosaicHeight(selectedDICOM.getRows());
			}

			// figure out if multiple frames are used
			numberOfFrames = selectedDICOM.getNumberOfFrames();
			isMultiFrame = (numberOfFrames > 1) || (isMosaic && (dicoms.size() > 1));
			isMultiFrameVolume = false;
			isMultiFrameTimeseries = false;
			isImplicitTimeseries = false;
			numberOfFramesInFile = 1;

			if (selectedDICOM.getSliceSizeBytes() == 0) {
				throw new InvalidHeaderException("No data was found!");
			}

			if (!selectedDICOM.isCompressed()) {
				numberOfFramesInFile = (FileUtilities.getLength(selectedURI) - selectedDICOM.getImageOffset()) / selectedDICOM.getSliceSizeBytes();
			}

			if (isMultiFrame) {
				final boolean hasFrameTime = (selectedDICOM.getFrameTime() > 0);
				if (isMosaic) {
					isMultiFrameTimeseries = true;
				} else {
					if (hasFrameTime) {
						isMultiFrameTimeseries = true;
					} else if (numberOfFramesInFile > 1) {
						isMultiFrameTimeseries = true;
						numberOfFrames = dicoms.size();
					} else {
						isMultiFrameVolume = true;
					}
				}
			}

			if (!isMosaic && (numberOfFrames <= 1)) { // check for implicit frame count
				final String sliceLoc = selectedDICOM.getImagePosition();
				numberOfFrames = 0;

				if (StringUtils.isNotBlank(sliceLoc) && selectedDICOM.hasValidImagePosition()) {
					for (final DICOM element : dicoms) {
						if (element.getImagePosition().equals(sliceLoc)) {
							numberOfFrames++;
						}
					}
				}

				if (numberOfFrames > 1) {
					isImplicitTimeseries = true;
				}
			}

			if (isMultiFrameTimeseries && (dicoms.size() == 1)
					&& ((dicoms.get(0).getNumTemporalPositions() > 1) || !DICOM.hasValidTemporalPosition(dicoms.toArray(new DICOM[dicoms.size()])))) {
				numberOfFrames = Math.max(1, dicoms.get(0).getNumTemporalPositions());
				numberOfFramesInFile = numberOfFramesInFile / numberOfFrames;
			}

			// order the DICOMs in space and time
			if (dicoms.size() > 1) {
				final String sliceDir = selectedDICOM.makeImageOrientationLabelFromImageOrientationPatient().toLowerCase();
				orderedDICOMs = DICOM.orderDicoms(dicoms.toArray(new DICOM[dicoms.size()]), numberOfFrames, sliceDir);
			} else {
				orderedDICOMs = new DICOM[] { dicoms.get(0) };
			}

			originalSize = dicoms.size();
			dicoms.clear();
			dicoms.addAll(Arrays.asList(orderedDICOMs));
			dicoms.removeAll(Collections.singleton(null));

			buildCount++;
		} while ((orderedDICOMs.length != originalSize) && (buildCount < 2));

		readEncapsulatedData();
	}



	public void buildDICOMFromURIs(final URI selectedURI, final List<URI> possibleURIs) throws IOException, InvalidHeaderException {
		final DICOM selectedDICOM = new DICOM(selectedURI);

		final List<DICOM> dicoms = new ArrayList<DICOM>();
		dicoms.add(selectedDICOM);

		final String selectedID = selectedDICOM.getSessionID() + selectedDICOM.getVolumeID();

		if (!isSkipFindAssociated()) {
			for (final URI uri : possibleURIs) {
				if (uri.equals(selectedURI)) {
					continue;
				}

				final DICOM dicom = new DICOM(uri);
				final String id = dicom.getSessionID() + dicom.getVolumeID();

				if (selectedID.equals(id) && (dicom.getSliceSizeBytes() > 0)) {
					dicoms.add(dicom);
				}
			}
		}

		buildDCM(selectedDICOM, dicoms);
	}



	public String getAnonymizedDicomName() {
		String modality = "UN";

		if (dicoms != null) {
			if (getDicom(0).getModality() != null) {
				modality = getDicom(0).getModality();
			}
		}

		return (modality + "-" + DateUtilities.todaySimple());
	}



	public List<CompressedPart> getCompressedParts() {
		return compressedParts;
	}



	@Deprecated
	@Override
	public ByteBuffer getImage() {
		return null;
	}



	@Override
	public ByteBuffer[] getImageBuffers() {
		return getImageBuffers(false);
	}



	public ByteBuffer[] getImageBuffers(final boolean forceSingleIteration) {
		if (buffers == null) {
			final boolean isMosaic = isMosaic();
			final boolean isDicomJpegBaseline = isJPEGBaseline();
			final boolean isDicomJpegLossless = isJPEGLossless();
			final boolean isDicomJpeg2000 = isJPEG2000();
			final boolean isDicomRle = isRLE();

			if (isMosaic) {
				buffers = readMosaic();
			} else if (isDicomJpegBaseline) {
				buffers = readJpegBaseline();
			} else if (isDicomJpegLossless) {
				buffers = readJpegLossless();
			} else if (isDicomJpeg2000) {
				buffers = readJpeg2000();
			} else if (isDicomRle) {
				buffers = readRLE();
			} else if (getImageDimensions().getTimepoints() > dicoms.size()) {
				buffers = readSeries();
			} else {
				buffers = readStandard(forceSingleIteration);
			}
		}

		return buffers;
	}



	@Override
	public ImageDescription getImageDescription() {
		final Date studyDate = getMostRecentDate(0);

		final ImageDescription id = new ImageDescription(getPatientID(0), getPatientName(0), studyDate, getImageDescription(0));

		final URI[] dicomURIs = new URI[dicoms.size()];
		for (int ctr = 0; ctr < dicoms.size(); ctr++) {
			dicomURIs[ctr] = dicoms.get(ctr).getImageFile();
		}

		id.setAssociatedFiles(dicomURIs);

		String title = "";

		if (getDicom(0).getStudyDescription() != null) {
			title += getDicom(0).getStudyDescription();
		}

		if (getDicom(0).getSeriesDescription() != null) {
			title += getDicom(0).getSeriesDescription();
		}

		title = title.trim();

		if (StringUtils.isNotBlank(title)) {
			id.setTitle(title);
		}

		id.setFile(selectedURI);
		id.setImageTypeDescription(Constants.SOP_CLASS_DEFINITIONS.get(dicoms.get(0).getSOPClass()));
		id.setDicomItems(dicoms.get(0).generateDicomItems());

		return id;
	}



	@Override
	public ImageDimensions getImageDimensions() {
		ImageDimensions id = null;

		final DICOM dicom = dicoms.get(0);

		if (isMosaic) {
			int numberOfSlices = dicom.getNumberOfMosaicImages();
			if (numberOfSlices == -1) {
				numberOfSlices = getMosaicCols() * getMosaicRows();
			}

			id = new ImageDimensions(getMosaicWidth() / getMosaicCols(), getMosaicHeight() / getMosaicRows(), numberOfSlices, dicoms.size());
			id.setTiled(true);
		} else if (isMultiFrameVolume) {
			id = new ImageDimensions(dicom.getCols(), dicom.getRows(), numberOfFrames, 1);
		} else if (isMultiFrameTimeseries) {
			id = new ImageDimensions(dicom.getCols(), dicom.getRows(), numberOfFramesInFile, numberOfFrames);
		} else if (isImplicitTimeseries) {
			id = new ImageDimensions(dicom.getCols(), dicom.getRows(), dicoms.size() / numberOfFrames, numberOfFrames);
		} else {
			id = new ImageDimensions(dicom.getCols(), dicom.getRows(), dicoms.size(), 1);
		}

		final int[] offsets = new int[dicoms.size()];
		for (int ctr = 0; ctr < dicoms.size(); ctr++) {
			offsets[ctr] = dicoms.get(ctr).getImageOffset();
		}

		id.setImageOffset(offsets[0]);
		id.setImageOffsets(offsets);
		setNumSlices(id.getSlices());

		return id;
	}



	@Override
	public URI getImageFile() {
		return selectedURI;
	}



	@Override
	public ImageRange getImageRange() {
		ImageRange ir = null;

		double gMax = 0;
		double gMin = 0;

		final float[] slopes = new float[dicoms.size()];
		final float[] intercepts = new float[dicoms.size()];

		for (int ctr = 0; ctr < dicoms.size(); ctr++) {
			final DICOM dicom = dicoms.get(ctr);
			final double max = dicom.getImageMax();
			final double min = dicom.getImageMin();
			final double slope = dicom.getDataScaleSlope();
			final double intercept = dicom.getDataScaleIntercept();

			slopes[ctr] = (float) slope;
			intercepts[ctr] = (float) intercept;

			if (ctr == 0) {
				gMax = (max * slope) + intercept;
				gMin = (min * slope) + intercept;
			} else {
				if (((max * slope) + intercept) > gMax) {
					gMax = (max * slope) + intercept;
				}

				if (((min * slope) + intercept) < gMin) {
					gMin = (min * slope) + intercept;
				}
			}
		}

		double windowWidth = 0;
		double windowCenter = 0;

		for (int ctr = 0; ctr < dicoms.size(); ctr++) {
			final DICOM dicom = dicoms.get(ctr);

			if (ctr == 0) {
				windowWidth = (dicom.getWindowWidth());
				windowCenter = (dicom.getWindowCenter());
			} else {
				if (windowCenter < dicom.getWindowCenter()) {
					windowWidth = (dicom.getWindowWidth());
					windowCenter = (dicom.getWindowCenter());
				}
			}
		}

		ir = new ImageRange(gMin, gMax, windowCenter - (windowWidth / 2), windowCenter + (windowWidth / 2));

		final ImageDimensions id = getImageDimensions();

		if (isMosaic) {
			final int numMosaicSlicesVolume = id.getSlices();
			final int numMosaicSlicesTotal = numMosaicSlicesVolume * dicoms.size();

			final float[] mosaicSlopes = new float[numMosaicSlicesTotal];
			final float[] mosaicIntercepts = new float[numMosaicSlicesTotal];

			for (int ctr = 0; ctr < numMosaicSlicesTotal; ctr++) {
				mosaicSlopes[ctr] = (float) dicoms.get(ctr / numMosaicSlicesVolume).getDataScaleSlope();
				mosaicIntercepts[ctr] = (float) dicoms.get(ctr / numMosaicSlicesVolume).getDataScaleIntercept();
			}

			ir.setDataScaleSlopes(mosaicSlopes);
			ir.setDataScaleIntercepts(mosaicIntercepts);
		} else if (isMultiFrame) {
			final int numSlices = id.getSlices() * id.getTimepoints();
			final float[] seriesSlopes = new float[numSlices];
			final float[] seriesIntercepts = new float[numSlices];
			final int ratio = seriesSlopes.length / dicoms.size();

			for (int ctr = 0; ctr < dicoms.size(); ctr++) {
				final float slope = (float) dicoms.get(ctr).getDataScaleSlope();
				final float intercept = (float) dicoms.get(ctr).getDataScaleIntercept();

				for (int ctrInner = 0; ctrInner < ratio; ctrInner++) {
					seriesSlopes[(ctr * ratio) + ctrInner] = slope;
					seriesIntercepts[(ctr * ratio) + ctrInner] = intercept;
				}
			}

			ir.setDataScaleSlopes(seriesSlopes);
			ir.setDataScaleIntercepts(seriesIntercepts);
		} else if (isImplicitTimeseries) {
			final int numSlicesTotal = id.getSlices() * id.getTimepoints();
			if (dicoms.size() != numSlicesTotal) {
				ir.setGlobalDataScaleSlope(numberOfFrames, (float) dicoms.get(0).getDataScaleSlope());
				ir.setGlobalDataScaleIntercept(numberOfFrames, (float) dicoms.get(0).getDataScaleIntercept());
			} else {
				ir.setDataScaleSlopes(slopes);
				ir.setDataScaleIntercepts(intercepts);
			}
		} else {
			ir.setDataScaleSlopes(slopes);
			ir.setDataScaleIntercepts(intercepts);
		}

		return ir;
	}



	@Override
	public ImageType getImageType() {
		final DICOM dicom = dicoms.get(0);

		final ImageType it = new ImageType(dicom.getBitsAllocated() / 8, dicom.getDataType(), dicom.getBitsStored(), dicom.isLittleEndian());
		it.setRGBBySample(dicom.getPlanerConfig() == 1);
		it.setRGBPalette(dicom.isPalette());
		it.setRGBMode((dicom.getDataType() == ImageType.BYTE_TYPE_RGB) || (dicom.isPalette()));

		it.setReadOnly((getImageDimensions().getNumVoxelsVolume() * 4) < 0); // if overflow, set as readOnly and open as 2-byte integer image

		if (isCompressed()) {
			if (isJPEGBaseline()) {
				it.setCompressionType(ImageType.COMPRESSION_TYPE_DICOM_JPEG_BASELINE);
			} else if (isJPEGLossless()) {
				it.setCompressionType(ImageType.COMPRESSION_TYPE_DICOM_JPEG_LOSSLESS);
			} else if (isJPEG2000()) {
				it.setCompressionType(ImageType.COMPRESSION_TYPE_DICOM_JPEG2000);
			} else if (isRLE()) {
				it.setCompressionType(ImageType.COMPRESSION_TYPE_DICOM_RLE);
			}
		} else {
			it.setCompressionType(ImageType.COMPRESSION_TYPE_NONE);
		}

		return it;
	}



	@Override
	public String getMinimumVersionSupported() {
		return null;
	}



	public int getMosaicCols() {
		return mosaicCols;
	}



	public int getMosaicHeight() {
		return mosaicHeight;
	}



	public int getMosaicRows() {
		return mosaicRows;
	}



	public int getMosaicWidth() {
		return mosaicWidth;
	}



	public int getNumSlices() {
		return numSlices;
	}



	@Override
	public String getOrientation() {
		String orientation = dicoms.get(0).getOrientation();

		if (orientation == null) {
			orientation = DEFAULT_ORIENTATION;
		}

		final String sliceDir = dicoms.get(0).makeImageOrientationLabelFromImageOrientationPatient().toLowerCase();
		final double sliceLocationFirst = dicoms.get(0).getImagePosition(sliceDir);

		boolean sliceSense;

		if (isMosaic) {
			sliceSense = true;
		} else if (isMultiFrame) {
			final double[] sliceLocations = dicoms.get(0).getSliceLocationVector();
			if (sliceLocations != null) {
				if (orientation.charAt(2) == 'Z') {
					sliceSense = (sliceLocations[0] - sliceLocations[sliceLocations.length - 1]) < 0;
				} else {
					sliceSense = (sliceLocations[0] - sliceLocations[sliceLocations.length - 1]) > 0;
				}
			} else {
				final List<Double> slicePositions = getImagePositions(sliceDir, 0);

				if ((slicePositions != null) && (slicePositions.size() > 0)) {
					final double first = slicePositions.get(0);
					final double last = slicePositions.get(slicePositions.size() - 1);
					final double sliceLocDiff = last - first;
					/*
					 * "The direction of the axes is defined fully by the patient's orientation. The x-axis is increasing to the left hand side of the patient. The
					 * y-axis is increasing to the posterior side of the patient. The z-axis is increasing toward the head of the patient."
					 */
					if ((sliceDir != null) && (sliceDir.equals("sagittal") || (sliceDir.equals("coronal")))) {
						if (sliceLocDiff > 0) {
							sliceSense = false;
						} else {
							sliceSense = true;
						}
					} else {
						if (sliceLocDiff > 0) {
							sliceSense = true;
						} else {
							sliceSense = false;
						}
					}
				} else {
					sliceSense = sliceLocationFirst < 0 ? false : true; // maybe???
				}
			}
		} else {
			final double sliceLocationLast = dicoms.get(dicoms.size() - 1).getImagePosition(sliceDir);
			final double sliceLocDiff = sliceLocationLast - sliceLocationFirst;
			/*
			 * "The direction of the axes is defined fully by the patient's orientation. The x-axis is increasing to the left hand side of the patient. The
			 * y-axis is increasing to the posterior side of the patient. The z-axis is increasing toward the head of the patient."
			 */
			if ((sliceDir != null) && (sliceDir.equals("sagittal") || (sliceDir.equals("coronal")))) {
				if (sliceLocDiff > 0) {
					sliceSense = false;
				} else {
					sliceSense = true;
				}
			} else {
				if (sliceLocDiff > 0) {
					sliceSense = true;
				} else {
					sliceSense = false;
				}
			}
		}

		orientation = orientation.substring(0, 5) + (sliceSense ? '+' : '-'); // this fixes the cross-slice orientation sense (i hope)

		return orientation;
	}



	public List<Integer> getTemporalPositions() {
		final List<Integer> positions = new ArrayList<>();

		for (final DICOM dicom : dicoms) {
			positions.addAll(dicom.getTemporalPositions());
		}

		return positions;
	}



	public List<Double> getImagePositions(final String dir) {
		final List<Double> positions = new ArrayList<>();

		for (final DICOM dicom : dicoms) {
			positions.addAll(dicom.getImagePositions(dir));
		}

		return positions;
	}



	public List<Double> getImagePositions(final String dir, final int temporalPos) {
		final List<Double> imagePositions = getImagePositions(dir);
		final List<Integer> temporalPositions = getTemporalPositions();
		final List<Double> imagePositionsForTempPos = new ArrayList<>();

		final int numPositions = imagePositions.size();

		if (numPositions != temporalPositions.size()) {
			AppLogger.warn("The number of image positions does not equal the number of temporal positions!");
			return null;
		}

		for (int ctr = 0; ctr < numPositions; ctr++) {
			final int tempPos = temporalPositions.get(ctr) - 1;

			if (tempPos == temporalPos) {
				imagePositionsForTempPos.add(imagePositions.get(ctr));
			}
		}

		return imagePositionsForTempPos;
	}



	@Override
	public int getOrientationCertainty() {
		int orientationCertainty = ORIENTATION_CERTAINTY_HIGH;

		if (dicoms.get(0).getOrientation() == null) {
			orientationCertainty = ORIENTATION_CERTAINTY_LOW;
		} else if (isMosaic || isMultiFrameVolume) {
			orientationCertainty = ORIENTATION_CERTAINTY_MID;
		}

		return orientationCertainty;
	}



	@Override
	public Coordinate getOrigin() {
		final ImageDimensions id = getImageDimensions();

		final int[] values = new int[3];
		values[0] = id.getCols();
		values[1] = id.getRows();
		values[2] = id.getSlices();

		final String orientation = getOrientation();
		final int[] ordered = new int[3];
		ordered[0] = values[orientation.indexOf('X')];
		ordered[1] = values[orientation.indexOf('Y')];
		ordered[2] = values[orientation.indexOf('Z')];

		final Coordinate origin = new Coordinate(Math.round(ordered[0] / 2.0), Math.round(ordered[1] / 2.0), Math.round(ordered[2] / 2.0));
		return origin;
	}



	@Override
	public String getPluginName() {
		return NAME;
	}



	@Override
	public URL getPluginURL() {
		return null;
	}



	@Override
	public String getPreferredFileExtension() {
		return PREFERRED_EXTENSION;
	}



	@Override
	public float[] getPreviewImage() {
		return CollectionUtilities.EMPTY_FLOAT_ARRAY;
	}



	@Override
	public ImageTransform[] getTransforms() {
		return new ImageTransform[0];
	}



	@Override
	public String getVersion() {
		return null;
	}



	@Override
	public VoxelDimensions getVoxelDimensions() {
		VoxelDimensions vd = null;
		final DICOM dicom = dicoms.get(0);

		final double sliceSpacing = Math.max(dicom.getSliceSpacing(), dicom.getSliceThickness());

		if (isMosaic || isMultiFrame) {
			vd = new VoxelDimensions(dicom.getColSpacing(), dicom.getRowSpacing(), sliceSpacing, dicom.getTR() / 1000.0);
		} else {
			if (dicoms.size() == 1) {
				vd = new VoxelDimensions(dicom.getColSpacing(), dicom.getRowSpacing(), sliceSpacing, dicom.getTR() / 1000.0);
			} else {
				double sliceDis = Math.abs(dicoms.get(0).getSliceLocation() - dicoms.get(1).getSliceLocation());

				if (sliceDis == 0) {
					sliceDis = Math.max(dicom.getSliceSpacing(), dicom.getSliceThickness());
				}

				vd = new VoxelDimensions(dicom.getColSpacing(), dicom.getRowSpacing(), sliceDis, dicom.getTR() / 1000.0);
			}
		}

		if (vd.getRowSize(true) == 0) {
			vd.setRowSize(1);
		}

		if (vd.getColSize(true) == 0) {
			vd.setColSize(1);
		}

		if (vd.getSliceThickness(true) == 0) {
			vd.setSliceThickness(1);
		}

		vd.setSpatialUnit(VoxelDimensions.UNITS_MM);
		vd.setTemporalUnit(VoxelDimensions.UNITS_SEC);
		vd.setTE1(dicom.getTE());

		return vd;
	}



	@Override
	public boolean hasNewerVersion() {
		return false;
	}



	public boolean isCompressed() {
		final String xferSyntax = dicoms.get(0).getTransferSyntaxID();
		return ((xferSyntax.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_PREFIX) != -1) || (xferSyntax
				.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_RLE) != -1));
	}



	public boolean isJPEG2000() {
		final String xferSyntax = dicoms.get(0).getTransferSyntaxID();
		return ((xferSyntax.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_2000) != -1) || (xferSyntax
				.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_2000_LOSSLESS) != -1));
	}



	public boolean isJPEGBaseline() {
		final String xferSyntax = dicoms.get(0).getTransferSyntaxID();
		return ((xferSyntax.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_8BIT) != -1) || (xferSyntax
				.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_12BIT) != -1));
	}



	public boolean isJPEGLossless() {
		final String xferSyntax = dicoms.get(0).getTransferSyntaxID();
		return ((xferSyntax.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_LOSSLESS_SEL1) != -1) || (xferSyntax
				.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_LOSSLESS) != -1));
	}



	public boolean isJPEGLS() {
		final String xferSyntax = dicoms.get(0).getTransferSyntaxID();
		return ((xferSyntax.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_LS) != -1) || (xferSyntax
				.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_LS_LOSSLESS) != -1));
	}



	public boolean isMosaic() {
		return mosaic;
	}



	public boolean isRLE() {
		final String xferSyntax = dicoms.get(0).getTransferSyntaxID();
		return (xferSyntax.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_RLE) != -1);
	}



	public boolean isSupportedTransferSyntax() {
		return SUPPORTED_TRASFER_SYNTAX.contains(dicoms.get(0).getTransferSyntaxID());
	}



	@Override
	public boolean isThisFormat(final File file) {
		BufferedInputStream fileIn = null;
		ByteBuffer byteBuffer = null;
		final byte[] byteArray = new byte[MAGIC_NUMBER_POSITION + MAGIC_NUMBER.length];

		try {
			fileIn = FileUtilities.getInputStream(file.toURI(), false);
			FileUtilities.readFully(fileIn, byteArray);
			byteBuffer = ByteBuffer.wrap(byteArray);

			// sometimes "DICM" is at the beginning of the file...
			final byte[] magicDicom1 = new byte[4];
			magicDicom1[0] = byteBuffer.get(0);
			magicDicom1[1] = byteBuffer.get(1);
			magicDicom1[2] = byteBuffer.get(2);
			magicDicom1[3] = byteBuffer.get(3);

			// ... and other times, it's 128 bytes in
			final byte[] magicDicom2 = new byte[4];
			magicDicom2[0] = byteBuffer.get(128);
			magicDicom2[1] = byteBuffer.get(129);
			magicDicom2[2] = byteBuffer.get(130);
			magicDicom2[3] = byteBuffer.get(131);

			final byte[] dcmMagicNumber = MAGIC_NUMBER;

			if (((magicDicom1[0] == dcmMagicNumber[0]) && (magicDicom1[1] == dcmMagicNumber[1]) && (magicDicom1[2] == dcmMagicNumber[2]) && (magicDicom1[3] == dcmMagicNumber[3]))
					|| ((magicDicom2[0] == dcmMagicNumber[0]) && (magicDicom2[1] == dcmMagicNumber[1]) && (magicDicom2[2] == dcmMagicNumber[2]) && (magicDicom2[3] == dcmMagicNumber[3]))) {
				return true;
			}
		} catch (final IOException ex) {
			AppLogger.error(ex);
		} finally {
			try {
				if (fileIn != null) {
					fileIn.close();
				}
			} catch (final IOException ex) {
				AppLogger.warn(ex);
			}
		}

		return false;
	}



	@Override
	public void readHeader(final File file, final File tempDir) throws InvalidHeaderException, IOException {
		readHeader(file.toURI(), tempDir);
	}



	@Override
	public void readHeader(final URI selectedURI, final File tempDir) throws InvalidHeaderException, IOException {
		final List<URI> possibleURIs = new ArrayList<URI>();

		if (FileUtilities.uriIsFile(selectedURI)) {
			final File selectedFile = new File(selectedURI);
			final File[] possibleFiles = selectedFile.getParentFile().listFiles((FileFilter) new DICOMFileFilter(false));

			if (CollectionUtilities.isNotEmpty(possibleFiles)) {
				for (final File possibleFile : possibleFiles) {
					possibleURIs.add(possibleFile.toURI());
				}
			}
		}

		buildDICOMFromURIs(selectedURI, possibleURIs);
	}



	public void setMosaic(final boolean bool) {
		mosaic = bool;
	}



	public void setMosaicCols(final int val) {
		mosaicCols = val;
	}



	public void setMosaicHeight(final int val) {
		mosaicHeight = val;
	}



	public void setMosaicRows(final int val) {
		mosaicRows = val;
	}



	public void setMosaicWidth(final int val) {
		mosaicWidth = val;
	}



	public void setNumSlices(final int val) {
		numSlices = val;
	}



	@Override
	public String toString() {
		return dicoms.get(0).toString();
	}



	private void addCompressedPart(final CompressedPart jpeg) {
		compressedParts.add(jpeg);
	}



	public DICOM getDicom(final int index) {
		return dicoms.get(index);
	}



	private DICOM getDicom(final URI uri) {
		for (final DICOM dicom : dicoms) {
			if (dicom.getImageFile().equals(uri)) {
				return dicom;
			}
		}

		return null;
	}



	private String getImageDescription(final int sliceNum) {
		if (mosaic || (sliceNum >= dicoms.size())) {
			return dicoms.get(0).getImageDescription();
		} else {
			return dicoms.get(sliceNum).getImageDescription();
		}
	}



	private String getPatientID(final int sliceNum) {
		if (mosaic || (sliceNum >= dicoms.size())) {
			return dicoms.get(0).getPatientID();
		} else {
			return dicoms.get(sliceNum).getPatientID();
		}
	}



	private String getPatientName(final int sliceNum) {
		if (mosaic || (sliceNum >= dicoms.size())) {
			return dicoms.get(0).getPatientName();
		} else {
			return dicoms.get(sliceNum).getPatientName();
		}
	}



	private String getStudyDate(final int sliceNum) {
		if (mosaic || (sliceNum >= dicoms.size())) {
			return dicoms.get(0).getStudyDate();
		} else {
			return dicoms.get(sliceNum).getStudyDate();
		}
	}



	private String getSeriesDate(final int sliceNum) {
		if (mosaic || (sliceNum >= dicoms.size())) {
			return dicoms.get(0).getSeriesDate();
		} else {
			return dicoms.get(sliceNum).getSeriesDate();
		}
	}



	private String getAcquisitionDate(final int sliceNum) {
		if (mosaic || (sliceNum >= dicoms.size())) {
			return dicoms.get(0).getAcquisitionDate();
		} else {
			return dicoms.get(sliceNum).getAcquisitionDate();
		}
	}



	private String getContentDate(final int sliceNum) {
		if (mosaic || (sliceNum >= dicoms.size())) {
			return dicoms.get(0).getContentDate();
		} else {
			return dicoms.get(sliceNum).getContentDate();
		}
	}



	private Date formatDate(final String dateStr) {
		Date studyDate = null;
		if (StringUtils.isNotBlank(dateStr)) {
			try {
				studyDate = DATE_FORMAT.parse(dateStr);
			} catch (final ParseException ex) {
				try {
					studyDate = DATE_FORMAT2.parse(dateStr);
				} catch (final ParseException e) {
					AppLogger.warn(ex);
				}
			}
		}

		return studyDate;
	}



	private Date getMostRecentDate(final int sliceNum) {
		final List<Date> dates = new ArrayList<>();

		Date date = formatDate(getStudyDate(sliceNum));
		if (date != null) {
			dates.add(date);
		}

		date = formatDate(getSeriesDate(sliceNum));
		if (date != null) {
			dates.add(date);
		}

		date = formatDate(getAcquisitionDate(sliceNum));
		if (date != null) {
			dates.add(date);
		}

		date = formatDate(getContentDate(sliceNum));
		if (date != null) {
			dates.add(date);
		}

		if (dates.size() == 0) {
			dates.add(new Date());
		}

		Collections.sort(dates);

		return dates.get(dates.size() - 1);
	}



	private ByteBuffer[] readSeries() {
		ByteBuffer[] buffers = null;
		final ImageDimensions id = getImageDimensions();

		final int numVoxelsInSlice = id.getNumVoxelsSlice();
		final int numVoxelsVolume = id.getNumVoxelsVolume();
		final int numTimepoints = id.getTimepoints();
		final int numSlices = id.getSlices();

		buffers = new ByteBuffer[numTimepoints];

		BufferedInputStream input = null;

		try {
			final DICOM dicom = dicoms.get(0);
			final URI file = dicom.getImageFile();
			final boolean needsDicomOverlay = dicom.isNeedsOverlayData();
			final boolean fileIsReadable = (FileUtilities.exists(file) && FileUtilities.canRead(file));

			final ImageType it = new ImageType(dicom.getBitsAllocated() / 8, dicom.getDataType(), dicom.getBitsStored(), dicom.isLittleEndian());
			final int sliceSize = numVoxelsInSlice * it.getNumBytesPerVoxel();
			final int numBytes = it.getNumBytesPerVoxel();
			final int numType = it.getByteType();
			final long mask = it.createBitMask();
			final long maskInv = ~mask;

			final ImageType itSeries = getImageType();
			final boolean rgbBySample = itSeries.isRGBBySample();
			final boolean isRGBMode = itSeries.isRGBMode();

			final byte[] buf = new byte[sliceSize];
			final ByteBuffer byteBuffer = ByteBuffer.wrap(buf);

			if (it.isLittleEndian()) {
				byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
			} else {
				byteBuffer.order(ByteOrder.BIG_ENDIAN);
			}

			final List<Integer> temporalPositions = getTemporalPositions();

			if (fileIsReadable) {
				input = FileUtilities.getInputStream(file, false);

				// skip bytes
				final long bytesToSkip = dicom.getImageOffset();
				long bytesSkipped = input.skip(bytesToSkip);
				long totalBytesSkipped = bytesSkipped;

				while (totalBytesSkipped < bytesToSkip) {
					bytesSkipped = input.skip(bytesToSkip - totalBytesSkipped);
					totalBytesSkipped += bytesSkipped;
				}

				for (int ctrT = 0; ctrT < numTimepoints; ctrT++) {
					final ByteBuffer workBuffer = makeBuffer(numVoxelsVolume * 4);
					buffers[ctrT] = workBuffer;
				}

				for (int ctrT = 0; ctrT < numTimepoints; ctrT++) {
					for (int ctrS = 0; ctrS < numSlices; ctrS++) {
						final int temporalPos = (temporalPositions.size() > 0) ? temporalPositions.get((ctrT * numSlices) + ctrS) - 1 : ctrT;
						final ByteBuffer workBuffer = buffers[temporalPos];

						int bytesRead = input.read(buf);
						int totalBytesRead = bytesRead;

						while (totalBytesRead < sliceSize) {
							bytesRead = input.read(buf, totalBytesRead, sliceSize - totalBytesRead);
							totalBytesRead += bytesRead;
						}

						if ((numBytes == 1) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
							if (needsDicomOverlay) {
								for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
									final byte value = byteBuffer.get(ctrV * numBytes);
									workBuffer.putFloat(value & mask);
									dicom.addInPlaneOverlayData((short) ((value & maskInv)), ctrV);
								}
							} else {
								for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
									workBuffer.putFloat(byteBuffer.get(ctrV * numBytes) & mask);
								}
							}
						} else if ((numBytes == 2) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
							if (needsDicomOverlay) {
								for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
									final short value = byteBuffer.getShort(ctrV * numBytes);
									workBuffer.putFloat(value & mask);
									dicom.addInPlaneOverlayData((short) ((value & maskInv)), ctrV);
								}
							} else {
								for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
									workBuffer.putFloat(byteBuffer.getShort(ctrV * numBytes) & mask);
								}
							}
						} else if (numType == ImageType.BYTE_TYPE_RGB) {
							if (rgbBySample) {
								final int[] tempSlice = new int[numVoxelsInSlice];

								if (isRGBMode) {
									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										final int red = 0xFF & byteBuffer.get(ctrV);
										tempSlice[ctrV] |= (red << 16);
									}

									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										final int green = 0xFF & byteBuffer.get(ctrV + numVoxelsInSlice);
										tempSlice[ctrV] |= (green << 8);
									}

									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										final int blue = 0xFF & byteBuffer.get(ctrV + (2 * numVoxelsInSlice));
										tempSlice[ctrV] |= blue;
									}

									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										workBuffer.putInt(tempSlice[ctrV]);
									}
								} else {
									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										final int red = 0xFF & byteBuffer.get(ctrV);
										tempSlice[ctrV] += red;
									}

									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										final int green = 0xFF & byteBuffer.get(ctrV + numVoxelsInSlice);
										tempSlice[ctrV] += green;
									}

									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										final int blue = 0xFF & byteBuffer.get(ctrV + (2 * numVoxelsInSlice));
										tempSlice[ctrV] += blue;
									}

									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										workBuffer.putFloat(tempSlice[ctrV] / 3.0f);
									}
								}
							} else {
								if (isRGBMode) {
									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										final int a = 0xFF & byteBuffer.get(ctrV * numBytes);
										final int b = 0xFF & byteBuffer.get((ctrV * numBytes) + 1);
										final int c = 0xFF & byteBuffer.get((ctrV * numBytes) + 2);
										workBuffer.putInt((c | (b << 8) | (a << 16)));
									}
								} else {
									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										final int a = 0xFF & byteBuffer.get(ctrV * numBytes);
										final int b = 0xFF & byteBuffer.get((ctrV * numBytes) + 1);
										final int c = 0xFF & byteBuffer.get((ctrV * numBytes) + 2);
										workBuffer.putFloat((a + b + c) / 3.0f);
									}
								}
							}
						}
					}
				}
			}
		} catch (final IOException ex) {
			AppLogger.error(ex);
		} finally {
			try {
				if (input != null) {
					input.close();
				}
			} catch (final IOException ex) {
				AppLogger.warn(ex);
			}

			input = null;
		}

		return buffers;
	}



	private ByteBuffer[] readStandard(final boolean forceSingleIteration) {
		ByteBuffer[] buffers = null;
		final ImageDimensions id = getImageDimensions();

		final boolean readOnly = getImageType().isReadOnly();
		final int numVoxelsInSlice = id.getNumVoxelsSlice();
		final int numVoxelsVolume = id.getNumVoxelsVolume();
		final int numTimepoints = id.getTimepoints();
		int iterations = (id.getSlices() * numTimepoints) / dicoms.size();
		final int numFilesPerVolume = dicoms.size() / numTimepoints;

		if (forceSingleIteration) {
			iterations = 1;
		}

		buffers = new ByteBuffer[numTimepoints];

		for (int ctrT = 0; ctrT < numTimepoints; ctrT++) {
			ByteBuffer workBuffer = null;

			if (forceSingleIteration) {
				workBuffer = makeBuffer(numVoxelsInSlice * 4);
			} else {
				if (numVoxelsVolume * 4 < 0) {  // overflow
					workBuffer = makeBuffer(numVoxelsVolume * 2);
				} else {
					workBuffer = makeBuffer(numVoxelsVolume * 4);
				}
			}

			buffers[ctrT] = workBuffer;

			for (int ctrS = 0; ctrS < numFilesPerVolume; ctrS++) {
				final int multiFileIndex = (ctrT * numFilesPerVolume) + ctrS;

				if (multiFileIndex >= dicoms.size()) {
					break;
				}

				BufferedInputStream input = null;

				try {
					final DICOM dicom = dicoms.get(multiFileIndex);
					final URI file = dicom.getImageFile();
					final boolean needsDicomOverlay = dicom.isNeedsOverlayData();
					final boolean fileIsReadable = (FileUtilities.exists(file) && FileUtilities.canRead(file));

					final ImageType it = new ImageType(dicom.getBitsAllocated() / 8, dicom.getDataType(), dicom.getBitsStored(), dicom.isLittleEndian());
					final int sliceSize = numVoxelsInSlice * it.getNumBytesPerVoxel();
					final int numBytes = it.getNumBytesPerVoxel();
					final int numType = it.getByteType();
					final long mask = it.createBitMask();
					final long maskInv = ~mask;

					final ImageType itSeries = getImageType();
					final boolean rgbBySample = itSeries.isRGBBySample();
					final boolean isRGBMode = itSeries.isRGBMode();

					final byte[] buf = new byte[sliceSize];
					final ByteBuffer byteBuffer = ByteBuffer.wrap(buf);

					if (it.isLittleEndian()) {
						byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
					} else {
						byteBuffer.order(ByteOrder.BIG_ENDIAN);
					}

					if (fileIsReadable) {
						input = FileUtilities.getInputStream(file, false);

						// skip bytes
						long sliceBytesToSkip = 0;

						if (forceSingleIteration) {
							sliceBytesToSkip = (id.getSlices() / 2) * sliceSize;
						}

						final long bytesToSkip = dicom.getImageOffset() + sliceBytesToSkip;
						long bytesSkipped = input.skip(bytesToSkip);
						long totalBytesSkipped = bytesSkipped;

						while (totalBytesSkipped < bytesToSkip) {
							bytesSkipped = input.skip(bytesToSkip - totalBytesSkipped);
							totalBytesSkipped += bytesSkipped;
						}

						for (int ctr = 0; ctr < iterations; ctr++) {
							int bytesRead = input.read(buf);
							int totalBytesRead = bytesRead;

							while (totalBytesRead < sliceSize) {
								bytesRead = input.read(buf, totalBytesRead, sliceSize - totalBytesRead);
								totalBytesRead += bytesRead;
							}

							boolean isPalette = false;

							int[] reds = null;
							int[] greens = null;
							int[] blues = null;
							if (isRGBMode && dicom.isPalette()) {
								reds = dicom.getPalleteRed();
								greens = dicom.getPalleteGreen();
								blues = dicom.getPalleteBlue();

								if (CollectionUtilities.isNotEmpty(reds) && CollectionUtilities.isNotEmpty(greens) && CollectionUtilities.isNotEmpty(blues)) {
									isPalette = true;
								}
							}

							if (isPalette) {
								for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
									final int value = byteBuffer.get(ctrV * numBytes) & 0xFF;
									final int red = reds[value];
									final int green = greens[value];
									final int blue = blues[value];
									workBuffer.putFloat(Float.intBitsToFloat(blue | (green << 8) | (red << 16)));
								}
							} else if ((numBytes == 1) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
								if (needsDicomOverlay) {
									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										final byte value = byteBuffer.get(ctrV * numBytes);
										workBuffer.putFloat(value & mask);
										dicom.addInPlaneOverlayData((short) ((value & maskInv)), ctrV);
									}
								} else {
									if (readOnly) {
										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											workBuffer.put((byte)(byteBuffer.get(ctrV * numBytes) & mask));
										}
									} else {
										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											workBuffer.putFloat(byteBuffer.get(ctrV * numBytes) & mask);
										}		
									}
								}
							} else if ((numBytes == 2) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
								if (needsDicomOverlay) {
									for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
										final short value = byteBuffer.getShort(ctrV * numBytes);
										workBuffer.putFloat(value & mask);
										dicom.addInPlaneOverlayData((short) ((value & maskInv)), ctrV);
									}
								} else {
									if (readOnly) {
										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											workBuffer.putShort((short)(byteBuffer.getShort(ctrV * numBytes) & mask));
										}
									} else {
										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											workBuffer.putFloat(byteBuffer.getShort(ctrV * numBytes) & mask);
										}	
									}
								}
							} else if (numType == ImageType.BYTE_TYPE_RGB) {
								if (rgbBySample) {
									final int[] tempSlice = new int[numVoxelsInSlice];

									if (isRGBMode) {
										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											final int red = 0xFF & byteBuffer.get(ctrV);
											tempSlice[ctrV] |= (red << 16);
										}

										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											final int green = 0xFF & byteBuffer.get(ctrV + numVoxelsInSlice);
											tempSlice[ctrV] |= (green << 8);
										}

										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											final int blue = 0xFF & byteBuffer.get(ctrV + (2 * numVoxelsInSlice));
											tempSlice[ctrV] |= blue;
										}

										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											workBuffer.putInt(tempSlice[ctrV]);
										}
									} else {
										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											final int red = 0xFF & byteBuffer.get(ctrV);
											tempSlice[ctrV] += red;
										}

										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											final int green = 0xFF & byteBuffer.get(ctrV + numVoxelsInSlice);
											tempSlice[ctrV] += green;
										}

										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											final int blue = 0xFF & byteBuffer.get(ctrV + (2 * numVoxelsInSlice));
											tempSlice[ctrV] += blue;
										}

										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											workBuffer.putFloat(tempSlice[ctrV] / 3.0f);
										}
									}
								} else {
									if (isRGBMode) {
										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											final int a = 0xFF & byteBuffer.get(ctrV * numBytes);
											final int b = 0xFF & byteBuffer.get((ctrV * numBytes) + 1);
											final int c = 0xFF & byteBuffer.get((ctrV * numBytes) + 2);
											workBuffer.putInt((c | (b << 8) | (a << 16)));
										}
									} else {
										for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
											final int a = 0xFF & byteBuffer.get(ctrV * numBytes);
											final int b = 0xFF & byteBuffer.get((ctrV * numBytes) + 1);
											final int c = 0xFF & byteBuffer.get((ctrV * numBytes) + 2);
											workBuffer.putFloat((a + b + c) / 3.0f);
										}
									}
								}
							}
						}
					} else {
						for (int ctr = 0; ctr < iterations; ctr++) {
							for (int ctrV = 0; ctrV < numVoxelsInSlice; ctrV++) {
								workBuffer.putFloat(0);
							}
						}
					}
				} catch (final IOException ex) {
					AppLogger.error(ex);
					break;
				} finally {
					try {
						if (input != null) {
							input.close();
						}
					} catch (final IOException ex) {
						AppLogger.warn(ex);
					}

					input = null;
				}
			}
		}

		return buffers;
	}



	private void readEncapsulatedData() throws VolumeIOException {
		try {
			if (isCompressed()) {
				for (int ctr = 0; ctr < dicoms.size(); ctr++) {
					final Parser io = new Parser(false);
					CompressedPart part = null;

					final DICOM dicom = dicoms.get(ctr);
					final URI uri = dicom.getImageFile();

					final List<Item> items = io.parseEncapsulatedFromOffset(uri, dicom.getImageOffset());
					for (final Item item : items) {
						if ((item.getGroup() == DICOM.TAG_SUBLIST_ITEM[0]) && (item.getElement() == DICOM.TAG_SUBLIST_ITEM[1])
								&& (item.getValueLengthEncapsulated() > 0)) {
							final int valueLength = (int) item.getValueLengthEncapsulated();
							int valueOffset = (int) (item.getOffsetEnd() - valueLength);

							if (isRLE() || CompressionUtilities.containsJpegTag(uri, valueOffset) || CompressionUtilities.containsJpeg2000Tag(uri, valueOffset)) {
								part = new CompressedPart(uri);
								addCompressedPart(part);
							} else if (isJPEG2000()) {
								final int foundOffset = (int) CompressionUtilities.findJpeg2000Tag(uri, valueOffset, 1024);
								if (foundOffset != -1) {
									valueOffset += foundOffset;
									if (CompressionUtilities.containsJpeg2000Tag(uri, valueOffset)) {
										part = new CompressedPart(uri);
										addCompressedPart(part);
									}
								}
							}

							if (part != null) {
								part.addPart(valueOffset, valueLength);
							}
						}
					}
				}
			}
		} catch (final Exception ex) {
			throw new VolumeIOException(ex);
		}
	}



	private ByteBuffer[] readJpeg2000() {
		ByteBuffer[] buffers = null;

		final ImageType it = getImageType();
		final ImageDimensions id = getImageDimensions();

		final List<CompressedPart> jpegs = getCompressedParts();
		final int numTimepoints = id.getTimepoints();
		final int numJpegsPerSeriesPoint = jpegs.size() / numTimepoints;
		final long mask = it.createBitMask();
		final int numBytes = it.getNumBytesPerVoxel();
		final int numType = it.getByteType();
		final int numVoxelsVolume = id.getNumVoxelsVolume();
		final boolean isRGBMode = it.isRGBMode();

		buffers = new ByteBuffer[numTimepoints];

		for (int ctrT = 0; ctrT < numTimepoints; ctrT++) {
			buffers[ctrT] = makeBuffer(numVoxelsVolume * 4);
			final FloatBuffer workBuffer = buffers[ctrT].asFloatBuffer();

			for (int ctrS = 0; ctrS < numJpegsPerSeriesPoint; ctrS++) {
				final int jpegIndex = (ctrT * numJpegsPerSeriesPoint) + ctrS;

				if (jpegIndex >= jpegs.size()) {
					break;
				}

				try {
					final CompressedPart jpeg = jpegs.get(jpegIndex);
					final URI file = jpeg.getUri();
					final boolean fileIsReadable = (FileUtilities.exists(file) && FileUtilities.canRead(file));

					if (fileIsReadable) {
						final byte[] jpegData = CompressedPart.concatenateCompressedInternalParts(jpeg, file);
						final ByteArrayInputStream input = new ByteArrayInputStream(jpegData);

						final J2000Decoder decoder = new J2000Decoder();
						final ByteBuffer output = decoder.decode(input, it.getByteType() == ImageType.BYTE_TYPE_INTEGER);

						if ((numBytes == 1) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
							final IntBuffer buffer = output.asIntBuffer();
							final int numElements = buffer.capacity();

							for (int ctr = 0; ctr < numElements; ctr++) {
								workBuffer.put(buffer.get(ctr) & mask);
							}
						} else if ((numBytes == 2) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
							final IntBuffer buffer = output.asIntBuffer();
							final int numElements = buffer.capacity();

							for (int ctr = 0; ctr < numElements; ctr++) {
								workBuffer.put(buffer.get(ctr) & mask);
							}
						} else if (numType == ImageType.BYTE_TYPE_RGB) {
							if (isRGBMode) {
								final IntBuffer buffer = output.asIntBuffer();
								final int numElements = buffer.capacity();

								for (int ctr = 0; ctr < numElements; ctr += 3) {
									final int red = buffer.get(ctr);
									final int green = buffer.get(ctr + 1);
									final int blue = buffer.get(ctr + 2);

									workBuffer.put(Float.intBitsToFloat(blue | (green << 8) | (red << 16)));
								}
							} else {
								final IntBuffer buffer = output.asIntBuffer();
								final int numElements = buffer.capacity();

								for (int ctr = 0; ctr < numElements; ctr += 3) {
									final int red = buffer.get(ctr);
									final int green = buffer.get(ctr + 1);
									final int blue = buffer.get(ctr + 2);

									workBuffer.put((float) ((red + green + blue) / 3.0));
								}
							}
						}
					}
				} catch (final IOException ex) {
					AppLogger.error(ex);
				} catch (final ColorSpaceException ex) {
					AppLogger.error(ex);
				} catch (final ICCProfileException ex) {
					AppLogger.error(ex);
				}
			}
		}

		return buffers;
	}



	private ByteBuffer[] readJpegBaseline() {
		ByteBuffer[] buffers = null;

		final ImageType it = getImageType();
		final ImageDimensions id = getImageDimensions();

		final List<CompressedPart> jpegs = getCompressedParts();
		final int numTimepoints = id.getTimepoints();
		final int numJpegsPerSeriesPoint = jpegs.size() / numTimepoints;
		final long mask = it.createBitMask();
		final int numBytes = it.getNumBytesPerVoxel();
		final int numType = it.getByteType();
		final int numVoxelsVolume = id.getNumVoxelsVolume();
		final boolean isRGBMode = it.isRGBMode();

		buffers = new ByteBuffer[numTimepoints];

		for (int ctrT = 0; ctrT < numTimepoints; ctrT++) {
			buffers[ctrT] = makeBuffer(numVoxelsVolume * 4);
			final FloatBuffer workBuffer = buffers[ctrT].asFloatBuffer();

			for (int ctrS = 0; ctrS < numJpegsPerSeriesPoint; ctrS++) {
				final int jpegIndex = (ctrT * numJpegsPerSeriesPoint) + ctrS;

				if (jpegIndex >= jpegs.size()) {
					break;
				}

				try {
					final CompressedPart jpeg = jpegs.get(jpegIndex);
					final URI file = jpeg.getUri();
					final boolean fileIsReadable = (FileUtilities.exists(file) && FileUtilities.canRead(file));

					if (fileIsReadable) {
						// ImageIO
						final byte[] jpegData = CompressedPart.concatenateCompressedInternalParts(jpeg, file);
						final ByteArrayInputStream input = new ByteArrayInputStream(jpegData);
						final Iterator<ImageReader> iterator = ImageIO.getImageReadersByFormatName("JPEG");
						ImageReader iir = null;
						while (iterator.hasNext()) {
							final Object o = iterator.next();
							iir = (ImageReader) o;
							if (iir.canReadRaster()) {
								break;
							}
						}

						ImageIO.setUseCache(false);
						final ImageInputStream iin = ImageIO.createImageInputStream(input);
						iir.setInput(iin, true);

						// this is the actual pixel data
						final Raster ras = iir.readRaster(0, null);
						final DataBuffer dataBuffer = ras.getDataBuffer();
						final int numElements = dataBuffer.getSize();

						if ((numBytes == 1) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
							for (int ctr = 0; ctr < numElements; ctr++) {
								workBuffer.put((byte) dataBuffer.getElem(ctr) & mask);
							}
						} else if ((numBytes == 2) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
							for (int ctr = 0; ctr < numElements; ctr++) {
								workBuffer.put((short) dataBuffer.getElem(ctr) & mask);
							}
						} else if (numType == ImageType.BYTE_TYPE_RGB) {
							if (isRGBMode) {
								for (int ctr = 0; ctr < numElements; ctr += 3) {
									final int red = (dataBuffer.getElem(ctr));
									final int green = (dataBuffer.getElem(ctr + 1));
									final int blue = (dataBuffer.getElem(ctr + 2));

									workBuffer.put(Float.intBitsToFloat(convertYBRToRGB(blue | (green << 8) | (red << 16))));
								}
							} else {
								for (int ctr = 0; ctr < numElements; ctr += 3) {
									final int red = (dataBuffer.getElem(ctr) & 0xFF);
									final int green = (dataBuffer.getElem(ctr + 1) & 0xFF);
									final int blue = (dataBuffer.getElem(ctr + 2) & 0xFF);

									workBuffer.put((red + green + blue) / 3.0f);
								}
							}
						}
					}
				} catch (final Exception ex) {
					AppLogger.error(ex);
				}
			}
		}

		return buffers;
	}



	// ImageIO incorrectly interprets DICOM JPEG as RGB when it is actually YBR
	private static int convertYBRToRGB(final int val) {
		final int y = (val >> 16) & 0xFF; // Y
		int b = ((val >> 8) & 0xFF) - 128; // Pb
		int r = (val & 0xFF) - 128; // Pr

		int g = ((y << 8) + (-88 * b) + (-183 * r)) >> 8; //
		b = ((y << 8) + (454 * b)) >> 8;
		r = ((y << 8) + (359 * r)) >> 8;

		if (r > 255) {
			r = 255;
		} else if (r < 0) {
			r = 0;
		}
		if (g > 255) {
			g = 255;
		} else if (g < 0) {
			g = 0;
		}
		if (b > 255) {
			b = 255;
		} else if (b < 0) {
			b = 0;
		}

		return 0xFF000000 | (((r << 8) | g) << 8) | b;
	}



	private ByteBuffer[] readJpegLossless() {
		ByteBuffer[] buffers = null;

		final ImageType it = getImageType();
		final ImageDimensions id = getImageDimensions();

		final List<CompressedPart> jpegs = getCompressedParts();
		final int numTimepoints = id.getTimepoints();
		final int numJpegsPerSeriesPoint = jpegs.size() / numTimepoints;
		final long mask = it.createBitMask();
		final long maskInv = ~mask;
		final int numBytes = it.getNumBytesPerVoxel();
		final int numType = it.getByteType();
		final int numVoxelsVolume = id.getNumVoxelsVolume();
		final boolean isRGBMode = it.isRGBMode();

		buffers = new ByteBuffer[numTimepoints];

		for (int ctrT = 0; ctrT < numTimepoints; ctrT++) {
			buffers[ctrT] = makeBuffer(numVoxelsVolume * 4);
			final FloatBuffer workBuffer = buffers[ctrT].asFloatBuffer();

			for (int ctrS = 0; ctrS < numJpegsPerSeriesPoint; ctrS++) {
				final int jpegIndex = (ctrT * numJpegsPerSeriesPoint) + ctrS;

				if (jpegIndex >= jpegs.size()) {
					break;
				}

				try {
					final CompressedPart jpeg = jpegs.get(jpegIndex);
					final URI file = jpeg.getUri();
					final boolean fileIsReadable = (FileUtilities.exists(file) && FileUtilities.canRead(file));
					final DICOM dicom = getDicom(file);
					final boolean needsDicomOverlay = dicom.isNeedsOverlayData();

					if (fileIsReadable) {
						final JPEGLosslessDecoder decoder = new JPEGLosslessDecoder(CompressedPart.concatenateCompressedInternalParts(jpeg, file));

						if ((numBytes == 1) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
							final int[] decompressedData = decoder.decode()[0];

							if (needsDicomOverlay) {
								for (int ctr = 0; ctr < decompressedData.length; ctr++) {
									workBuffer.put(((byte) decompressedData[ctr] & mask));
									dicom.addInPlaneOverlayData((short) ((decompressedData[ctr] & maskInv)), ctr);
								}
							} else {
								for (final int element : decompressedData) {
									workBuffer.put(((byte) element & mask));
								}
							}
						} else if ((numBytes == 2) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
							final int[] decompressedData = decoder.decode()[0];

							int max = Integer.MIN_VALUE;
							int min = Integer.MAX_VALUE;

							for (final int val : decompressedData) {
								if (val > max) {
									max = val;
								}

								if (val < min) {
									min = val;
								}
							}

							if (needsDicomOverlay) {
								for (int ctr = 0; ctr < decompressedData.length; ctr++) {
									workBuffer.put(((short) decompressedData[ctr] & mask));
									dicom.addInPlaneOverlayData((short) ((decompressedData[ctr] & maskInv)), ctr);
								}
							} else {
								for (final int element : decompressedData) {
									workBuffer.put(((short) element & mask));
								}
							}
						} else if (numType == ImageType.BYTE_TYPE_RGB) {
							final int[][] decompressedData = decoder.decode();
							final int[] redData = decompressedData[0];
							final int[] greenData = decompressedData[1];
							final int[] blueData = decompressedData[2];

							if (isRGBMode) {
								for (int ctr = 0; ctr < redData.length; ctr++) {
									final int red = redData[ctr];
									final int green = greenData[ctr];
									final int blue = blueData[ctr];

									workBuffer.put(Float.intBitsToFloat(blue | (green << 8) | (red << 16)));
								}
							} else {
								for (int ctr = 0; ctr < redData.length; ctr++) {
									final int red = redData[ctr];
									final int green = greenData[ctr];
									final int blue = blueData[ctr];

									workBuffer.put((red + green + blue) / 3.0f);
								}
							}
						}
					}
				} catch (final Exception ex) {
					AppLogger.error(ex);
				}
			}
		}

		return buffers;
	}



	private ByteBuffer[] readMosaic() {
		ByteBuffer[] buffers = null;

		final ImageType it = getImageType();
		final ImageDimensions id = getImageDimensions();

		final int numVoxelsVolume = id.getCols() * id.getRows() * (getMosaicCols() * getMosaicRows());
		final int mosaicVolSize = numVoxelsVolume * it.getNumBytesPerVoxel();
		final int numTimepoints = id.getTimepoints();
		final int numBytes = it.getNumBytesPerVoxel();
		final int numType = it.getByteType();
		final long mask = it.createBitMask();
		final int numSlices = id.getSlices();
		final int numRows = id.getRows();
		final int numCols = id.getCols();

		final int mosaicWidth = getMosaicWidth();
		final int mosaicHeight = getMosaicHeight();
		final int mosaicRows = getMosaicRows();
		final int mosaicCols = getMosaicCols();
		final int mosaicRowHeight = mosaicHeight / mosaicRows;
		final int mosaicColWidth = mosaicWidth / mosaicCols;

		final byte[] buf = new byte[mosaicVolSize];
		final ByteBuffer byteBuffer = ByteBuffer.wrap(buf);
		if (it.isLittleEndian()) {
			byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
		} else {
			byteBuffer.order(ByteOrder.BIG_ENDIAN);
		}

		buffers = new ByteBuffer[numTimepoints];

		for (int ctrT = 0; ctrT < numTimepoints; ctrT++) {
			final int multiFileIndex = ctrT;
			if (multiFileIndex >= dicoms.size()) {
				break;
			}

			final ByteBuffer workBuffer = makeBuffer(numVoxelsVolume * 4);
			buffers[ctrT] = workBuffer;

			BufferedInputStream input = null;

			try {
				final DICOM dicom = dicoms.get(multiFileIndex);
				final URI file = dicom.getImageFile();
				final boolean fileIsReadable = (FileUtilities.exists(file) && FileUtilities.canRead(file));

				if (fileIsReadable) {
					input = FileUtilities.getInputStream(file, false);
					// skip bytes
					final long bytesToSkip = dicom.getImageOffset();
					long bytesSkipped = input.skip(bytesToSkip);
					long totalBytesSkipped = bytesSkipped;
					while (totalBytesSkipped < bytesToSkip) {
						bytesSkipped = input.skip(bytesToSkip - totalBytesSkipped);
						totalBytesSkipped += bytesSkipped;
					}

					FileUtilities.readFully(input, buf);

					if ((numBytes == 1) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
						for (int ctrS = 0; ctrS < numSlices; ctrS++) {
							for (int ctrR = 0; ctrR < numRows; ctrR++) {
								for (int ctrC = 0; ctrC < numCols; ctrC++) {
									workBuffer.putFloat(byteBuffer.get(getMosaicOffset(mosaicCols, mosaicColWidth, mosaicRowHeight, mosaicWidth, ctrC, ctrR,
											ctrS) * numBytes)
											& mask);
								}
							}
						}
					} else if ((numBytes == 2) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
						for (int ctrS = 0; ctrS < numSlices; ctrS++) {
							for (int ctrR = 0; ctrR < numRows; ctrR++) {
								for (int ctrC = 0; ctrC < numCols; ctrC++) {
									workBuffer.putFloat(byteBuffer.getShort(getMosaicOffset(mosaicCols, mosaicColWidth, mosaicRowHeight, mosaicWidth, ctrC,
											ctrR, ctrS) * numBytes)
											& mask);
								}
							}
						}
					} else if ((numBytes == 4) && ((numType == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) || (numType == ImageType.BYTE_TYPE_INTEGER))) {
						for (int ctrS = 0; ctrS < numSlices; ctrS++) {
							for (int ctrR = 0; ctrR < numRows; ctrR++) {
								for (int ctrC = 0; ctrC < numCols; ctrC++) {
									workBuffer.putFloat(byteBuffer.getInt(getMosaicOffset(mosaicCols, mosaicColWidth, mosaicRowHeight, mosaicWidth, ctrC, ctrR,
											ctrS) * numBytes)
											& mask);
								}
							}
						}
					} else if ((numBytes == 4) && (numType == ImageType.BYTE_TYPE_FLOAT)) {
						for (int ctrS = 0; ctrS < numSlices; ctrS++) {
							for (int ctrR = 0; ctrR < numRows; ctrR++) {
								for (int ctrC = 0; ctrC < numCols; ctrC++) {
									workBuffer.putFloat(byteBuffer.getFloat(getMosaicOffset(mosaicCols, mosaicColWidth, mosaicRowHeight, mosaicWidth, ctrC,
											ctrR, ctrS) * numBytes));
								}
							}
						}
					}
				} else {
					for (int ctrS = 0; ctrS < numSlices; ctrS++) {
						for (int ctrR = 0; ctrR < numRows; ctrR++) {
							for (int ctrC = 0; ctrC < numCols; ctrC++) {
								workBuffer.putFloat(0);
							}
						}
					}
				}
			} catch (final IOException ex) {
				AppLogger.error(ex);
				break;
			} finally {
				try {
					if (input != null) {
						input.close();
					}
				} catch (final IOException ex) {
					AppLogger.warn(ex);
				}
				input = null;
			}
		}

		return buffers;
	}



	private ByteBuffer[] readRLE() {
		ByteBuffer[] buffers = null;

		final ImageType it = getImageType();
		final ImageDimensions id = getImageDimensions();
		final List<CompressedPart> parts = getCompressedParts();

		parts.remove(0); // always skip first encapsulated section?
		final int numTimepoints = id.getTimepoints();
		final int numJpegsPerSeriesPoint = parts.size() / numTimepoints;
		final long mask = it.createBitMask();
		final int numVoxelsVolume = id.getNumVoxelsVolume();
		final boolean isRGBMode = it.isRGBMode();

		buffers = new ByteBuffer[numTimepoints];

		for (int ctrT = 0; ctrT < numTimepoints; ctrT++) {
			buffers[ctrT] = makeBuffer(numVoxelsVolume * 4);
			final FloatBuffer workBuffer = buffers[ctrT].asFloatBuffer();

			for (int ctrS = 0; ctrS < numJpegsPerSeriesPoint; ctrS++) {
				final int partIndex = (ctrT * numJpegsPerSeriesPoint) + ctrS;

				if (partIndex >= parts.size()) {
					break;
				}

				try {
					final RLEDecoder decoder = new RLEDecoder();
					final int byteSize = (int) (id.getNumVoxelsTimeseries() / parts.size()) * it.getNumBytesPerVoxel();
					final ByteBuffer output = decoder.decode(parts.get(partIndex), it.isLittleEndian(), (int) (id.getNumVoxelsTimeseries() / parts.size()));
					final int elementsRead = byteSize / it.getNumBytesPerVoxel();
					final int numSegments = decoder.getNumSegments();
					final DICOM dicom = getDicom(parts.get(partIndex).getUri());
					int[] reds = null;
					int[] greens = null;
					int[] blues = null;
					boolean isPalette = false;

					if (isRGBMode && dicom.isPalette()) {
						reds = dicom.getPalleteRed();
						greens = dicom.getPalleteGreen();
						blues = dicom.getPalleteBlue();

						if (CollectionUtilities.isNotEmpty(reds) && CollectionUtilities.isNotEmpty(greens) && CollectionUtilities.isNotEmpty(blues)) {
							isPalette = true;
						}
					}

					if (isPalette) {
						if (numSegments == 1) {
							for (int ctr = 0; ctr < elementsRead; ctr++) {
								final int value = (output.get(ctr + (0 * elementsRead))) & 0xFF;

								final int red = reds[value];
								final int green = greens[value];
								final int blue = blues[value];
								workBuffer.put(Float.intBitsToFloat(blue | (green << 8) | (red << 16)));
							}
						} else if (numSegments == 2) {
							for (int ctr = 0; ctr < elementsRead; ctr++) {
								final byte temp1 = (output.get(ctr + (0 * elementsRead)));
								final byte temp2 = (output.get(ctr + (1 * elementsRead)));
								final int value = (((temp1 & 0xFF) << 8) | (temp2 & 0xFF));

								final int red = reds[value];
								final int green = greens[value];
								final int blue = blues[value];
								workBuffer.put(Float.intBitsToFloat(blue | (green << 8) | (red << 16)));
							}
						}
					} else if (numSegments == 1) {
						for (int ctr = 0; ctr < elementsRead; ctr++) {
							final byte value = (output.get(ctr + (0 * elementsRead)));
							workBuffer.put(value & mask);
						}
					} else if (numSegments == 2) {
						for (int ctr = 0; ctr < elementsRead; ctr++) {
							final byte temp1 = (output.get(ctr + (0 * elementsRead)));
							final byte temp2 = (output.get(ctr + (1 * elementsRead)));
							final short value = (short) (((temp1 & 0xFF) << 8) | (temp2 & 0xFF));
							workBuffer.put(value & mask);
						}
					} else if (numSegments == 3) { // rgb
						if (isRGBMode) {
							for (int ctr = 0; ctr < elementsRead; ctr++) {
								final int temp1 = (output.get(ctr + (0 * elementsRead))) & 0xFF;
								final int temp2 = (output.get(ctr + (1 * elementsRead))) & 0xFF;
								final int temp3 = (output.get(ctr + (2 * elementsRead))) & 0xFF;
								workBuffer.put(Float.intBitsToFloat(temp3 | (temp2 << 8) | (temp1 << 16)));
							}
						} else {
							for (int ctr = 0; ctr < elementsRead; ctr++) {
								final int temp1 = (output.get(ctr + (0 * elementsRead))) & 0xFF;
								final int temp2 = (output.get(ctr + (1 * elementsRead))) & 0xFF;
								final int temp3 = (output.get(ctr + (2 * elementsRead))) & 0xFF;
								workBuffer.put((temp1 + temp2 + temp3) / 3.0f);
							}
						}
					} else {
						AppLogger.warn("RLE data with " + numSegments + " segments is not supported!");
					}
				} catch (final Exception ex) {
					AppLogger.error(ex);
				}
			}
		}

		return buffers;
	}



	@Override
	public String getSliceMetadata(final int slice, final int seriesPoint) {
		final int numFilesPerVolume = dicoms.size() / getImageDimensions().getTimepoints();
		final int index = (seriesPoint * numFilesPerVolume) + slice;

		if ((index >= 0) && (index < dicoms.size())) {
			return dicoms.get(index).toString();
		}

		return toString();
	}



	@Override
	public short[] getSliceAnnotation(final int slice, final int seriesPoint) {
		final int numFilesPerVolume = dicoms.size() / getImageDimensions().getTimepoints();
		final int index = (seriesPoint * numFilesPerVolume) + slice;

		if ((index >= 0) && (index < dicoms.size())) {
			return dicoms.get(index).getOverlayMask();
		}

		return CollectionUtilities.EMPTY_SHORT_ARRAY;
	}



	@Override
	public Map<String, List<String>> getEditableFieldValues() {
		final Map<String, List<String>> map = new LinkedHashMap<String, List<String>>();

		Collection<Item> items = getDicom(0).getAllItems();

		for (int ctr = 1; ctr < dicoms.size(); ctr++) {
			final Collection<Item> items2 = getDicom(ctr).getAllItems();
			items = CollectionUtils.intersection(items, items2);
		}

		final List<Item> common = new ArrayList<Item>(items);
		Collections.sort(common);
		addAllItems(map, common, 0, false);

		return map;
	}



	private void addAllItems(final Map<String, List<String>> map, final List<Item> items, final int indent, final boolean anon) {
		for (final Item item : items) {
			if (!item.isPrivateData() && !item.isPixelData() && !item.isRawData() && !item.isGroupLength()) {
				if (item.isSublist()) {
					addAllItems(map, item.getSublist(), indent + 1, anon);
				} else {
					addItem(map, item, indent, anon);
				}
			}

			Item next = item.getNext();
			while (next != null) {
				addItem(map, next, indent, anon);
				next = next.getNext();
			}
		}
	}



	private void addItem(final Map<String, List<String>> map, final Item item, final int indent, final boolean anon) {
		String id = "";

		final StringBuffer indentation = new StringBuffer();
		for (int ctr = 0; ctr < indent; ctr++) {
			indentation.append("  ");
		}

		if (indent > 0) {
			id = (" [@" + item.getOffsetStart() + "]");
		}

		final String value = item.getValueAsString();
		if (value != null) {
			final String[] values = value.split("\\\\");

			if (anon) {
				final Tag tag = Dictionary.getTag(item.getGroup(), item.getElement());
				if (ANONYMIZABLE_TAGS.contains(tag)) {
					map.put(indentation + item.getHexTag() + " " + Dictionary.getDescription(item.getGroup(), item.getElement()) + id, Arrays.asList(values));
				}
			} else {
				map.put(indentation + item.getHexTag() + " " + Dictionary.getDescription(item.getGroup(), item.getElement()) + id, Arrays.asList(values));
			}
		}
	}



	@Override
	public Map<String, List<CodeOption>> getHumanReadableFieldOptions() {
		return null;
	}



	@Override
	public boolean isTextEditableHeader() {
		return false;
	}



	@Override
	public void setEditableFieldValue(final String name, final String value, final int index) {
		int startIndex = name.indexOf('(') + 1;
		int endIndex = name.indexOf(',', startIndex);
		final String groupStr = name.substring(startIndex, endIndex);
		final int group = Integer.parseInt(groupStr, 16);

		startIndex = endIndex + 1;
		endIndex = name.indexOf(')', startIndex);
		final String elemStr = name.substring(startIndex, endIndex);
		final int element = Integer.parseInt(elemStr, 16);

		int offset = -1;
		startIndex = name.indexOf("[@");
		if (startIndex != -1) {
			startIndex += 2;
			endIndex = name.indexOf(']', startIndex);
			final String offsetStr = name.substring(startIndex, endIndex);
			offset = Integer.parseInt(offsetStr);
		}

		for (final DICOM dicom : dicoms) {
			final Item item = dicom.getItem(group, element, offset);
			if (item != null) {
				item.setValue(value, index);
			}
		}
	}



	@Override
	public void writeHeader() {
		OutputStream output = null;
		InputStream input = null;

		for (final DICOM dicom : dicoms) {
			try {
				final URI uri = dicom.getImageFile();
				final File file = new File(uri);
				final byte[] header = ExportableSOPClass.makeHeader(dicom.getAllItems(), getImageDimensions(), getImageType(), dicom.getTransferSyntaxID(),
						dicoms.size());
				final int offset = dicom.getImageOffset();
				final File outputFile = File.createTempFile(FileUtilities.removeExtension(file.getName()), "." + FileUtilities.getFullExtension(file),
						file.getParentFile());

				output = new BufferedOutputStream(new FileOutputStream(outputFile));
				output.write(header);

				final byte[] buffer = new byte[BUFFER_SIZE];
				int bytesRead = 0;
				input = FileUtilities.getInputStream(uri, false);

				final long bytesToSkip = offset;
				long bytesSkipped = input.skip(bytesToSkip);
				long totalBytesSkipped = bytesSkipped;

				while (totalBytesSkipped < bytesToSkip) {
					bytesSkipped = input.skip(bytesToSkip - totalBytesSkipped);
					totalBytesSkipped += bytesSkipped;
				}

				while ((bytesRead = input.read(buffer, 0, buffer.length)) > 0) {
					output.write(buffer, 0, bytesRead);
				}

				FileUtilities.renameTo(outputFile, file);
			} catch (final UnsupportedEncodingException ex) {
				AppLogger.error(ex);
			} catch (final IOException ex) {
				AppLogger.error(ex);
			} finally {
				if (output != null) {
					try {
						output.close();
					} catch (final IOException ex) {
						AppLogger.error(ex);
					}
				}

				if (input != null) {
					try {
						input.close();
					} catch (final IOException ex) {
						AppLogger.error(ex);
					}
				}
			}
		}
	}



	@Override
	public List<String> getAnonymizableFields() {
		final Map<String, List<String>> map = new LinkedHashMap<String, List<String>>();
		Collection<Item> items = getDicom(0).getAllItems();

		for (int ctr = 1; ctr < dicoms.size(); ctr++) {
			final Collection<Item> items2 = getDicom(ctr).getAllItems();
			items = CollectionUtils.intersection(items, items2);
		}

		final List<Item> common = new ArrayList<Item>(items);
		Collections.sort(common);
		addAllItems(map, common, 0, true);

		final List<String> list = new ArrayList<String>();
		list.addAll(map.keySet());
		Collections.sort(list);
		return list;
	}



	@Override
	public Map<String, List<String>> anonymize() {
		for (final Tag tag : ANONYMIZABLE_TAGS) {
			for (final DICOM dicom : dicoms) {
				final Item item = dicom.getItem(tag.getGroup(), tag.getElement(), -1);
				if (item != null) {
					item.setValue("ANONYMIZED", 0);
				}
			}
		}

		return getEditableFieldValues();
	}



	private static ByteBuffer makeBuffer(final int size) {
		ByteBuffer bb = null;

		try {
			bb = ByteBuffer.allocateDirect(size); // try to allocate a native buffer...
			bb.order(ByteOrder.nativeOrder());
		} catch (final OutOfMemoryError err) {
			AppLogger.error(err);
			bb = ByteBuffer.allocate(size); // ... if not, try in Java
		} catch (Exception ex) {
			ex.printStackTrace();
		}

		return bb;
	}



	public static boolean isSkipFindAssociated() {
		return skipFindAssociated;
	}



	public static void setSkipFindAssociated(final boolean skipFindAssociated) {
		Series.skipFindAssociated = skipFindAssociated;
	}
}
