
package edu.uthscsa.ric.volume.formats.dicom;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import edu.uthscsa.ric.utilities.AppLogger;
import edu.uthscsa.ric.utilities.ByteUtilities;
import edu.uthscsa.ric.utilities.CollectionUtilities;
import edu.uthscsa.ric.utilities.FileUtilities;


public class Parser {

	private boolean explicit;
	private boolean littleEndian;
	private boolean metaFinished;
	private boolean metaFound;
	private long metaFinishedOffset;
	private long fileSize;
	private final boolean quiet;
	private final byte[] buffer = new byte[4];
	private Item pixelData;

	public static final String MAGIC_NUMBER = "DICM";
	public static final int MAGIC_COOKIE_OFFSET = 128;
	public static final int UNDEFINED_LENGTH = -1;



	public Parser(final boolean quiet) {
		littleEndian = true;
		explicit = true;
		this.quiet = quiet;
		metaFinishedOffset = -1;
	}



	public List<Item> parse(final URI uri) {
		DICOMInputStream input = null;
		final Vector<Item> elements = new Vector<Item>();
		boolean foundPixelData = false;

		try {
			fileSize = FileUtilities.getLength(uri);
			input = new DICOMInputStream(FileUtilities.getInputStream(uri, false));
			long currentOffset = findFirstTagOffset(input);

			if (currentOffset == -1) {
				throw new IOException("No DICOM magic number found!");
			}

			while (currentOffset < fileSize) {
				final Item de = getNextTag(input);

				if (de == null) {
					break;
				} else if (de.isValid()) {
					elements.add(de);
				}

				if (de.isPixelData()) {
					foundPixelData = true;
					break;
				}

				currentOffset = de.getOffsetStart() + de.getElementSize();
			}
		} catch (final Exception ex) {
			if (!quiet) {
				AppLogger.warn("Problem reading DICOM file (" + FileUtilities.getName(uri) + "): " + ex.getMessage());
			}

			// some possible corrupted DICOMs put pixel data inside sublist
			if (!foundPixelData && (pixelData != null)) {
				elements.add(pixelData);
			}
		} finally {
			try {
				if (input != null) {
					input.close();
				}
			} catch (final IOException ex) {
				AppLogger.warn(ex);
			}
		}

		return elements;
	}



	protected List<Item> parseEncapsulatedFromOffset(final URI uri, final int offset) throws IOException {
		DICOMInputStream input = null;
		List<Item> elements = null;

		fileSize = FileUtilities.getLength(uri);

		if (fileSize > Integer.MAX_VALUE) {
			return null;
		}

		try {
			input = new DICOMInputStream(FileUtilities.getInputStream(uri, false));
			FileUtilities.skipFully(input, offset);
			elements = parseEncapsulated(input);
		} finally {
			try {
				if (input != null) {
					input.close();
				}
			} catch (final IOException ex) {
				AppLogger.warn(ex);
			}
		}

		return elements;
	}



	private List<Item> parseEncapsulated(final DICOMInputStream input) throws IOException {
		final List<Item> tags = new ArrayList<Item>();
		Item tag = this.getNextTag(input, true);

		while ((tag != null) && !tag.isSequenceDelim()) {
			if (tag.isSublistItem()) {
				tags.add(tag);
			}

			tag = this.getNextTag(input, true);
		}

		return tags;
	}



	private long findFirstTagOffset(final DICOMInputStream input) throws IOException {
		input.mark(1000);

		// check at the beginning of the file
		final byte[] magic = new byte[4];
		FileUtilities.readFully(input, magic);

		String magicStr = new String(magic);
		if (magicStr.equals(MAGIC_NUMBER)) {
			return 4;
		}

		// there should be a 4 byte "DICM" at byte 128-131
		FileUtilities.skipFully(input, MAGIC_COOKIE_OFFSET - 4);
		FileUtilities.readFully(input, magic);

		magicStr = new String(magic);
		if (magicStr.equals(MAGIC_NUMBER)) {
			return MAGIC_COOKIE_OFFSET + 4;
		}

		if (!input.markSupported()) {
			return -1;
		} else {
			input.reset();
		}

		// last shot, just check if a valid tag exists
		final short tagA1 = getShort(input, true);
		final short tagA2 = getShort(input, true);

		if ((tagA1 > 0) && (tagA1 <= 20) && (tagA2 >= 0) && (tagA2 <= 20)) {
			input.reset();
			return 0;
		}

		final short tagB1 = ByteUtilities.swap(tagA1);
		final short tagB2 = ByteUtilities.swap(tagA2);

		if ((tagB1 > 0) && (tagB1 <= 20) && (tagB2 >= 0) && (tagB2 <= 20)) {
			input.reset();
			return 0;
		}

		return -1;
	}



	private int getInt(final DICOMInputStream input, final boolean little) throws IOException {
		FileUtilities.readFully(input, buffer, 0, 4);

		if (little) {
			return ByteUtilities.swapInt(buffer, 0);
		} else {
			return ByteUtilities.getInt(buffer, 0);
		}
	}



	private Item getNextTag(final DICOMInputStream input) throws IOException {
		return getNextTag(input, false);
	}



	private Item getNextTag(final DICOMInputStream input, final boolean skipValue) throws IOException {
		boolean little;
		int group;
		final int element;
		final long offsetStart = input.getPosition();
		long offset = offsetStart;

		if (offset >= fileSize) {
			return null;
		}

		if (metaFinished) {
			little = littleEndian;
			group = getShort(input, little);
		} else {
			group = getShort(input, true);

			if (((metaFinishedOffset != -1) && (offset >= metaFinishedOffset)) || (group != 0x0002)) {
				metaFinished = true;
				little = littleEndian;

				if (!little) {
					group = ByteUtilities.swap((short) (group & 0xFFFF));
				}
			} else {
				little = true;
			}
		}

		offset += 2;

		if (!metaFound && (group == 0x0002)) {
			metaFound = true;
		}

		element = getShort(input, little);
		offset += 2;

		String vr = null;
		long length = 0;

		if (explicit || !metaFinished) {
			vr = getVR(input);

			if (!metaFound && metaFinished && (Dictionary.VRS_ALL.indexOf(vr) == -1)) {
				if (!input.markSupported()) {
					throw new IOException("Missing transfer syntax!");
				} else {
					input.reset();
				}

				vr = Dictionary.getVR(group, element);
				length = getInt(input, little);
				explicit = false;
				offset += 4;
			} else {
				offset += 2;

				if (Dictionary.VRS_DATA.indexOf(vr) != -1) {
					FileUtilities.skipFully(input, 2);
					offset += 2;

					length = getInt(input, little);
					offset += 4;
				} else {
					length = getShort(input, little);
					offset += 2;
				}
			}
		} else {
			vr = Dictionary.getVR(group, element);
			length = getInt(input, little);

			if (length == UNDEFINED_LENGTH) {
				vr = "SQ";
			}

			offset += 4;
		}

		final long offsetValue = offset;
		byte[] value = null;
		List<Item> sublist = null;

		if ("SQ".equals(vr)) {
			sublist = this.parseSublist(input, length);

			if (length == UNDEFINED_LENGTH) {
				length = sublist.get(sublist.size() - 1).getOffsetEnd() - offset;
				sublist.remove(sublist.size() - 1); // remove sequence delimeter
			}
		} else if (length != 0) {
			final boolean isPixelData = ((group == DICOM.TAG_PIXEL_DATA[0]) && (element == DICOM.TAG_PIXEL_DATA[1]));

			if (length <= 0) {
				if (isPixelData) {
					length = (int) (fileSize - offset);
				}
			}

			if (skipValue || isPixelData) {
				FileUtilities.skipFully(input, (int) length);
			} else {
				value = new byte[(int) length];
				FileUtilities.readFully(input, value);
			}
		} else {
			value = CollectionUtilities.EMPTY_BYTE_ARRAY;
		}

		offset += length;
		Item tag = null;

		if (value != null) {
			tag = new Item(group, element, vr, offsetStart, offsetValue, offset, littleEndian, value);
		} else {
			tag = new Item(group, element, vr, offsetStart, offsetValue, offset, littleEndian, sublist);
		}

		if (tag.isPixelData()) {
			pixelData = tag;
		} else if (tag.isPrivateData()) {
			tag.readPrivateData();
		}

		tag.setValueLength(length);

		if (skipValue) {
			tag.setValueLengthEncapsulated(length);
		}

		if (tag.isTransferSyntax()) {
			if (tag.getValueAsString().equals(Constants.TRANSFER_SYNTAX_ID_IMPLICIT)) {
				explicit = false;
				littleEndian = true;
			} else if (tag.getValueAsString().equals(Constants.TRANSFER_SYNTAX_ID_EXPLICIT_BIG)) {
				explicit = true;
				littleEndian = false;
			} else {
				explicit = true;
				littleEndian = true;
			}
		} else if (tag.isMetaLength()) {
			metaFinishedOffset = tag.getValueAsInteger() + offset;
		}

		return tag;
	}



	private short getShort(final DICOMInputStream input, final boolean little) throws IOException {
		FileUtilities.readFully(input, buffer, 0, 2);

		if (little) {
			return ByteUtilities.swapShort(buffer, 0);
		} else {
			return ByteUtilities.getShort(buffer, 0);
		}
	}



	private String getVR(final DICOMInputStream input) throws IOException {
		input.mark(1000);
		FileUtilities.readFully(input, buffer, 0, 2);
		return new String(buffer, 0, 2);
	}



	private List<Item> parseSublist(final DICOMInputStream input, final long length) throws IOException {
		Item sublistItem = null;
		long offset = input.getPosition();
		final long offsetEnd = offset + length;
		final List<Item> tags = new ArrayList<Item>();

		if (length == UNDEFINED_LENGTH) {
			sublistItem = this.parseSublistItem(input);

			while (!sublistItem.isSequenceDelim()) {
				tags.addAll(sublistItem.getSublist());
				offset = sublistItem.getOffsetEnd();
				sublistItem = this.parseSublistItem(input);
			}

			tags.add(sublistItem);
		} else {
			while (offset < offsetEnd) {
				sublistItem = this.parseSublistItem(input);
				tags.addAll(sublistItem.getSublist());
				offset = sublistItem.getOffsetEnd();
			}
		}

		return tags;
	}



	private Item parseSublistItem(final DICOMInputStream input) throws IOException {
		int group, element, length;
		long offsetEnd;
		Item tag = null;
		Item sublistItemTag;
		final List<Item> tags = new ArrayList<Item>();
		final long offsetStart = input.getPosition();
		long offset = offsetStart;
		long offsetValue;

		group = getShort(input, littleEndian);
		offset += 2;

		element = getShort(input, littleEndian);
		offset += 2;

		length = getInt(input, littleEndian);
		offset += 4;

		offsetValue = offset;

		if (length == UNDEFINED_LENGTH) {
			Item previousTag = null;
			tag = this.getNextTag(input);

			while (!tag.isSublistItemDelim()) {
				if (previousTag != null) {
					previousTag.setNext(tag);
				} else {
					tags.add(tag);
				}

				previousTag = tag;

				offset = tag.getOffsetEnd();
				tag = this.getNextTag(input);
			}

			offset = tag.getOffsetEnd();
		} else {
			offsetEnd = offset + length;
			Item previousTag = null;

			while (offset < offsetEnd) {
				tag = this.getNextTag(input);

				if (previousTag != null) {
					previousTag.setNext(tag);
				} else {
					tags.add(tag);
				}

				previousTag = tag;

				offset = tag.getOffsetEnd();
			}
		}

		sublistItemTag = new Item(group, element, null, offsetStart, offsetValue, offset, littleEndian, tags);

		return sublistItemTag;
	}
}
