
package edu.uthscsa.ric.volume.formats.dicom;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.vecmath.Vector3d;

import edu.uthscsa.ric.utilities.AppLogger;
import edu.uthscsa.ric.utilities.ByteUtilities;
import edu.uthscsa.ric.utilities.CollectionUtilities;
import edu.uthscsa.ric.utilities.FileUtilities;
import edu.uthscsa.ric.volume.Coordinate;
import edu.uthscsa.ric.volume.ImageDescription;
import edu.uthscsa.ric.volume.ImageDimensions;
import edu.uthscsa.ric.volume.ImageRange;
import edu.uthscsa.ric.volume.ImageTransform;
import edu.uthscsa.ric.volume.ImageType;
import edu.uthscsa.ric.volume.VolumeIOException;
import edu.uthscsa.ric.volume.VoxelDimensions;
import edu.uthscsa.ric.volume.WritableHeader;


public class ExportableSOPClassMR extends ExportableSOPClass implements WritableHeader {

	public static final String MODALITY = "MR";
	public static final String NAME = "DICOM (MR)";
	public static final String SOP_CLASS_ENHANCED_MR = "1.2.840.10008.5.1.4.1.1.4.1";



	@Override
	public boolean canCompress() {
		return false;
	}



	@Override
	public File formatImageFile(final File headerFile) {
		return headerFile;
	}



	@Override
	public String getMinimumVersionSupported() {
		return null;
	}



	@Override
	public String getPluginName() {
		return NAME;
	}



	@Override
	public URL getPluginURL() {
		return null;
	}



	@Override
	public String getSOPClass(final ImageType it) {
		return SOP_CLASS_ENHANCED_MR;
	}



	@Override
	public int[] getSupportedBytesForType(final int type) {
		if (type == ImageType.BYTE_TYPE_INTEGER) {
			return null;
		} else if (type == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) {
			return new int[] { 2 };
		} else {
			return null;
		}
	}



	@Override
	public String[] getSupportedOrientations() {
		return CollectionUtilities.EMPTY_STRING_ARRAY;
	}



	@Override
	public int[] getSupportedTypesForBytes(final int bytes) {
		if (bytes == 1) {
			return null;
		} else if (bytes == 2) {
			return new int[] { ImageType.BYTE_TYPE_INTEGER_UNSIGNED };
		} else {
			return null;
		}
	}



	@Override
	public String getVersion() {
		return null;
	}



	@Override
	public boolean hasNewerVersion() {
		return false;
	}



	@Override
	public boolean isSupportedBigEndian() {
		return false;
	}



	@Override
	public boolean isSupportedDataScaleIntercept() {
		return true;
	}



	@Override
	public boolean isSupportedDataScaleSlope() {
		return true;
	}



	@Override
	public boolean isSupportedLittleEndian() {
		return true;
	}



	@Override
	public boolean isSupportedMultipleDataScales() {
		return false;
	}



	@Override
	public boolean isThisFormat(final File file) {
		if (!super.isThisFormat(file)) {
			return false;
		}

		BufferedInputStream fileIn = null;
		final byte[] data = new byte[256];

		try {
			fileIn = FileUtilities.getInputStream(file.toURI(), false);
			FileUtilities.readFully(fileIn, data);

			final byte[] sopClass = SOP_CLASS_ENHANCED_MR.getBytes();

			if (ByteUtilities.indexOf(data, sopClass) != -1) {
				return true;
			}

			return false;
		} catch (final IOException ex) {
			AppLogger.error(ex);
		} finally {
			try {
				if (fileIn != null) {
					fileIn.close();
				}
			} catch (final IOException ex) {
				AppLogger.warn(ex);
			}
		}

		return false;
	}



	@Override
	public boolean willWriteImage() {
		return false;
	}



	@Override
	public byte[] writeHeader(final ImageDimensions id, final VoxelDimensions vd, final ImageType it, final ImageRange ir, final ImageDescription in,
			final boolean compress, final Coordinate origin, final String orientation, final ImageTransform[] transforms, final File headerFile,
			final File imageFile, final ByteBuffer[] imageBuffer, final File tempDir) throws VolumeIOException {
		// create IDs
		final String studyUID = DICOMUtils.createUID();
		final String seriesUID = DICOMUtils.createUID();
		final String frameOfRefUID = DICOMUtils.createUID();
		final String studyId = studyNumber;
		final int seriesId = getNextSeriesNumber();
		final int imageId = getImageNumber();
		final String accessionId = getNextAccessionNumber();
		final String dimOrgUID = DICOMUtils.createUID();
		final Date now = new Date();

		return writeHeader(headerFile, id, vd, it, ir, in, origin, orientation, 0, studyUID, seriesUID, frameOfRefUID, studyId, seriesId, imageId, dimOrgUID,
				accessionId, now);
	}



	@Override
	public byte[][] writeSeriesHeaders(final ImageDimensions id, final VoxelDimensions vd, final ImageType it, final ImageRange ir, final ImageDescription in,
			final boolean compress, final Coordinate origin, final String orientation, final ImageTransform[] transforms, final File headerFile,
			final File imageFile, final ByteBuffer[] imageBuffer, final File tempDir) throws VolumeIOException {
		// create IDs
		final String studyUID = DICOMUtils.createUID();
		final String seriesUID = DICOMUtils.createUID();
		final String frameOfRefUID = DICOMUtils.createUID();
		final String studyId = studyNumber;
		final int seriesId = getNextSeriesNumber();
		int imageId = getImageNumber();
		final String accessionId = getNextAccessionNumber();
		final String dimOrgUID = DICOMUtils.createUID();
		final Date now = new Date();

		final int numSeriesPoints = id.getTimepoints();
		final byte[][] data = new byte[numSeriesPoints][];
		for (int ctr = 0; ctr < numSeriesPoints; ctr++) {
			data[ctr] = writeHeader(headerFile, id, vd, it, ir, in, origin, orientation, ctr, studyUID, seriesUID, frameOfRefUID, studyId, seriesId, imageId,
					dimOrgUID, accessionId, now);
			imageId++;
		}

		return data;
	}



	private byte[] writeHeader(final File headerFile, final ImageDimensions id, final VoxelDimensions vd, final ImageType it, final ImageRange ir,
			final ImageDescription in, final Coordinate origin, final String orientation, final int seriesIndex, final String studyUID, final String seriesUID,
			final String frameOfRefUID, final String studyId, final int seriesId, final int imageId, final String dimOrgUID, final String accessionId,
			final Date now) throws VolumeIOException {
		final Vector<Item> tags = new Vector<Item>();

		final Map<Long, String> mappedDicomTags = in.getDicomItems();

		// Enhanced MR Image Information Object Definition (A.36.2)

		// create IDs
		final String sopInstanceUID = DICOMUtils.createUID();

		try {
			// Meta Header
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_FILE_META_INFO_VERSION, new byte[] { 0, 1 }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_MEDIA_STORED_SOP_CLASS_UID, getSOPClass(it)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_MEDIA_STORED_SOP_INSTANCE_UID, sopInstanceUID));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_TRANSFER_SYNTAX_UID, XFER_SYNTAX_EXPLICIT_VT_LITLE_ENDIAN));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_IMPLEMENTATION_CLASS_UID, IMPLEMENTATION_CLASS_UID + ".1." + VERSION));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_IMPLEMENTATION_VERSION_NAME, IMPLEMENTATION_VERSION_NAME + " " + VERSION));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_GROUP_LENGTH, countSize(tags)));

			// SOP Common Module (C.12.1)
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_SOP_CLASS_UID, getSOPClass(it)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_SOP_INSTANCE_UID, sopInstanceUID));

			// Patient Module (C.7.1.1)
			addTag(tags, createTag(Dictionary.TAG_0010_PATIENT_NAME, mappedDicomTags, makePatientName(in, headerFile)));
			addTag(tags, createTag(Dictionary.TAG_0010_PATIENT_ID, mappedDicomTags, makePatientId(in, headerFile)));
			addTag(tags, createTag(Dictionary.TAG_0010_PATIENT_BIRTH_DATE, mappedDicomTags));
			addTag(tags, createTagType2(Dictionary.TAG_0010_PATIENT_SEX, mappedDicomTags));

			// General Study Module (C.7.2.1)
			addTag(tags, createTag(Dictionary.TAG_0020_STUDY_INSTANCE_ID, mappedDicomTags, studyUID));
			addTag(tags, createTag(Dictionary.TAG_0008_STUDY_DATE, mappedDicomTags, DATE_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0008_STUDY_TIME, mappedDicomTags, TIME_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0008_REFERRING_PHYSICIAN_NAME, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0020_STUDY_ID, mappedDicomTags, studyId));
			addTag(tags, createTag(Dictionary.TAG_0008_ACCESSION_NUMBER, mappedDicomTags, accessionId));
			addTag(tags, createTag(Dictionary.TAG_0008_STUDY_DESCRIPTION, mappedDicomTags));

			// General Series Module (C.7.3.1)
			addTag(tags, createTag(Dictionary.TAG_0008_MODALITY, mappedDicomTags, MODALITY));
			addTag(tags, createTag(Dictionary.TAG_0020_SERIES_INSTANCE_ID, mappedDicomTags, seriesUID));
			addTag(tags, createTag(Dictionary.TAG_0020_SERIES_ID, mappedDicomTags, new int[] { seriesId }));
			addTag(tags, createTag(Dictionary.TAG_0008_SERIES_DESCRIPTION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_PATIENT_POSITION, mappedDicomTags));

			// MR Series Module (C.8.13.6)
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_MODALITY, MODALITY));

			// Frame of Reference Module (C.7.4.1)
			addTag(tags, createExplicitTag(Dictionary.TAG_0020_FRAME_OF_REF_ID, frameOfRefUID));
			addTag(tags, createExplicitTag(Dictionary.TAG_0020_POS_REF_INDICATOR, null));

			// General Equipment Module (C.7.5.1)
			addTag(tags, createTag(Dictionary.TAG_0008_MANUFACTURER, mappedDicomTags));

			// Enhanced General Equipment Module (C.7.5.2)
			addTag(tags, createTag(Dictionary.TAG_0008_MANUFACTURER, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_MANUFACTURER_MODEL_NAME, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_DEVICE_SERIAL_NUMBER, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_SOFTWARE_VERIONS, mappedDicomTags));

			// Image Pixel Module (C.7.6.3)
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SAMPLES_PER_PIXEL, SAMPLES_PER_PIXEL));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PHOTOMETRIC_INTERP, PHOTOMETRIC_INTERP));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_COLUMNS, id.getCols()));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_ROWS, id.getRows()));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_BITS_ALLOCATED, it.getNumBytesPerVoxel() * 8));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_BITS_STORED, it.getNumBytesPerVoxel() * 8));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_HIGH_BIT, (it.getNumBytesPerVoxel() * 8) - 1));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PIXEL_REP, PIXEL_REPRESENTATION));

			// SC Multi-frame Vector Module (C.8.6.4)
			addTag(tags,
					createExplicitTag(Dictionary.TAG_0018_SLICE_LOCATION_VECTOR,
							makeSliceLocationVector(vd.getSliceThickness(true), id.getSlices(), 1, id, vd, orientation, origin)));

			// Multi-frame Functional Groups (C.7.6.16)
			addTag(tags, createTag(Dictionary.TAG_0020_IMAGE_NUMBER, mappedDicomTags, new int[] { imageId }));
			addTag(tags, createTag(Dictionary.TAG_0008_CONTENT_DATE, mappedDicomTags, DATE_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0008_CONTENT_TIME, mappedDicomTags, TIME_FORMAT.format(now)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_NUMBER_OF_FRAMES, new int[] { id.getSlices() }));

			{
				// Shared
				{
					// Pixel Measures Functional Group (C.7.6.16.2.1)
					final Item sliceThickness = createExplicitTag(Dictionary.TAG_0018_SLICE_THICKNESS, new double[] { vd.getSliceThickness(true) });
					final Item pixelSpacing = createExplicitTag(Dictionary.TAG_0028_PIXEL_SPACING, new double[] { vd.getRowSize(true), vd.getColSize(true) });
					sliceThickness.setNext(pixelSpacing);

					final Item pixelMeasuresSequence = createSequenceTag(Dictionary.TAG_0028_PIXEL_MEASURES_SQ, createListFromItem(sliceThickness));

					// Plane Orientation Functional Group (C.7.6.16.2.4)
					final Coordinate lastPixelFirstRow = makeRealWorldCoordinateAt(id.getCols() - 1, 0, 0, orientation, id, vd, origin);
					final Coordinate firstPixelFirstRow = makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin);
					final Vector3d vecRow = new Vector3d(lastPixelFirstRow.xDbl - firstPixelFirstRow.xDbl, lastPixelFirstRow.yDbl - firstPixelFirstRow.yDbl,
							lastPixelFirstRow.zDbl - firstPixelFirstRow.zDbl);
					vecRow.normalize();

					final Coordinate lastPixelFirstCol = makeRealWorldCoordinateAt(0, id.getRows() - 1, 0, orientation, id, vd, origin);
					final Coordinate firstPixelFirstCol = makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin);
					final Vector3d vecCol = new Vector3d(lastPixelFirstCol.xDbl - firstPixelFirstCol.xDbl, lastPixelFirstCol.yDbl - firstPixelFirstCol.yDbl,
							lastPixelFirstCol.zDbl - firstPixelFirstCol.zDbl);
					vecCol.normalize();

					final Item imageOrientation = createExplicitTag(Dictionary.TAG_0020_IMAGE_ORIENTATION, new double[] { vecRow.x, vecRow.y, vecRow.z,
							vecCol.x, vecCol.y, vecCol.z });
					final Item planeOrientationSequence = createSequenceTag(Dictionary.TAG_0020_PLANE_ORIENTATION_SQ, createListFromItem(imageOrientation));

					// Frame Anatomy Functional Group (C.7.6.16.2.8)
					final Item anatomicRegionSequence = makeCodeSequence(Dictionary.TAG_0008_ANATOMIC_REGION_SQ, mappedDicomTags);
					final Item frameLaterality = createTag(Dictionary.TAG_0020_FRAME_LATERALITY, mappedDicomTags);

					anatomicRegionSequence.setNext(frameLaterality);
					final Item frameAnatomySequence = createSequenceTag(Dictionary.TAG_0020_FRAME_ANATOMY_SQ, createListFromItem(anatomicRegionSequence));

					// MR Image Frame Type Group (C.8.13.5.1)
					final Item frameType = createExplicitTag(Dictionary.TAG_0008_FRAME_TYPE, CollectionUtilities.array(FRAME_TYPE));
					final Item pixelPresentation = createExplicitTag(Dictionary.TAG_0008_PIXEL_PRESENTATION, PIXEL_PRESENTATION);
					final Item volumetricProperties = createTag(Dictionary.TAG_0008_VOLUMETRIC_PROPERTIES, mappedDicomTags);
					final Item volBasedCalcTech = createTag(Dictionary.TAG_0008_VOLUME_BASED_CALCULATION_TECH, mappedDicomTags);
					final Item complexImageComp = createTag(Dictionary.TAG_0008_COMPLEX_IMAGE_COMP, mappedDicomTags);
					final Item acquisitionContrast = createTag(Dictionary.TAG_0008_ACQUISITION_CONTRAST, mappedDicomTags);

					frameType.setNext(pixelPresentation);
					pixelPresentation.setNext(volumetricProperties);
					volumetricProperties.setNext(volBasedCalcTech);
					volBasedCalcTech.setNext(complexImageComp);
					complexImageComp.setNext(acquisitionContrast);

					final Item mrImageFrameTypeSeq = createSequenceTag(Dictionary.TAG_0018_MR_IMAGE_FRAME_TYPE_SEQ, createListFromItem(frameType));

					// MR Timing and Related Parameters Group (C.8.13.5.2)
					final Item repetitionTime = createTag(Dictionary.TAG_0018_REPETITION_TIME, mappedDicomTags);
					final Item echoTrainLength = createTag(Dictionary.TAG_0018_ECHO_TRAIN_LENGTH, mappedDicomTags);
					final Item flipAngle = createTag(Dictionary.TAG_0018_FLIP_ANGLE, mappedDicomTags);
					final Item rfEchoTrainLength = createTag(Dictionary.TAG_0018_RF_ECHO_TRAIN_LENGTH, mappedDicomTags);
					final Item gradientEchoTrainLength = createTag(Dictionary.TAG_0018_GRADIENT_ECHO_TRAIN_LENGTH, mappedDicomTags);

					final Item specificAbsorptionRateDef = createTag(Dictionary.TAG_0018_SPECIFIC_ABSORPTION_RATE_DEF, mappedDicomTags);
					final Item specificAbsorptionRateValue = createTag(Dictionary.TAG_0018_SPECIFIC_ABSORPTION_RATE_VALUE, mappedDicomTags);
					specificAbsorptionRateDef.setNext(specificAbsorptionRateValue);
					final Item specificAbsorptionRateSeq = createSequenceTag(Dictionary.TAG_0018_SPECIFIC_ABSORPTION_RATE_SEQ,
							createListFromItem(specificAbsorptionRateDef));

					final Item operatingModeType = createTag(Dictionary.TAG_0018_OPERATING_MODE_TYPE, mappedDicomTags);
					final Item operatingMode = createTag(Dictionary.TAG_0018_OPERATING_MODE, mappedDicomTags);
					operatingModeType.setNext(operatingMode);
					final Item operatingModeSeq = createSequenceTag(Dictionary.TAG_0018_OPERATING_MODE_SEQ, createListFromItem(operatingModeType));

					repetitionTime.setNext(echoTrainLength);
					echoTrainLength.setNext(flipAngle);
					flipAngle.setNext(operatingModeSeq);
					operatingModeSeq.setNext(specificAbsorptionRateSeq);
					specificAbsorptionRateSeq.setNext(rfEchoTrainLength);
					rfEchoTrainLength.setNext(gradientEchoTrainLength);

					final Item mrTimingSeq = createSequenceTag(Dictionary.TAG_0018_MR_TIMING_SEQ, createListFromItem(repetitionTime));

					// MR Modifier Group (C.8.13.5.5)
					final Item inversionRecovery = createTag(Dictionary.TAG_0018_INVERSION_RECOVERY, mappedDicomTags);
					final Item flowCompensation = createTag(Dictionary.TAG_0018_FLOW_COMPENSATION, mappedDicomTags);
					final Item t2Preparation = createTag(Dictionary.TAG_0018_T2_PREPARATION, mappedDicomTags);
					final Item spectrallySelectedExcitation = createTag(Dictionary.TAG_0018_SPECTRALLY_SELECTED_EXCITATION, mappedDicomTags);
					final Item spatialPresaturation = createTag(Dictionary.TAG_0018_SPATIAL_PRESATURATION, mappedDicomTags);
					final Item parallelAcquisition = createTag(Dictionary.TAG_0018_PARALLEL_ACQUSITION, mappedDicomTags);
					final Item partialFourier = createTag(Dictionary.TAG_0018_PARTIAL_FOURIER, mappedDicomTags);

					inversionRecovery.setNext(flowCompensation);
					flowCompensation.setNext(t2Preparation);
					t2Preparation.setNext(spectrallySelectedExcitation);
					spectrallySelectedExcitation.setNext(spatialPresaturation);
					spatialPresaturation.setNext(parallelAcquisition);
					parallelAcquisition.setNext(partialFourier);

					final Item mrModifierSeq = createSequenceTag(Dictionary.TAG_0018_MR_MODIFIER_SEQ, createListFromItem(inversionRecovery));

					// MR Imaging Modifier Group (C.8.13.5.6)
					final Item pixelBandwidth = createTag(Dictionary.TAG_0018_PIXEL_BANDWIDTH, mappedDicomTags);
					final Item magnetizationTransfer = createTag(Dictionary.TAG_0018_MAGNETIZATION_TRANSFER, mappedDicomTags);
					final Item bloodSignalNulling = createTag(Dictionary.TAG_0018_BLOOD_SIGNAL_NULLING, mappedDicomTags);
					final Item tagging = createTag(Dictionary.TAG_0018_TAGGING, mappedDicomTags);
					final Item transmitterFreq = createTag(Dictionary.TAG_0018_TRANSMITTER_FREQ, mappedDicomTags);

					pixelBandwidth.setNext(magnetizationTransfer);
					magnetizationTransfer.setNext(bloodSignalNulling);
					bloodSignalNulling.setNext(tagging);
					tagging.setNext(transmitterFreq);

					final Item mrImagingModiferSeq = createSequenceTag(Dictionary.TAG_0018_MR_IMAGING_MODIFIER_SEQ, createListFromItem(pixelBandwidth));

					// MR Receive Coil Group (C.8.13.5.7)
					final Item receiveCoilName = createTag(Dictionary.TAG_0018_RECEIVE_COIL_NAME, mappedDicomTags);
					final Item receiveCoilManufacturerName = createTag(Dictionary.TAG_0018_RECIEVE_COIL_MANUFACTURER_NAME, mappedDicomTags);
					final Item receiveCoilType = createTag(Dictionary.TAG_0018_RECIEVE_COIL_TYPE, mappedDicomTags);
					final Item quadratureReceiveCoil = createTag(Dictionary.TAG_0018_QUADRATURE_RECEIVE_COIL, mappedDicomTags);

					receiveCoilName.setNext(receiveCoilManufacturerName);
					receiveCoilManufacturerName.setNext(receiveCoilType);
					receiveCoilType.setNext(quadratureReceiveCoil);

					final Item mrReceiveCoilSeq = createSequenceTag(Dictionary.TAG_0018_MR_RECEIVE_COIL_SEQ, createListFromItem(receiveCoilName));

					// MR Diffusion Group (C.8.13.5.9)
					final Item diffusionDirectionality = createTag(Dictionary.TAG_0018_DIFFUSION_DIRECTIONALITY, mappedDicomTags);
					final Item diffusionBValue = createTag(Dictionary.TAG_0018_DIFFUSION_B_VALUE, mappedDicomTags);
					final Item diffusionAnisotropyType = createTag(Dictionary.TAG_0018_DIFFUSION_ANISOTROPY_TYPE, mappedDicomTags);

					diffusionDirectionality.setNext(diffusionBValue);
					diffusionBValue.setNext(diffusionAnisotropyType);

					final Item mrDiffusionSeq = createSequenceTag(Dictionary.TAG_0018_MR_DIFFUSION_SEQ, createListFromItem(diffusionDirectionality));

					// MR Averages Group (C.8.13.5.10)
					final Item numberOfAverages = createTag(Dictionary.TAG_0018_NUM_AVERAGES, mappedDicomTags);

					final Item mrAveragesSeq = createSequenceTag(Dictionary.TAG_0018_MR_AVERAGES_SEQ, createListFromItem(numberOfAverages));

					// Pixel Value Transformation Group (C.7.6.16.2.9)
					final Item rescaleIntercept = createExplicitTag(Dictionary.TAG_0028_RESCALE_INTERCEPT,
							new double[] { ir.getDataScaleIntercepts()[seriesIndex * id.getSlices()] });
					final Item rescaleSlope = createExplicitTag(Dictionary.TAG_0028_RESCALE_SLOPE,
							new double[] { ir.getDataScaleSlopes()[seriesIndex * id.getSlices()] });
					final Item rescaleType = createExplicitTag(Dictionary.TAG_0028_RESCALE_TYPE, RESCALE_TYPE);

					rescaleIntercept.setNext(rescaleSlope);
					rescaleSlope.setNext(rescaleType);

					final Item pixelTransSequence = createSequenceTag(Dictionary.TAG_0028_PIXEL_TRANSFORMATION_SQ, createListFromItem(rescaleIntercept));

					// MR Echo Group (C.8.13.5.4)
					final Item effectiveEchoTime = createTag(Dictionary.TAG_0018_EFFECTIVE_ECHO_TIME, new Tag[] { Dictionary.TAG_0018_ECHO_TIME,
							Dictionary.TAG_0018_EFFECTIVE_ECHO_TIME }, mappedDicomTags);

					final Item mrEchoSeq = createSequenceTag(Dictionary.TAG_0018_MR_ECHO_SEQ, createListFromItem(effectiveEchoTime));

					// MR Transmit Coil Group (C.8.13.5.8)
					final Item transmitCoilName = createTag(Dictionary.TAG_0018_TRASMIT_COIL_NAME, mappedDicomTags);
					final Item transmitCoilManufacturerName = createTag(Dictionary.TAG_0018_TRANSMIT_COIL_MANUFACTURER_NAME, mappedDicomTags);
					final Item transmitCoilType = createTag(Dictionary.TAG_0018_TRANSMIT_COIL_TYPE, mappedDicomTags);

					transmitCoilName.setNext(transmitCoilManufacturerName);
					transmitCoilManufacturerName.setNext(transmitCoilType);

					final Item mrTransmitCoilSeq = createSequenceTag(Dictionary.TAG_0018_MR_TRANSMIT_COIL_SEQUENCE, createListFromItem(transmitCoilName));

					// collect shared macros
					mrImagingModiferSeq.setNext(mrReceiveCoilSeq);
					mrReceiveCoilSeq.setNext(mrTransmitCoilSeq);
					mrTransmitCoilSeq.setNext(mrTimingSeq);
					mrTimingSeq.setNext(mrEchoSeq);
					mrEchoSeq.setNext(mrModifierSeq);
					mrModifierSeq.setNext(mrDiffusionSeq);
					mrDiffusionSeq.setNext(mrAveragesSeq);
					mrAveragesSeq.setNext(mrImageFrameTypeSeq);
					mrImageFrameTypeSeq.setNext(frameAnatomySequence);
					frameAnatomySequence.setNext(planeOrientationSequence);
					planeOrientationSequence.setNext(pixelMeasuresSequence);
					pixelMeasuresSequence.setNext(pixelTransSequence);

					addTag(tags, createSequenceTag(Dictionary.TAG_5200_SHARED_FUNCTIONAL_GROUPS_SQ, createListFromItem(mrImagingModiferSeq)));
				}

				// Per-frame
				final List<Item> perFrameItems = new ArrayList<Item>();
				final int numSlices = id.getSlices();
				for (int ctr = 0; ctr < numSlices; ctr++) {
					// Frame Content Functional Group (C.7.6.16.2.2)
					final Item stackId = createExplicitTag(Dictionary.TAG_0020_STACK_ID, "1");
					final Item stackPosition = createExplicitTag(Dictionary.TAG_0020_IN_STACK_POSITION, ctr + 1);
					final Item temporalPosition = createExplicitTag(Dictionary.TAG_0020_TEMPORAL_POSITION_INDEX, 1);
					final Item dimIndexValues = createExplicitTag(Dictionary.TAG_0020_DIMENSION_INDEX_VALUES, ctr + 1);

					stackId.setNext(stackPosition);
					stackPosition.setNext(temporalPosition);
					temporalPosition.setNext(dimIndexValues);

					final Item frameContentSequence = createSequenceTag(Dictionary.TAG_0020_FRAME_CONTENT_SQ, createListFromItem(stackId));

					// Plane Position Functional Group (C.7.6.16.2.3)
					final Item imagePos = createExplicitTag(Dictionary.TAG_0020_IMAGE_POSITION,
							makeRealWorldCoordinateAt(0, 0, ctr, orientation, id, vd, origin).getValues());
					final Item planePositionSequence = createSequenceTag(Dictionary.TAG_0020_PLANE_POSITION_SQ, createListFromItem(imagePos));

					frameContentSequence.setNext(planePositionSequence);

					perFrameItems.add(frameContentSequence);
				}

				addTag(tags, createExplicitTag(Dictionary.TAG_5200_PER_FRAME_FUNCTIONAL_GROUPS_SQ, perFrameItems));
			}

			// Multi-frame Dimension Module (C.7.6.17)
			final Item dimOrgId = createExplicitTag(Dictionary.TAG_0020_DIMENSION_ORGANIZATION_ID, dimOrgUID);
			addTag(tags, createSequenceTag(Dictionary.TAG_0020_DIMENSION_ORGANIZATION_SQ, createListFromItem(dimOrgId)));

			final Item dimOrgId2 = createExplicitTag(Dictionary.TAG_0020_DIMENSION_ORGANIZATION_ID, dimOrgUID);
			final Item dimensionIndexPointer = createExplicitTag(Dictionary.TAG_0020_DIMENSION_INDEX_POINTER,
					new short[] { Dictionary.TAG_0020_IMAGE_POSITION.getGroupId(), Dictionary.TAG_0020_IMAGE_POSITION.getElementId() });
			final Item functionalGroupPointer = createExplicitTag(Dictionary.TAG_0020_FUNCTIONAL_GROUP_POINTER, new short[] {
					Dictionary.TAG_0020_PLANE_POSITION_SQ.getGroupId(), Dictionary.TAG_0020_PLANE_POSITION_SQ.getElementId() });

			dimOrgId2.setNext(dimensionIndexPointer);
			dimensionIndexPointer.setNext(functionalGroupPointer);
			addTag(tags, createSequenceTag(Dictionary.TAG_0020_DIMENSION_INDEX_SQ, createListFromItem(dimOrgId2)));

			// Acquisition Context Module (C.7.6.14)
			addTag(tags, createSequenceTag(Dictionary.TAG_0040_ACQUISITION_CONTEXT_SQ, createListFromItem(null)));

			// MR Pulse Sequence Module (C.8.13.4)
			addTag(tags, createTag(Dictionary.TAG_0018_PULSE_SEQUENCE_NAME, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_MR_ACQUISITION_TYPE, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_ECHO_PULSE_SEQUENCE, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_MULTI_PLANAR_EXCITATION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_PHASE_CONTRAST, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_TIME_OF_FLIGHT, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_STEADY_STATE_PULSE_SEQUENCE, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_ECHO_PLANAR_PULSE_SEQUENCE, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_SATURATION_RECOVERY, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_SPECTRALLY_SELECTED_SUPPRESSION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_OVERSAMPLING_PHASE, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_GEOMETRY_K_SPACE_TRAVERSAL, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_SEGMENTED_K_SPACE_TRAVERSAL, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_COVERAGE_OF_K_SPACE, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_NUMBER_K_SPACE_TRAJECTORIES, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_MULTIPLE_SPIN_ECHO, mappedDicomTags));

			// Enhanced MR Image Module (C.8.13.1)
			addTag(tags, createTag(Dictionary.TAG_0008_ACQUISITION_DATETIME, mappedDicomTags, DATE_TIME_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0018_ACQUISITION_DURATION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_CONTENT_QUALIFICATION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_RESONANT_NUCLEUS, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_K_SPACE_FILTERING, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_MAGNETIC_FIELD_STRENGTH, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_APPLICABLE_SAFETY_STANDARD_AGENCY, mappedDicomTags));
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_IMAGE_TYPE, CollectionUtilities.array(FRAME_TYPE)));
			addTag(tags, createTag(Dictionary.TAG_0008_PIXEL_PRESENTATION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_VOLUMETRIC_PROPERTIES, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_VOLUME_BASED_CALCULATION_TECH, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_COMPLEX_IMAGE_COMP, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_ACQUISITION_CONTRAST, mappedDicomTags));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SAMPLES_PER_PIXEL, SAMPLES_PER_PIXEL));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PHOTOMETRIC_INTERP, PHOTOMETRIC_INTERP));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_BITS_ALLOCATED, it.getNumBytesPerVoxel() * 8));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_BITS_STORED, it.getNumBytesPerVoxel() * 8));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_HIGH_BIT, (it.getNumBytesPerVoxel() * 8) - 1));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PIXEL_REP, PIXEL_REPRESENTATION));
			addTag(tags, createTag(Dictionary.TAG_0028_BURNED_IN_ANNOTATION, mappedDicomTags));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_LOSSY_IMAGE_COMPRESSION, LOSSY_IMAGE_COMPRESSION));
			addTag(tags, createExplicitTag(Dictionary.TAG_2050_PRESENTATION_LUT_SHAPE, PRESENTATION_LUT_SHAPE));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_COLUMNS, id.getCols()));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_ROWS, id.getRows()));

			// Additional Tags
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SMALLEST_IMAGE_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMin(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_LARGEST_IMAGE_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMax(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SMALLEST_SERIES_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMin(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_LARGEST_SERIES_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMax(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_WINDOW_WIDTH, new double[] { ir.getDisplayMax() - ir.getDisplayMin() }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_WINDOW_CENTER, new double[] { (ir.getDisplayMax() + ir.getDisplayMin()) / 2.0 }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PIXEL_SPACING, new double[] { vd.getRowSize(true), vd.getColSize(true) }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0018_SLICE_THICKNESS, new double[] { vd.getSliceThickness(true) }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0020_IMAGE_POSITION, makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin).getValues()));
			//addTag(tags, createExplicitTag(Dictionary.TAG_0028_RESCALE_INTERCEPT, new double[] { ir.getDataScaleIntercepts()[seriesIndex * id.getSlices()] }));
			//addTag(tags, createExplicitTag(Dictionary.TAG_0028_RESCALE_SLOPE, new double[] { ir.getDataScaleSlopes()[seriesIndex * id.getSlices()] }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0018_REPETITION_TIME, new double[] { vd.getTR() * 1000 }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0018_ECHO_TIME, new double[] { vd.getTE1() }));

			final Coordinate lastPixelFirstRow = makeRealWorldCoordinateAt(id.getCols() - 1, 0, 0, orientation, id, vd, origin);
			final Coordinate firstPixelFirstRow = makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin);
			final Vector3d vecRow = new Vector3d(lastPixelFirstRow.xDbl - firstPixelFirstRow.xDbl, lastPixelFirstRow.yDbl - firstPixelFirstRow.yDbl,
					lastPixelFirstRow.zDbl - firstPixelFirstRow.zDbl);
			vecRow.normalize();

			final Coordinate lastPixelFirstCol = makeRealWorldCoordinateAt(0, id.getRows() - 1, 0, orientation, id, vd, origin);
			final Coordinate firstPixelFirstCol = makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin);
			final Vector3d vecCol = new Vector3d(lastPixelFirstCol.xDbl - firstPixelFirstCol.xDbl, lastPixelFirstCol.yDbl - firstPixelFirstCol.yDbl,
					lastPixelFirstCol.zDbl - firstPixelFirstCol.zDbl);
			vecCol.normalize();

			addTag(tags, createExplicitTag(Dictionary.TAG_0020_IMAGE_ORIENTATION, new double[] { vecRow.x, vecRow.y, vecRow.z, vecCol.x, vecCol.y, vecCol.z }));
		} catch (final IOException ex) {
			throw new VolumeIOException(ex.getMessage());
		}

		try {
			return makeHeader(tags, id, it, XFER_SYNTAX_EXPLICIT_VT_LITLE_ENDIAN, 1);
		} catch (final UnsupportedEncodingException ex) {
			throw new VolumeIOException(ex.getMessage());
		}
	}
}
