
package edu.uthscsa.ric.volume.formats.dicom;

import java.io.IOException;
import java.net.URI;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import edu.uthscsa.ric.utilities.AppLogger;
import edu.uthscsa.ric.utilities.ByteUtilities;
import edu.uthscsa.ric.utilities.CollectionUtilities;
import edu.uthscsa.ric.utilities.FileUtilities;
import edu.uthscsa.ric.volume.ImageType;


public class DICOM {

    private List<Item> elements;
    private URI imageFile;
    private boolean isFileSet;
    private boolean simpleToString;
    private boolean littleEndian;
    private int orderNumber;
    private short[] overlayData;
    private boolean needsOverlayData;

    public static final double OBLIQUITY_THRESHOLD_COSINE_VALUE = 0.8;

    // image dims
    private static final short[] TAG_ROWS = { 0x0028, 0x0010 };
    private static final short[] TAG_COLS = { 0x0028, 0x0011 };
    private static final short[] TAG_ACQUISITION_MATRIX = { 0x0018, 0x1310 };
    private static final short[] TAG_NUMBER_OF_FRAMES = { 0x0028, 0x0008 };
    private static final short[] TAG_NUMBER_TEMPORAL_POSITIONS = { 0x0020, 0x0105 };

    // voxel dims
    private static final short[] TAG_PIXEL_SPACING = { 0x0028, 0x0030 };
    private static final short[] TAG_SLICE_THICKNESS = { 0x0018, 0x0050 };
    private static final short[] TAG_SLICE_GAP = { 0x0018, 0x0088 };
    private static final short[] TAG_TR = { 0x0018, 0x0080 };
    private static final short[] TAG_TE = { 0x0018, 0x0081 };
    private static final short[] TAG_FRAME_TIME = { 0x0018, 0x1063 };
    private static final short[] TAG_PIXEL_ASPECT_RATIO = { 0x0028, 0x0034 };

    // datatype
    private static final short[] TAG_BITS_ALLOCATED = { 0x0028, 0x0100 };
    private static final short[] TAG_BITS_STORED = { 0x0028, 0x0101 };
    private static final short[] TAG_PIXEL_REPRESENTATION = { 0x0028, 0x0103 };
    private static final short[] TAG_HIGH_BIT = { 0x0028, 0x0102 };
    private static final short[] TAG_PHOTOMETRIC_INTERPRETATION = { 0x0028, 0x0004 };
    private static final short[] TAG_SAMPLES_PER_PIXEL = { 0x0028, 0x0002 };
    private static final short[] TAG_PLANAR_CONFIG = { 0x0028, 0x0006 };
    private static final short[] TAG_PALETTE_RED = { 0x0028, 0x1201 };
    private static final short[] TAG_PALETTE_GREEN = { 0x0028, 0x1202 };
    private static final short[] TAG_PALETTE_BLUE = { 0x0028, 0x1203 };

    // data scale
    private static final short[] TAG_DATA_SCALE_SLOPE = { 0x0028, 0x1053 };
    private static final short[] TAG_DATA_SCALE_INTERCEPT = { 0x0028, 0x1052 };
    private static final short[] TAG_DATA_SCALE_ELSCINT = { 0x0207, 0x101F };
    private static final short[] TAG_PIXEL_BANDWIDTH = { 0x0018, 0x0095 };

    // range
    private static final short[] TAG_IMAGE_MIN = { 0x0028, 0x0106 };
    private static final short[] TAG_IMAGE_MAX = { 0x0028, 0x0107 };
    private static final short[] TAG_WINDOW_CENTER = { 0x0028, 0x1050 };
    private static final short[] TAG_WINDOW_WIDTH = { 0x0028, 0x1051 };

    // descriptors
    private static final short[] TAG_PATIENT_NAME = { 0x0010, 0x0010 };
    private static final short[] TAG_PATIENT_ID = { 0x0010, 0x0020 };
    private static final short[] TAG_STUDY_DATE = { 0x0008, 0x0020 };
    private static final short[] TAG_SERIES_DATE = { 0x0008, 0x0021 };
    private static final short[] TAG_ACQUISITION_DATE = { 0x0008, 0x0022 };
    private static final short[] TAG_CONTENT_DATE = { 0x0008, 0x0023 };
    private static final short[] TAG_STUDY_TIME = { 0x0008, 0x0030 };
    private static final short[] TAG_STUDY_DES = { 0x0008, 0x1030 };
    private static final short[] TAG_IMAGE_TYPE = { 0x0008, 0x0008 };
    private static final short[] TAG_IMAGE_COMMENTS = { 0x0020, 0x4000 };
    private static final short[] TAG_SEQUENCE_NAME = { 0x0018, 0x0024 };
    private static final short[] TAG_MODALITY = { 0x0008, 0x0060 };

    // dicom ID
    static final short[] TAG_SOP_CLASS_UID = { 0x0002, 0x0002 };
    static final short[] TAG_TRANSFER_SYNTAX_ID = { 0x0002, 0x0010 };

    // session ID
    private static final short[] TAG_FRAME_OF_REF_UID = { 0x0020, 0x0052 };

    // study ID
    private static final short[] TAG_STUDY_UID = { 0x0020, 0x000D };

    // volume ID
    private static final short[] TAG_SERIES_DESCRIPTION = { 0x0008, 0x103E };
    private static final short[] TAG_SERIES_INSTANCE_UID = { 0x0020, 0x000E };
    private static final short[] TAG_SERIES_NUMBER = { 0x0020, 0x0011 };
    private static final short[] TAG_ECHO_NUMBER = { 0x0018, 0x0086 };
    private static final short[] TAG_TEMPORAL_POSITION = { 0x0020, 0x0100 };

    // slice ID
    private static final short[] TAG_IMAGE_NUM = { 0x0020, 0x0013 };
    private static final short[] TAG_SLICE_LOCATION = { 0x0020, 0x1041 };

    // orientation
    private static final short[] TAG_IMAGE_ORIENTATION = { 0x0020, 0x0037 };
    private static final short[] TAG_IMAGE_POSITION = { 0x0020, 0x0032 };
    private static final short[] TAG_SLICE_LOCATION_VECTOR = { 0x0018, 0x2005 };

    // sublists
    static final short[] TAG_SUBLIST_ITEM = { (short) 0xFFFE, (short) 0xE000 };
    static final short[] TAG_SUBLIST_ITEM_DELIM = { (short) 0xFFFE, (short) 0xE00D };
    static final short[] TAG_SUBLIST_SEQ_DELIM = { (short) 0xFFFE, (short) 0xE0DD };

    // file set
    static final String DICOMDIR = "DICOMDIR";
    static final short[] FIRST_RECORD = { (short) 0x0004, (short) 0x1200 };
    static final short[] NEXT_RECORD = { (short) 0x0004, (short) 0x1400 };
    static final short[] CHILD_RECORD = { (short) 0x0004, (short) 0x1420 };
    static final short[] FILE_SET_FLAG = { (short) 0x0004, (short) 0x1212 };
    static final short[] FILE_PATH = { (short) 0x0004, (short) 0x1500 };
    static final short[] DIRECTORY_RECORD_TYPE = { (short) 0x0004, (short) 0x1430 };

    // image data
    static final short[] TAG_PIXEL_DATA = { 0x7FE0, 0x0010 };

    // overlay
    static final short[] TAG_OVERLAY_ROWS = { 0x6000, 0x0010 };
    static final short[] TAG_OVERLAY_COLS = { 0x6000, 0x0011 };
    static final short[] TAG_OVERLAY_TYPE = { 0x6000, 0x0040 };
    static final short[] TAG_OVERLAY_ORIGIN = { 0x6000, 0x0050 };
    static final short[] TAG_OVERLAY_BITS_ALLOCATED = { 0x6000, 0x0100 };
    static final short[] TAG_OVERLAY_BIT_POSITION = { 0x6000, 0x0102 };
    static final short[] TAG_OVERLAY_DATA = { 0x6000, 0x3000 };
    static final short TAG_OVERLAY_GROUP_MIN = 0x6000;
    static final short TAG_OVERLAY_GROUP_MAX = 0x601E;

    // meta
    static final short[] TAG_META_LENGTH = { 0x0002, 0x0000 };



    public static boolean hasValidImagePosition(final DICOM[] dicoms, final String sliceDir) {
        for (final DICOM dicom : dicoms) {
            if ((dicom != null) && (dicom.getImagePosition(sliceDir) != 0)) {
                return true;
            }
        }
        return false;
    }



    public static boolean hasValidSliceLocation(final DICOM[] dicoms) {
        for (final DICOM dicom : dicoms) {
            if ((dicom != null) && (dicom.getSliceLocation() != 0)) {
                return true;
            }
        }
        return false;
    }



    public static DICOM[] orderDicoms(final DICOM[] dicoms, final int numFrames, final String sliceDir) {
        final boolean hasImagePosition = hasValidImagePosition(dicoms, sliceDir);
        final boolean hasSliceLocation = hasValidSliceLocation(dicoms);
        final boolean hasImageNumber = hasValidImageNumber(dicoms);
        int dicomIndex = 0;

        final TreeMap<Integer, Vector<DICOM>> timeMap = orderByTime(dicoms, numFrames, sliceDir, hasImagePosition, hasSliceLocation);
        final Iterator<Vector<DICOM>> timeIt = timeMap.values().iterator();
        final List<DICOM> orderedList = new ArrayList<DICOM>();

        while (timeIt.hasNext()) {
            final Vector<DICOM> dg = timeIt.next();
            Collection<DICOM> ordered = null;
            if (hasImagePosition) {
                ordered = orderByImagePosition(dg, sliceDir);
            } else if (hasSliceLocation) {
                ordered = orderBySliceLocation(dg);
            } else if (hasImageNumber) {
                ordered = orderByImageNumber(dg);
            } else {
                ordered = dg;
            }

            final Iterator<DICOM> volIt = ordered.iterator();
            while (volIt.hasNext()) {
                final DICOM dicom = volIt.next();
                dicom.setOrderNumber(dicomIndex++);
            }

            orderedList.addAll(ordered);
        }

        final DICOM[] dicomOrdered = new DICOM[dicomIndex];

        for (int ctr = 0; ctr < dicomIndex; ctr++) { // order them
            dicomOrdered[orderedList.get(ctr).getOrderNumber()] = orderedList.get(ctr);
        }

        return dicomOrdered;
    }



    protected static Item getItem(final long group, final long elem, final long offset, final List<Item> someItems) {
        for (final Item item : someItems) {
            long currentGroup = item.getGroup();
            long currentElem = item.getElement();

            if ((currentGroup == group) && (currentElem == elem) && ((offset == -1) || (item.getOffsetStart() == offset))) {
                return item;
            }

            Item next = item.getNext();
            while (next != null) {
                currentGroup = next.getGroup();
                currentElem = next.getElement();

                if ((((currentGroup == group) && (currentElem == elem) && ((offset == -1))) || (next.getOffsetStart() == offset))) {
                    return next;
                }

                next = next.getNext();
            }
        }

        for (final Item item : someItems) {
            if (item.isSublist()) {
                final Item foundItem = getItem(group, elem, offset, item.getSublist());
                if (foundItem != null) {
                    return foundItem;
                }
            }

            Item next = item.getNext();
            while (next != null) {
                if (next.isSublist()) {
                    final Item foundItem = getItem(group, elem, offset, next.getSublist());
                    if (foundItem != null) {
                        return foundItem;
                    }
                }

                next = next.getNext();
            }
        }

        return null;
    }



    public List<Integer> getTemporalPositions() {
        final List<Item> tags = new ArrayList<Item>();
        getAllItems(TAG_TEMPORAL_POSITION[0], TAG_TEMPORAL_POSITION[1], elements, tags);
        final List<Integer> positions = new ArrayList<>();

        for (final Item tag : tags) {
            final byte[] value = tag.getValue();

            if (value == null) {
                positions.add(0);
            }

            final String imageNum = new String(value);
            int imageNumI = 0;

            try {
                imageNumI = Integer.parseInt(imageNum.trim());
            } catch (final NumberFormatException ex) {
                AppLogger.info(ex);
            }

            positions.add(imageNumI);
        }

        return positions;
    }



    public double getImagePosition(final String direction) {
        if (direction == null) {
            return 0;
        }

        final String dir = direction.toLowerCase();
        double value = 0;

        final byte[] valuePos = getValue(TAG_IMAGE_POSITION[0], TAG_IMAGE_POSITION[1]);

        if (valuePos != null) {
            try {
                final String valuePosString = new String(valuePos);
                final StringTokenizer valuePosTok = new StringTokenizer(valuePosString, "\\");

                final double posX = Double.parseDouble(valuePosTok.nextToken());
                final double posY = Double.parseDouble(valuePosTok.nextToken());
                final double posZ = Double.parseDouble(valuePosTok.nextToken());

                if (dir.equals("sagittal")) {
                    value = posX;
                } else if (dir.equals("coronal")) {
                    value = posY;
                } else if (dir.equals("axial")) {
                    value = posZ;
                } else {
                    return 0;
                }
            } catch (final NumberFormatException ex) {
                AppLogger.info(ex);
            }
        }

        return value;
    }



    public boolean hasValidImagePosition() {
        final List<Double> values = parseDoubles(getImagePosition());
        boolean found = false;
        for (final Double value : values) {
            if (value != 0) {
                found = true;
                break;
            }
        }

        return found;
    }



    public static List<Double> parseDoubles(final String str) {
        final StringTokenizer tok = new StringTokenizer(str, "\\");
        final List<Double> values = new ArrayList<>();

        while (tok.hasMoreTokens()) {
            values.add(Double.parseDouble(tok.nextToken()));
        }

        return values;
    }



    public List<Double> getImagePositions(final String direction) {
        final List<Item> tags = new ArrayList<Item>();
        getAllItems(TAG_IMAGE_POSITION[0], TAG_IMAGE_POSITION[1], elements, tags);
        final List<Double> positions = new ArrayList<>();

        final String dir = direction.toLowerCase();

        for (final Item tag : tags) {
            final byte[] valuePos = tag.getValue();

            if (valuePos != null) {
                try {
                    final String valuePosString = new String(valuePos);
                    final StringTokenizer valuePosTok = new StringTokenizer(valuePosString, "\\");

                    final double posX = Double.parseDouble(valuePosTok.nextToken());
                    final double posY = Double.parseDouble(valuePosTok.nextToken());
                    final double posZ = Double.parseDouble(valuePosTok.nextToken());

                    if (dir.equals("sagittal")) {
                        positions.add(posX);
                    } else if (dir.equals("coronal")) {
                        positions.add(posY);
                    } else if (dir.equals("axial")) {
                        positions.add(posZ);
                    } else {
                        positions.add(0.0);
                    }
                } catch (final NumberFormatException ex) {
                    AppLogger.info(ex);
                }
            }
        }

        return positions;
    }



    protected static void getAllItems(final long group, final long elem, final List<Item> someItems, final List<Item> foundList) {
        if (someItems != null) {
            for (final Item item : someItems) {
                long currentGroup = item.getGroup();
                long currentElem = item.getElement();

                if ((currentGroup == group) && (currentElem == elem)) {
                    foundList.add(item);
                }

                Item next = item.getNext();
                while (next != null) {
                    currentGroup = next.getGroup();
                    currentElem = next.getElement();

                    if ((currentGroup == group) && (currentElem == elem)) {
                        foundList.add(next);
                    }

                    next = next.getNext();
                }
            }

            for (final Item item : someItems) {

                if (item.isSublist()) {
                    getAllItems(group, elem, item.getSublist(), foundList);
                }

                Item next = item.getNext();

                while (next != null) {
                    if (next.isSublist()) {
                        getAllItems(group, elem, next.getSublist(), foundList);
                    }

                    next = next.getNext();
                }
            }
        }
    }



    private static String getPrivateTags(final List<Item> someItems) {
        return getPrivateTags(someItems, true);
    }



    private static String getPrivateTags(final List<Item> someItems, final boolean ignoreSublist) {
        final StringBuffer sb = new StringBuffer();

        if (someItems != null) {
            for (final Item someItem : someItems) {
                String pt = null;

                if (someItem.isSublist() && !ignoreSublist) {
                    pt = getPrivateTags(someItem.getSublist());
                } else {
                    pt = someItem.getPrivateTags();
                }

                if (pt != null) {
                    sb.append(pt);
                }
            }
        }

        return sb.toString();
    }



    private static boolean hasMatchingSlice(final Vector<DICOM> dg, final DICOM dicom, final String sliceDir, final boolean doImagePos,
            final boolean doSliceLoc) {
        double matchingNum = 0;
        if (doImagePos) {
            matchingNum = dicom.getImagePosition(sliceDir);
        } else if (doSliceLoc) {
            matchingNum = dicom.getSliceLocation();
        } else {
            matchingNum = dicom.getImageNumber();
        }

        final Iterator<DICOM> it = dg.iterator();
        while (it.hasNext()) {
            final DICOM current = it.next();
            if (doImagePos) {
                final double imagePos = current.getImagePosition(sliceDir);
                if (imagePos == matchingNum) {
                    return true;
                }
            } else if (doSliceLoc) {
                final double sliceLoc = current.getSliceLocation();
                if (sliceLoc == matchingNum) {
                    return true;
                }
            } else {
                final double imageNum = current.getImageNumber();
                if (imageNum == matchingNum) {
                    return true;
                }
            }
        }
        return false;
    }



    private static boolean hasValidImageNumber(final DICOM[] dicoms) {
        for (final DICOM dicom : dicoms) {
            if ((dicom != null) && (dicom.getImageNumber() != 0)) {
                return true;
            }
        }
        return false;
    }



    private static boolean hasValidTemporalNumber(final DICOM[] dicoms, final int numFrames) {
        for (final DICOM dicom : dicoms) {
            if (dicom != null) {
                final int pos = dicom.getNumTemporalPositions();

                if ((pos != 0) && (pos != numFrames)) {
                    return false;
                }
            }
        }
        return true;
    }



    protected static boolean hasValidTemporalPosition(final DICOM[] dicoms) {
        for (final DICOM dicom : dicoms) {
            if ((dicom != null) && (dicom.getTemporalPosition() != 0)) {
                return true;
            }
        }
        return false;
    }



    private static Collection<DICOM> orderByImageNumber(final Vector<DICOM> dicoms) {
        final TreeMap<Double, DICOM> dicomMap = new TreeMap<Double, DICOM>();

        final Iterator<DICOM> it = dicoms.iterator();
        while (it.hasNext()) {
            final DICOM dicom = it.next();
            dicomMap.put(new Double(dicom.getImageNumber()), dicom);
        }

        return dicomMap.values();
    }



    private static Collection<DICOM> orderByImagePosition(final Vector<DICOM> dicoms, final String sliceDir) {
        final TreeMap<Double, DICOM> dicomMap = new TreeMap<Double, DICOM>();

        final Iterator<DICOM> it = dicoms.iterator();
        while (it.hasNext()) {
            final DICOM dicom = it.next();
            dicomMap.put(dicom.getImagePosition(sliceDir), dicom);
        }

        return dicomMap.values();
    }



    private static Collection<DICOM> orderBySliceLocation(final Vector<DICOM> dicoms) {
        final TreeMap<Double, DICOM> dicomMap = new TreeMap<Double, DICOM>();

        final Iterator<DICOM> it = dicoms.iterator();
        while (it.hasNext()) {
            final DICOM dicom = it.next();
            dicomMap.put(dicom.getSliceLocation(), dicom);
        }

        return dicomMap.values();
    }



    private static TreeMap<Integer, Vector<DICOM>> orderByTime(final DICOM[] dicoms, final int numFrames, final String sliceDir, final boolean hasImagePosition,
            final boolean hasSliceLocation) {
        final TreeMap<Integer, Vector<DICOM>> dicomMap = new TreeMap<Integer, Vector<DICOM>>();
        final boolean hasTemporalPosition = (numFrames > 1) && hasValidTemporalPosition(dicoms);
        final boolean hasTemporalNumber = (numFrames > 1) && hasValidTemporalNumber(dicoms, numFrames);

        if (hasTemporalPosition && hasTemporalNumber) { // explicit series
            for (final DICOM dicom : dicoms) {
                if (dicom != null) {
                    final Integer tempPos = dicom.getTemporalPosition();
                    Vector<DICOM> dg = dicomMap.get(tempPos);
                    if (dg == null) {
                        dg = new Vector<DICOM>();
                        dicomMap.put(tempPos, dg);
                    }
                    dg.add(dicom);
                }
            }
        } else { // implicit series
            // order data by slice then time
            final TreeMap<Double, TreeMap<Integer, DICOM>> timeBySliceMap = new TreeMap<Double, TreeMap<Integer, DICOM>>();
            for (int ctr = 0; ctr < dicoms.length; ctr++) {
                if (dicoms[ctr] != null) {
                    final int imageNum = dicoms[ctr].getImageNumber();
                    double sliceMarker = ctr;
                    if (hasImagePosition) {
                        sliceMarker = dicoms[ctr].getImagePosition(sliceDir);
                    } else if (hasSliceLocation) {
                        sliceMarker = dicoms[ctr].getSliceLocation();
                    }

                    TreeMap<Integer, DICOM> slice = timeBySliceMap.get(sliceMarker);
                    if (slice == null) {
                        slice = new TreeMap<Integer, DICOM>();
                        timeBySliceMap.put(sliceMarker, slice);
                    }

                    //					while (slice.containsKey(imageNum)) { // there was a case where a series contained two identical images, only differing by instance UIDs
                    //						imageNum += 10000;
                    //					}

                    slice.put(imageNum, dicoms[ctr]);
                }
            }

            // copy into DICOM array (ordered by slice by time)
            final DICOM[] dicomsCopy = new DICOM[dicoms.length];
            int dicomsCopyIndex = 0;
            final Iterator<TreeMap<Integer, DICOM>> sliceIt = timeBySliceMap.values().iterator();
            while (sliceIt.hasNext()) {
                final TreeMap<Integer, DICOM> slice = sliceIt.next();
                final Iterator<DICOM> timeIt = slice.values().iterator();
                while (timeIt.hasNext()) {
                    dicomsCopy[dicomsCopyIndex++] = timeIt.next();
                }
            }

            // groups dicoms by timepoint
            for (int ctr = 0; ctr < dicomsCopy.length; ctr++) {
                if (dicomsCopy[ctr] != null) {
                    Vector<DICOM> dgFound = null;
                    final Iterator<Vector<DICOM>> it = dicomMap.values().iterator();
                    while (it.hasNext()) {
                        final Vector<DICOM> dg = it.next();
                        if (!hasMatchingSlice(dg, dicomsCopy[ctr], sliceDir, hasImagePosition, hasSliceLocation)) {
                            dgFound = dg;
                            break;
                        }
                    }

                    if (dgFound == null) {
                        dgFound = new Vector<DICOM>();
                        dicomMap.put(dicomMap.size(), dgFound);
                    }
                    dgFound.add(dicomsCopy[ctr]);
                }
            }
        }

        return dicomMap;
    }



    public DICOM() {}



    public DICOM(final URI uri) throws IOException {
        imageFile = uri;

        final Parser parser = new Parser(false);
        this.addElements(parser.parse(uri));
    }



    public void addElements(final List<Item> someElements) {
        elements = someElements;
        final String transferSyntax = getTransferSyntaxID();
        if (Constants.TRANSFER_SYNTAX_ID_EXPLICIT_BIG.equals(transferSyntax)) {
            littleEndian = false;
        } else {
            littleEndian = true;
        }

        makeOverlayData();
    }



    public void clearPrivateTagData() {
        clearPrivateTagData(elements);
    }



    public int[] getAcquisitionMatrix() {
        final byte[] value = getValue(TAG_ACQUISITION_MATRIX[0], TAG_ACQUISITION_MATRIX[1]);
        final int[] acquisitionMatrix = new int[2];

        if (value != null) {
            if ((getShort(value, 0) > 0)) {
                acquisitionMatrix[1] = getShort(value, 0);
            } else if ((getShort(value, 4) > 0)) {
                acquisitionMatrix[1] = getShort(value, 4);
            }

            if ((getShort(value, 2) > 0)) {
                acquisitionMatrix[0] = getShort(value, 2);
            } else if ((getShort(value, 6) > 0)) {
                acquisitionMatrix[0] = getShort(value, 6);
            }
        }

        final String privateTags = getPrivateTags();

        if (((acquisitionMatrix[0] == 0) || (acquisitionMatrix[1] == 0)) && (privateTags != null)) {
            try {
                final int index = privateTags.indexOf("AcquisitionMatrixText");
                if (index != -1) {
                    final Pattern intsOnly = Pattern.compile("\\d+");
                    final Matcher makeMatch = intsOnly.matcher(privateTags.substring(index));

                    if (makeMatch.find()) {
                        final int val = Integer.parseInt(makeMatch.group());

                        if (val > 0) {
                            acquisitionMatrix[0] = val;
                        }
                    }

                    if (makeMatch.find()) {
                        final int val = Integer.parseInt(makeMatch.group());

                        if (val > 0) {
                            acquisitionMatrix[1] = val;
                        }
                    }
                }
            } catch (final NumberFormatException ex) {
                AppLogger.info(ex);
            }
        }

        if (acquisitionMatrix[1] == 0) {
            acquisitionMatrix[1] = acquisitionMatrix[0];
        }

        return acquisitionMatrix;
    }



    public String getAllDescriptiveData() {
        String string = "";

        byte[] value = getValue(TAG_PATIENT_NAME[0], TAG_PATIENT_NAME[1]);
        if (value != null) {
            string += new String(value);
        }

        value = getValue(TAG_PATIENT_ID[0], TAG_PATIENT_ID[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        value = getValue(TAG_STUDY_DATE[0], TAG_STUDY_DATE[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        value = getValue(TAG_STUDY_TIME[0], TAG_STUDY_TIME[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        value = getValue(TAG_STUDY_DES[0], TAG_STUDY_DES[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        value = getValue(TAG_SERIES_DESCRIPTION[0], TAG_SERIES_DESCRIPTION[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        value = getValue(TAG_IMAGE_COMMENTS[0], TAG_IMAGE_COMMENTS[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        return string;
    }



    public int getBitsAllocated() {
        final byte[] value = getValue(TAG_BITS_ALLOCATED[0], TAG_BITS_ALLOCATED[1]);

        if (value == null) {
            return 16;
        }

        final byte[] samplesValue = getValue(TAG_SAMPLES_PER_PIXEL[0], TAG_SAMPLES_PER_PIXEL[1]);
        int multiplier = 1;

        if (getDataType() == ImageType.BYTE_TYPE_RGB) {
            multiplier = getShort(samplesValue, 0);
            if (multiplier == 0) {
                multiplier = 1;
            }
        }

        int bitsAlloc = getShort(value, 0);
        if (bitsAlloc == 0) {
            bitsAlloc = 16;
        }

        return multiplier * bitsAlloc;
    }



    private short getShort(final byte[] buffer, final int index) {
        if (littleEndian) {
            return ByteUtilities.swapShort(buffer, index);
        } else {
            return ByteUtilities.getShort(buffer, index);
        }
    }



    public int getBitsStored() {
        final byte[] value = getValue(TAG_BITS_STORED[0], TAG_BITS_STORED[1]);

        if (value == null) {
            return getBitsAllocated();
        }

        return getShort(value, 0);
    }



    public int getCols() {
        final byte[] value = getValue(TAG_COLS[0], TAG_COLS[1]);

        if (value == null) {
            return 0;
        }

        return getShort(value, 0);
    }



    public double getColSpacing() {
        byte[] value = getValue(TAG_PIXEL_SPACING[0], TAG_PIXEL_SPACING[1]);
        double colSpacingValue = 0;

        if (value != null) {
            final String pixelSpacing = new String(value);
            final String colSpacing = pixelSpacing.substring(pixelSpacing.indexOf('\\') + 1);

            try {
                colSpacingValue = Double.parseDouble(colSpacing.trim());
            } catch (final NumberFormatException ex) {
                AppLogger.info(ex);
            }
        }

        if (colSpacingValue == 0) {
            value = getValue(TAG_PIXEL_ASPECT_RATIO[0], TAG_PIXEL_ASPECT_RATIO[1]);

            if (value != null) {
                colSpacingValue = 1;
            }
        }

        return colSpacingValue;
    }



    public float[] getCosines() {
        float[] cosines = null;

        final byte[] valueOrient = getValue(TAG_IMAGE_ORIENTATION[0], TAG_IMAGE_ORIENTATION[1]);

        if (valueOrient == null) {
            return null;
        }

        try {
            final String valueOrientString = new String(valueOrient);
            final StringTokenizer valueOrientTok = new StringTokenizer(valueOrientString, "\\");

            final double xA = Double.parseDouble(valueOrientTok.nextToken());
            final double xB = Double.parseDouble(valueOrientTok.nextToken());
            final double xC = Double.parseDouble(valueOrientTok.nextToken());
            final double yA = Double.parseDouble(valueOrientTok.nextToken());
            final double yB = Double.parseDouble(valueOrientTok.nextToken());
            final double yC = Double.parseDouble(valueOrientTok.nextToken());

            cosines = new float[] { (float) xA, (float) xB, (float) xC, (float) yA, (float) yB, (float) yC };
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return cosines;
    }



    private boolean hasElscintDataScale() {
        final byte[] value = getValue(TAG_DATA_SCALE_ELSCINT[0], TAG_DATA_SCALE_ELSCINT[1]);
        return CollectionUtilities.isNotEmpty(value);
    }



    public double getDataScaleElscint() {
        final byte[] value = getValue(TAG_DATA_SCALE_ELSCINT[0], TAG_DATA_SCALE_ELSCINT[1]);

        if (value == null) {
            return 0;
        }

        final double tempScale = ByteUtilities.getDouble2(value);
        final double bandwidth = getPixelBandwidth();
        double slope = (float) (Math.sqrt(bandwidth) / (10 * tempScale));

        if (slope <= 0) {
            slope = 1;
        }

        return slope;
    }



    public double getDataScaleIntercept() {
        if (hasElscintDataScale()) {
            return 0;
        }

        final byte[] value = getValue(TAG_DATA_SCALE_INTERCEPT[0], TAG_DATA_SCALE_INTERCEPT[1]);

        if (value == null) {
            return 0;
        }

        final String intercept = new String(value);

        double interceptD = 0;

        try {
            if (StringUtils.isNotBlank(intercept)) {
                interceptD = Double.parseDouble(intercept.trim());
            }
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return interceptD;
    }



    public double getDataScaleSlope() {
        if (hasElscintDataScale()) {
            return getDataScaleElscint();
        }

        final byte[] value = getValue(TAG_DATA_SCALE_SLOPE[0], TAG_DATA_SCALE_SLOPE[1]);

        if (value == null) {
            return 1;
        }

        final String slope = new String(value);

        double slopeD = 1;

        try {
            if (StringUtils.isNotBlank(slope)) {
                slopeD = Double.parseDouble(slope.trim());
            }
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return slopeD;
    }



    public boolean isPalette() {
        final byte[] value = getValue(TAG_PHOTOMETRIC_INTERPRETATION[0], TAG_PHOTOMETRIC_INTERPRETATION[1]);
        if (value != null) {
            final String str = (new String(value)).trim();

            if (str.toLowerCase().indexOf("palette") != -1) {
                return true;
            }
        }

        return false;
    }



    private int[] scalePalette(final int[] pal) {
        int min, max;
        min = max = pal[0];

        for (int ctr = 1; ctr < pal.length; ctr++) {
            if (pal[ctr] > max) {
                max = pal[ctr];
            } else if (pal[ctr] < min) {
                min = pal[ctr];
            }
        }

        if ((max > 255) || (min < 0)) {
            final double slope = 255.0 / (max - min);
            final double intercept = min;

            for (int ctr = 0; ctr < pal.length; ctr++) {
                pal[ctr] = (int) Math.round((pal[ctr] - intercept) * slope);
            }
        }

        return pal;
    }



    public int[] getPalleteRed() {
        int[] redsBig = null;
        int[] redsLittle = null;
        final byte[] value = getValue(TAG_PALETTE_RED[0], TAG_PALETTE_RED[1]);

        if (CollectionUtilities.isNotEmpty(value)) {
            final int numVals = value.length / 2;
            redsBig = new int[numVals];
            redsLittle = new int[numVals];

            for (int ctr = 0; ctr < numVals; ctr++) {
                redsBig[ctr] = ByteUtilities.getShort(value, ctr * 2) & 0xFFFF;
                redsLittle[ctr] = ByteUtilities.swapShort(value, ctr * 2) & 0xFFFF;
            }

            final List<Integer> redBigList = Arrays.asList(ArrayUtils.toObject(redsBig));
            final int redBigDiff = Collections.max(redBigList) - Collections.min(redBigList);

            final List<Integer> redLittleList = Arrays.asList(ArrayUtils.toObject(redsLittle));
            final int redLittleDiff = Collections.max(redLittleList) - Collections.min(redLittleList);

            if (Math.abs(redBigDiff) < Math.abs(redLittleDiff)) {
                return scalePalette(redsBig);
            } else {
                return scalePalette(redsLittle);
            }
        }

        return CollectionUtilities.EMPTY_INT_ARRAY;
    }



    public int[] getPalleteGreen() {
        int[] greensBig = null;
        int[] greensLittle = null;
        final byte[] value = getValue(TAG_PALETTE_GREEN[0], TAG_PALETTE_GREEN[1]);

        if (CollectionUtilities.isNotEmpty(value)) {
            final int numVals = value.length / 2;
            greensBig = new int[numVals];
            greensLittle = new int[numVals];

            for (int ctr = 0; ctr < numVals; ctr++) {
                greensBig[ctr] = ByteUtilities.getShort(value, ctr * 2) & 0xFFFF;
                greensLittle[ctr] = ByteUtilities.swapShort(value, ctr * 2) & 0xFFFF;
            }

            final List<Integer> greenBigList = Arrays.asList(ArrayUtils.toObject(greensBig));
            final int greenBigDiff = Collections.max(greenBigList) - Collections.min(greenBigList);

            final List<Integer> greenLittleList = Arrays.asList(ArrayUtils.toObject(greensLittle));
            final int greenLittleDiff = Collections.max(greenLittleList) - Collections.min(greenLittleList);

            if (Math.abs(greenBigDiff) < Math.abs(greenLittleDiff)) {
                return scalePalette(greensBig);
            } else {
                return scalePalette(greensLittle);
            }
        }

        return CollectionUtilities.EMPTY_INT_ARRAY;
    }



    public int[] getPalleteBlue() {
        int[] bluesBig = null;
        int[] bluesLittle = null;
        final byte[] value = getValue(TAG_PALETTE_BLUE[0], TAG_PALETTE_BLUE[1]);

        if (CollectionUtilities.isNotEmpty(value)) {
            final int numVals = value.length / 2;
            bluesBig = new int[numVals];
            bluesLittle = new int[numVals];

            for (int ctr = 0; ctr < numVals; ctr++) {
                bluesBig[ctr] = ByteUtilities.getShort(value, ctr * 2) & 0xFFFF;
                bluesLittle[ctr] = ByteUtilities.swapShort(value, ctr * 2) & 0xFFFF;
            }

            final List<Integer> blueBigList = Arrays.asList(ArrayUtils.toObject(bluesBig));
            final int blueBigDiff = Collections.max(blueBigList) - Collections.min(blueBigList);

            final List<Integer> blueLittleList = Arrays.asList(ArrayUtils.toObject(bluesLittle));
            final int blueLittleDiff = Collections.max(blueLittleList) - Collections.min(blueLittleList);

            if (Math.abs(blueBigDiff) < Math.abs(blueLittleDiff)) {
                return scalePalette(bluesBig);
            } else {
                return scalePalette(bluesLittle);
            }
        }

        return CollectionUtilities.EMPTY_INT_ARRAY;
    }



    public int getDataType() {
        byte[] value = getValue(TAG_PIXEL_REPRESENTATION[0], TAG_PIXEL_REPRESENTATION[1]);

        if (value == null) {
            return ImageType.BYTE_TYPE_UNKNOWN;
        }

        final int dataType = getShort(value, 0);

        value = getValue(TAG_PHOTOMETRIC_INTERPRETATION[0], TAG_PHOTOMETRIC_INTERPRETATION[1]);
        if (value != null) {
            final String str = (new String(value)).trim();

            if (str.equals("RGB") || str.startsWith("YBR")) {
                return ImageType.BYTE_TYPE_RGB;
            }
        }

        if (dataType == 0) {
            return ImageType.BYTE_TYPE_INTEGER_UNSIGNED;
        } else if (dataType == 1) {
            return ImageType.BYTE_TYPE_INTEGER;
        } else {
            return ImageType.BYTE_TYPE_UNKNOWN;
        }
    }



    public String getEchoNumber() {
        final byte[] value = getValue(TAG_ECHO_NUMBER[0], TAG_ECHO_NUMBER[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getFrameOfReferenceUID() {
        final byte[] value = getValue(TAG_FRAME_OF_REF_UID[0], TAG_FRAME_OF_REF_UID[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public double getFrameTime() {
        final byte[] value = getValue(TAG_FRAME_TIME[0], TAG_FRAME_TIME[1]);

        if (value == null) {
            return 0;
        }

        final String frameTime = new String(value);
        double frameTimeD = 0;

        try {
            frameTimeD = Double.parseDouble(frameTime.trim());
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return frameTimeD;
    }



    public int getHighBit() {
        return getShort(getValue(TAG_HIGH_BIT[0], TAG_HIGH_BIT[1]), 0);
    }



    public String getImageDescription() {
        String string = "";

        byte[] value = getValue(TAG_STUDY_DES[0], TAG_STUDY_DES[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        value = getValue(TAG_SERIES_DESCRIPTION[0], TAG_SERIES_DESCRIPTION[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        value = getValue(TAG_IMAGE_COMMENTS[0], TAG_IMAGE_COMMENTS[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        return string;
    }



    public URI getImageFile() {
        return imageFile;
    }



    public String getImageFilename() {
        return FileUtilities.getName(imageFile);
    }



    public String getImageID() {
        String string = "";

        byte[] value = getValue(TAG_PATIENT_NAME[0], TAG_PATIENT_NAME[1]);
        if (value != null) {
            string += new String(value);
        }

        value = getValue(TAG_PATIENT_ID[0], TAG_PATIENT_ID[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        return string;
    }



    public int getImageMax() {
        final byte[] value = getValue(TAG_IMAGE_MAX[0], TAG_IMAGE_MAX[1]);

        if (value == null) {
            return 0;
        }

        return (getShort(value, 0)) & 0xFFFF;
    }



    public int getImageMin() {
        final byte[] value = getValue(TAG_IMAGE_MIN[0], TAG_IMAGE_MIN[1]);

        if (value == null) {
            return 0;
        }

        return (getShort(value, 0)) & 0xFFFF;
    }



    public int getImageNumber() {
        byte[] value = getValue(TAG_IMAGE_NUM[0], TAG_IMAGE_NUM[1]);

        if (value == null) {
            value = getValue(TAG_IMAGE_NUM[0], TAG_IMAGE_NUM[1]);
            if (value == null) {
                return 0;
            }
        }

        final String imageNum = new String(value);
        int imageNumI = 0;

        if (StringUtils.isNotBlank(imageNum)) {
            try {
                imageNumI = Integer.parseInt(imageNum.trim());
            } catch (final NumberFormatException ex) {
                AppLogger.info(ex);
            }
        }

        return imageNumI;
    }



    public int getImageOffset() {
        return findImageOffset(elements);
    }



    private int findImageOffset(final List<Item> items) {
        for (final Item item : items) {
            if (item.isPixelData()) {
                return (int) item.getValueOffset();
            }
        }

        for (final Item item : items) {
            if (item.isSublist()) {
                final int offset = findImageOffset(item.getSublist());
                if (offset > -1) {
                    return offset;
                }
            }
        }

        return -1;
    }



    public String getImagePosition() {
        final byte[] valuePos = getValue(TAG_IMAGE_POSITION[0], TAG_IMAGE_POSITION[1]);
        String valuePosString = "";

        if (valuePos != null) {
            valuePosString = new String(valuePos);
        }

        return valuePosString;
    }



    public String getImageTime() {
        String string = "";

        byte[] value = getValue(TAG_STUDY_DATE[0], TAG_STUDY_DATE[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        value = getValue(TAG_STUDY_TIME[0], TAG_STUDY_TIME[1]);
        if (value != null) {
            string += (" " + new String(value));
        }

        return string;
    }



    public String getImageType() {
        final byte[] value = getValue(TAG_IMAGE_TYPE[0], TAG_IMAGE_TYPE[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getModality() {
        final byte[] value = getValue(TAG_MODALITY[0], TAG_MODALITY[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public int getNumberOfFrames() {
        final byte[] value = getValue(TAG_NUMBER_OF_FRAMES[0], TAG_NUMBER_OF_FRAMES[1]);

        if (value == null) {
            return 0;
        }

        int temp = 1;

        try {
            temp = Integer.parseInt((new String(value)).trim());
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return temp;
    }



    public int getNumberOfMosaicImages() {
        int numberOfImages = -1;

        final String privateTags = getPrivateTags();
        if (privateTags != null) {
            try {
                final int index = privateTags.indexOf("NumberOfImagesInMosaic");
                if (index != -1) {
                    final Pattern intsOnly = Pattern.compile("\\d+");
                    final Matcher makeMatch = intsOnly.matcher(privateTags.substring(index));
                    makeMatch.find();
                    numberOfImages = Integer.parseInt(makeMatch.group());
                }
            } catch (final NumberFormatException ex) {
                AppLogger.info(ex);
            }
        }

        return numberOfImages;
    }



    public int getNumTemporalPositions() {
        final byte[] value = getValue(TAG_NUMBER_TEMPORAL_POSITIONS[0], TAG_NUMBER_TEMPORAL_POSITIONS[1]);

        if (value == null) {
            return 0;
        }

        int temp = 1;

        try {
            temp = Integer.parseInt((new String(value)).trim());
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return temp;
    }



    public int getOrderNumber() {
        return orderNumber;
    }



    public String getOrientation() {
        String orientation = null;

        final byte[] valueOrient = getValue(TAG_IMAGE_ORIENTATION[0], TAG_IMAGE_ORIENTATION[1]);
        final byte[] valuePos = getValue(TAG_IMAGE_POSITION[0], TAG_IMAGE_POSITION[1]);

        if ((valueOrient == null) || (valuePos == null)) {
            return null;
        }

        try {
            final String valueOrientString = new String(valueOrient);
            final StringTokenizer valueOrientTok = new StringTokenizer(valueOrientString, "\\");

            final double xA = Double.parseDouble(valueOrientTok.nextToken());
            final double xB = Double.parseDouble(valueOrientTok.nextToken());
            final double xC = Double.parseDouble(valueOrientTok.nextToken());
            final double yA = Double.parseDouble(valueOrientTok.nextToken());
            final double yB = Double.parseDouble(valueOrientTok.nextToken());
            final double yC = Double.parseDouble(valueOrientTok.nextToken());

            final float[] dirCos = new float[] { (float) xA, (float) xB, (float) xC, (float) yA, (float) yB, (float) yC };
            final float spacing = (float) getRowSpacing();
            final boolean swapZ = true;

            int i, bigRow = 0, bigCol = 0;
            float biggest = 0;
            final StringBuffer orient = new StringBuffer();

            for (i = 0; i < 3; i++) {
                if (Math.abs(dirCos[i]) > biggest) {
                    biggest = Math.abs(dirCos[i]);
                    bigRow = i;
                }
            }

            biggest = 0;
            for (; i < 6; i++) {
                if (Math.abs(dirCos[i]) > biggest) {
                    biggest = Math.abs(dirCos[i]);
                    bigCol = i;
                }
            }

            switch (bigRow) {
                case 0:
                    orient.append('X');
                    if (bigCol == 4) {
                        orient.append("YZ");
                    } else {
                        orient.append("ZY");
                    }
                    break;
                case 1:
                    orient.append('Y');
                    if (bigCol == 3) {
                        orient.append("XZ");
                    } else {
                        orient.append("ZX");
                    }
                    break;
                case 2:
                    orient.append('Z');
                    if (bigCol == 3) {
                        orient.append("XY");
                    } else {
                        orient.append("YX");
                    }
                    break;
                default:
                    break;
            }

            switch (bigRow) {
                case 0:
                    if (dirCos[bigRow] > 0.0) {
                        orient.append('-');
                    } else {
                        orient.append('+');
                    }
                    if (bigCol == 4) {
                        if (dirCos[bigCol] > 0.0) {
                            orient.append('-');
                        } else {
                            orient.append('+');
                        }
                    } else {
                        if (dirCos[bigCol] > 0.0) {
                            orient.append('+');
                        } else {
                            orient.append('-');
                        }
                    }
                    break;
                case 1:
                    if (dirCos[bigRow] > 0.0) {
                        orient.append('-');
                    } else {
                        orient.append('+');
                    }
                    if (bigCol == 3) {
                        if (dirCos[bigCol] > 0.0) {
                            orient.append('-');
                        } else {
                            orient.append('+');
                        }
                    } else {
                        if (dirCos[bigCol] > 0.0) {
                            orient.append('+');
                        } else {
                            orient.append('-');
                        }
                    }
                    break;
                case 2:
                    if (dirCos[bigRow] > 0.0) {
                        orient.append('+');
                    } else {
                        orient.append('-');
                    }
                    //Has to be X or Y so opposite senses
                    if (dirCos[bigCol] > 0.0) {
                        orient.append('-');
                    } else {
                        orient.append('+');
                    }
                    break;
                default:
                    break;
            }

            if (spacing == 0.0) {
                orient.append('+');
                return (new String(orient));
            }

            if (swapZ) {
                switch (orient.charAt(2)) {
                    case 'X':
                        if (spacing > 0.0) {
                            orient.append('-');
                        } else {
                            orient.append('+');
                        }
                        break;
                    case 'Y':
                    case 'Z':
                        if (spacing > 0.0) {
                            orient.append('+');
                        } else {
                            orient.append('-');
                        }
                        break;
                    default:
                        break;
                }
            } else {
                switch (orient.charAt(2)) {
                    case 'X':
                        if (spacing > 0.0) {
                            orient.append('+');
                        } else {
                            orient.append('-');
                        }
                        break;
                    case 'Y':
                    case 'Z':
                        if (spacing > 0.0) {
                            orient.append('-');
                        } else {
                            orient.append('+');
                        }
                        break;
                    default:
                        break;
                }
            }

            orientation = (new String(orient));
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
            orientation = null;
        }

        return orientation;
    }



    public String getPatientID() {
        final byte[] value = getValue(TAG_PATIENT_ID[0], TAG_PATIENT_ID[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getPatientName() {
        final byte[] value = getValue(TAG_PATIENT_NAME[0], TAG_PATIENT_NAME[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public double getPixelBandwidth() {
        final byte[] value = getValue(TAG_PIXEL_BANDWIDTH[0], TAG_PIXEL_BANDWIDTH[1]);

        if (value == null) {
            return 0;
        }

        final String pixelBandwidth = new String(value);
        double pixelBandwidthD = 0;

        try {
            pixelBandwidthD = Double.parseDouble(pixelBandwidth.trim());
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return pixelBandwidthD;
    }



    public int getRows() {
        final byte[] value = getValue(TAG_ROWS[0], TAG_ROWS[1]);

        if (value == null) {
            return 0;
        }

        return getShort(value, 0);
    }



    public int getPlanerConfig() {
        final byte[] value = getValue(TAG_PLANAR_CONFIG[0], TAG_PLANAR_CONFIG[1]);

        if (value == null) {
            return 0;
        }

        int temp = 0;

        try {
            temp = getShort(value, 0);
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return temp;
    }



    public double getRowSpacing() {
        byte[] value = getValue(TAG_PIXEL_SPACING[0], TAG_PIXEL_SPACING[1]);
        double rowSpacingValue = 0;

        if (value != null) {
            final String pixelSpacing = new String(value);
            final String rowSpacing = pixelSpacing.substring(0, pixelSpacing.indexOf('\\'));

            try {
                rowSpacingValue = Double.parseDouble(rowSpacing.trim());
            } catch (final NumberFormatException ex) {
                AppLogger.info(ex);
            }
        }

        if (rowSpacingValue == 0) {
            value = getValue(TAG_PIXEL_ASPECT_RATIO[0], TAG_PIXEL_ASPECT_RATIO[1]);

            if (value != null) {
                rowSpacingValue = getPixelAspectRatio();
            }
        }

        return rowSpacingValue;
    }



    public double getPixelAspectRatio() {
        final byte[] value = getValue(TAG_PIXEL_ASPECT_RATIO[0], TAG_PIXEL_ASPECT_RATIO[1]);
        double ratio = 1;

        if (value != null) {
            final String pixelRatio = new String(value);

            if (StringUtils.isNotBlank(pixelRatio)) {
                final String verticalSize = pixelRatio.substring(0, pixelRatio.indexOf('\\'));
                final String horizontalSize = pixelRatio.substring(pixelRatio.indexOf('\\') + 1);

                double verticalSizeValue = 1;
                double horizontalSizeValue = 1;

                try {
                    verticalSizeValue = Double.parseDouble(verticalSize.trim());
                    horizontalSizeValue = Double.parseDouble(horizontalSize.trim());
                } catch (final NumberFormatException ex) {
                    AppLogger.info(ex);
                }

                ratio = (verticalSizeValue / horizontalSizeValue);
            }
        }

        return ratio;
    }



    public String getSequenceName() {
        final byte[] value = getValue(TAG_SEQUENCE_NAME[0], TAG_SEQUENCE_NAME[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getSeriesDescription() {
        final byte[] value = getValue(TAG_SERIES_DESCRIPTION[0], TAG_SERIES_DESCRIPTION[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getSeriesNumber() {
        final byte[] value = getValue(TAG_SERIES_NUMBER[0], TAG_SERIES_NUMBER[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getSeriesUID() {
        final byte[] value = getValue(TAG_SERIES_INSTANCE_UID[0], TAG_SERIES_INSTANCE_UID[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getSessionID() {
        String idString = "";
        final String ref = getFrameOfReferenceUID();

        if (ref != null) {
            idString += ref;
        }

        return idString;
    }



    public double getSliceLocation() {
        final byte[] value = getValue(TAG_SLICE_LOCATION[0], TAG_SLICE_LOCATION[1]);

        if (value == null) {
            return 0;
        }

        final String sliceLoc = new String(value);
        double sliceLocD = 0;

        try {
            sliceLocD = Double.parseDouble(sliceLoc.trim());
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return sliceLocD;
    }



    public double[] getSliceLocationVector() {
        final byte[] value = getValue(TAG_SLICE_LOCATION_VECTOR[0], TAG_SLICE_LOCATION_VECTOR[1]);

        if (value == null) {
            return null;
        }

        final String sliceLocVecStr = new String(value);
        double[] sliceLocVector = null;

        try {
            final StringTokenizer tokenizer = new StringTokenizer(sliceLocVecStr, "\\");

            final int count = tokenizer.countTokens();
            sliceLocVector = new double[count];
            int index = 0;

            while (tokenizer.hasMoreElements()) {
                final String token = tokenizer.nextToken();
                sliceLocVector[index++] = Double.parseDouble(token);
            }
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return sliceLocVector;
    }



    public int getSliceSizeBytes() {
        return getCols() * getRows() * (getBitsAllocated() / 8);
    }



    public double getSliceSpacing() {
        final byte[] value = getValue(TAG_SLICE_GAP[0], TAG_SLICE_GAP[1]);

        if (value == null) {
            return 0;
        }

        final String gap = new String(value);

        double gapD = 0;

        try {
            gapD = Double.parseDouble(gap.trim());
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return gapD;
    }



    public double getSliceThickness() {
        final byte[] value = getValue(TAG_SLICE_THICKNESS[0], TAG_SLICE_THICKNESS[1]);

        if (value == null) {
            return 0;
        }

        final String sliceThickness = new String(value);
        double sliceThicknessD = 0;

        try {
            sliceThicknessD = Double.parseDouble(sliceThickness.trim());
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return sliceThicknessD;
    }



    public String getStudyDate() {
        final byte[] value = getValue(TAG_STUDY_DATE[0], TAG_STUDY_DATE[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getSeriesDate() {
        final byte[] value = getValue(TAG_SERIES_DATE[0], TAG_SERIES_DATE[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getAcquisitionDate() {
        final byte[] value = getValue(TAG_ACQUISITION_DATE[0], TAG_ACQUISITION_DATE[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getContentDate() {
        final byte[] value = getValue(TAG_CONTENT_DATE[0], TAG_CONTENT_DATE[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getStudyDescription() {
        final byte[] value = getValue(TAG_STUDY_DES[0], TAG_STUDY_DES[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getStudyTime() {
        final byte[] value = getValue(TAG_STUDY_TIME[0], TAG_STUDY_TIME[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public String getStudyUID() {
        final byte[] value = getValue(TAG_STUDY_UID[0], TAG_STUDY_UID[1]);

        if (value == null) {
            return null;
        }

        return new String(value);
    }



    public double getTE() {
        final byte[] value = getValue(TAG_TE[0], TAG_TE[1]);

        if (value == null) {
            return 0;
        }

        final String te = new String(value);

        double teD = 0;

        try {
            teD = Double.parseDouble(te.trim());
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return teD;
    }



    public int getTemporalPosition() {
        final byte[] value = getValue(TAG_TEMPORAL_POSITION[0], TAG_TEMPORAL_POSITION[1]);

        if (value == null) {
            return 0;
        }

        final String imageNum = new String(value);
        int imageNumI = 0;

        try {
            imageNumI = Integer.parseInt(imageNum.trim());
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return imageNumI;
    }



    public double getTR() {
        final byte[] value = getValue(TAG_TR[0], TAG_TR[1]);

        if (value == null) {
            return 0;
        }

        final String tr = new String(value);

        double trD = 0;

        try {
            trD = Double.parseDouble(tr.trim());
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return trD;
    }



    public String getTransferSyntaxID() {
        final byte[] value = getValue(TAG_TRANSFER_SYNTAX_ID[0], TAG_TRANSFER_SYNTAX_ID[1]);

        if (value == null) {
            return Constants.TRANSFER_SYNTAX_ID_EXPLICIT_LITTLE;
        }

        return (new String(value)).trim();
    }



    public String getSOPClass() {
        final byte[] value = getValue(TAG_SOP_CLASS_UID[0], TAG_SOP_CLASS_UID[1]);

        if (value != null) {
            return (new String(value)).trim();
        }

        return "";
    }



    public String getVolumeID() {
        final String des = getSeriesDescription();
        final String uid = getSeriesUID();
        final String num = getSeriesNumber();
        final String echo = getEchoNumber();
        final String orientation = getOrientation();
        final String rows = String.valueOf(getRows());
        final String cols = String.valueOf(getCols());
        final String imageType = getImageType();

        String idString = "";

        if (des != null) {
            idString += des;
        }

        if (uid != null) {
            idString += uid;
        }

        if (num != null) {
            idString += num;
        }

        if (echo != null) {
            idString += echo;
        }

        if (orientation != null) {
            idString += orientation;
        }

        if (rows != null) {
            idString += rows;
        }

        if (cols != null) {
            idString += cols;
        }

        if (imageType != null) {
            idString += imageType;
        }

        return idString;
    }



    public double getWindowCenter() {
        final byte[] value = getValue(TAG_WINDOW_CENTER[0], TAG_WINDOW_CENTER[1]);

        if (value == null) {
            return 0;
        }

        String center = new String(value);
        double centerD = 0;

        try {
            if (StringUtils.isNotBlank(center)) {
                center = ((new StringTokenizer(center, "\\")).nextToken());
                centerD = Double.parseDouble(center.trim());
            }
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        if (hasElscintDataScale()) {
            centerD *= getDataScaleSlope();
        }

        return centerD;
    }



    public double getWindowWidth() {
        final byte[] value = getValue(TAG_WINDOW_WIDTH[0], TAG_WINDOW_WIDTH[1]);

        if (value == null) {
            return 0;
        }

        String width = new String(value);
        double widthD = 0;

        try {
            if (StringUtils.isNotBlank(width)) {
                width = ((new StringTokenizer(width, "\\")).nextToken());
                widthD = Double.parseDouble(width.trim());
            }
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        if (hasElscintDataScale()) {
            widthD *= getDataScaleSlope();
        }

        return widthD;
    }



    public boolean isFileSet() {
        return isFileSet;
    }



    public boolean isLittleEndian() {
        return littleEndian;
    }



    public boolean isCompressed() {
        final String xferSyntax = getTransferSyntaxID();
        return ((xferSyntax.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_JPEG_PREFIX) != -1)
                || (xferSyntax.indexOf(Constants.TRANSFER_SYNTAX_ID_COMPRESSION_RLE) != -1));
    }



    public boolean isMosaic() {
        final String imageType = getImageType();
        final boolean labeledAsMosaic = ((imageType != null) && (imageType.toUpperCase().indexOf("MOSAIC") != -1));
        final int[] matSize = getAcquisitionMatrix();
        final boolean canReadAsMosaic = (matSize[0] > 0) && ((matSize[0] < getRows()) || (matSize[1] < getCols()));
        return labeledAsMosaic && canReadAsMosaic;
    }



    // originally from: http://public.kitware.com/pipermail/insight-users/2005-March/012246.html
    /**
     * Get a label describing the axial, coronal or sagittal plane from row and column unit vectors (direction cosines) as found in ImageOrientationPatient.
     *
     * Some degree of deviation from one of the standard orthogonal planes is allowed before deciding the plane is OBLIQUE.
     *
     * @param rowX
     * @param rowY
     * @param rowZ
     * @param colX
     * @param colY
     * @param colZ
     * @return the string describing the plane of orientation, AXIAL, CORONAL, SAGITTAL or OBLIQUE
     */
    public String makeImageOrientationLabelFromImageOrientationPatient() {
        final byte[] valueOrient = getValue(TAG_IMAGE_ORIENTATION[0], TAG_IMAGE_ORIENTATION[1]);
        String label = null;

        if (valueOrient == null) {
            return "UNKNOWN";
        }

        try {
            final String valueOrientString = new String(valueOrient);
            final StringTokenizer valueOrientTok = new StringTokenizer(valueOrientString, "\\");

            final double rowX = Double.parseDouble(valueOrientTok.nextToken());
            final double rowY = Double.parseDouble(valueOrientTok.nextToken());
            final double rowZ = Double.parseDouble(valueOrientTok.nextToken());
            final double colX = Double.parseDouble(valueOrientTok.nextToken());
            final double colY = Double.parseDouble(valueOrientTok.nextToken());
            final double colZ = Double.parseDouble(valueOrientTok.nextToken());

            final String rowAxis = getMajorAxisFromPatientRelativeDirectionCosine(rowX, rowY, rowZ);
            final String colAxis = getMajorAxisFromPatientRelativeDirectionCosine(colX, colY, colZ);
            if ((rowAxis != null) && (colAxis != null)) {
                if ((rowAxis.equals("R") || rowAxis.equals("L")) && (colAxis.equals("A") || colAxis.equals("P"))) {
                    label = "AXIAL";
                } else if ((colAxis.equals("R") || colAxis.equals("L")) && (rowAxis.equals("A") || rowAxis.equals("P"))) {
                    label = "AXIAL";
                } else if ((rowAxis.equals("R") || rowAxis.equals("L")) && (colAxis.equals("H") || colAxis.equals("F"))) {
                    label = "CORONAL";
                } else if ((colAxis.equals("R") || colAxis.equals("L")) && (rowAxis.equals("H") || rowAxis.equals("F"))) {
                    label = "CORONAL";
                } else if ((rowAxis.equals("A") || rowAxis.equals("P")) && (colAxis.equals("H") || colAxis.equals("F"))) {
                    label = "SAGITTAL";
                } else if ((colAxis.equals("A") || colAxis.equals("P")) && (rowAxis.equals("H") || rowAxis.equals("F"))) {
                    label = "SAGITTAL";
                }
            } else {
                label = "OBLIQUE";
            }
        } catch (final NumberFormatException ex) {
            AppLogger.info(ex);
        }

        return label;
    }



    public void setFileSet(final boolean bool) {
        isFileSet = bool;
    }



    public void setImageFile(final URI aFile) {
        imageFile = aFile;
    }



    public void setOrderNumber(final int val) {
        orderNumber = val;
    }



    public void setSimpleToString(final boolean bool) {
        simpleToString = bool;
    }



    @Override
    public String toString() {
        final StringBuffer sb = new StringBuffer();

        if (simpleToString) {
            sb.append(getSeriesDescription());
        } else {
            if (elements != null) {
                if (imageFile != null) {
                    sb.append("Filename: " + getImageFilename() + "\n");
                }

                for (final Item element : elements) {
                    final String currentString = element.toString();

                    if (currentString != null) {
                        sb.append(currentString);
                        sb.append('\n');
                    }
                }
            }
        }

        return sb.toString();
    }



    private void clearPrivateTagData(final List<Item> someItems) {
        if (someItems != null) {
            for (final Item someItem : someItems) {
                if (someItem.isSublist()) {
                    clearPrivateTagData(someItem.getSublist());
                } else {
                    someItem.setPrivateTags(null);
                }
            }
        }
    }



    /**
     * Get a label describing the major axis from a unit vector (direction cosine) as found in ImageOrientationPatient.
     *
     * Some degree of deviation from one of the standard orthogonal axes is allowed before deciding no major axis applies and returning null.
     *
     * @param x
     * @param y
     * @param z
     * @return the string describing the orientation of the vector, or null if oblique
     */
    // originally from: http://public.kitware.com/pipermail/insight-users/2005-March/012246.html
    private String getMajorAxisFromPatientRelativeDirectionCosine(final double x, final double y, final double z) {
        String axis = null;

        final String orientationX = x < 0 ? "R" : "L";
        final String orientationY = y < 0 ? "A" : "P";
        final String orientationZ = z < 0 ? "F" : "H";

        final double absX = Math.abs(x);
        final double absY = Math.abs(y);
        final double absZ = Math.abs(z);

        // The tests here really don't need to check the other dimensions,
        // just the threshold, since the sum of the squares should be == 1.0
        // but just in case ...

        if ((absX > OBLIQUITY_THRESHOLD_COSINE_VALUE) && (absX > absY) && (absX > absZ)) {
            axis = orientationX;
        } else if ((absY > OBLIQUITY_THRESHOLD_COSINE_VALUE) && (absY > absX) && (absY > absZ)) {
            axis = orientationY;
        } else if ((absZ > OBLIQUITY_THRESHOLD_COSINE_VALUE) && (absZ > absX) && (absZ > absY)) {
            axis = orientationZ;
        }

        return axis;
    }



    private String getPrivateTags() {
        return getPrivateTags(elements, true);
    }



    protected Item getItem(final long group, final long elem) {
        return getItem(group, elem, -1, elements);
    }



    protected Item getItem(final long group, final long elem, final long offset) {
        return getItem(group, elem, offset, elements);
    }



    protected byte[] getValue(final long group, final long elem) { // NOPMD
        final Item item = getItem(group, elem, -1, elements);
        if (item != null) {
            return item.getValue();
        }

        return null;
    }



    public void makeOverlayData() {
        int overlayIndex = 0;

        for (int ctr = TAG_OVERLAY_GROUP_MIN; ctr <= TAG_OVERLAY_GROUP_MAX; ctr += 2) {
            final byte[] value = getValue(ctr, TAG_OVERLAY_BITS_ALLOCATED[1]);

            if (CollectionUtilities.isNotEmpty(value)) {
                final int cols = getCols();
                final int rows = getRows();
                final int ovlyCols = getOverlayCols(ctr);
                final int ovlyRows = getOverlayRows(ctr);
                final int ovlyBitsAlloc = getOverlayBitsAllocated(ctr);
                final int ovlyBitPos = getOverlayBitsPosition(ctr);
                final int size = cols * rows;
                final int bitsStored = getBitsStored();
                final int bitsAlloc = getBitsAllocated();

                if ((cols != 0) && (cols == ovlyCols) && (rows != 0) && (rows == ovlyRows)) {
                    if (overlayData == null) {
                        overlayData = new short[cols * rows];
                    }

                    final byte[] overlayDataNative = getOverlayData(ctr);

                    if (CollectionUtilities.isNotEmpty(overlayDataNative) && (ovlyBitPos == 0) && (ovlyBitsAlloc == 1)) {
                        final BitSet bits = BitSet.valueOf(overlayDataNative);
                        for (int ctrB = 0; ctrB < size; ctrB++) {
                            if (bits.get(ctrB)) {
                                overlayData[ctrB] |= (short) ((1 << overlayIndex) & 0xFFFF);
                            }
                        }

                        overlayIndex++;
                    } else if (bitsStored < bitsAlloc) {
                        setNeedsOverlayData(true);
                    }
                }
            }
        }
    }



    public void addInPlaneOverlayData(final short mask, final int index) {
        overlayData[index] |= mask;
    }



    public short[] getOverlayMask() {
        return overlayData;
    }



    public int getOverlayCols(final int group) {
        final byte[] value = getValue(group, TAG_OVERLAY_COLS[1]);

        if (value == null) {
            return 0;
        }

        return getShort(value, 0);
    }



    public int getOverlayRows(final int group) {
        final byte[] value = getValue(group, TAG_OVERLAY_ROWS[1]);

        if (value == null) {
            return 0;
        }

        return getShort(value, 0);
    }



    public int getOverlayBitsAllocated(final int group) {
        final byte[] value = getValue(group, TAG_OVERLAY_BITS_ALLOCATED[1]);

        if (value == null) {
            return 0;
        }

        return getShort(value, 0);
    }



    public int getOverlayBitsPosition(final int group) {
        final byte[] value = getValue(group, TAG_OVERLAY_BIT_POSITION[1]);

        if (value == null) {
            return 0;
        }

        return getShort(value, 0);
    }



    public byte[] getOverlayData(final int group) {
        return getValue(group, TAG_OVERLAY_DATA[1]);
    }



    public boolean isNeedsOverlayData() {
        return needsOverlayData;
    }



    public void setNeedsOverlayData(final boolean needsOverlayData) {
        this.needsOverlayData = needsOverlayData;
    }



    public List<Item> getAllItems() {
        return elements;
    }



    public Map<Long, String> generateDicomItems() {
        final Map<Long, String> map = new HashMap<Long, String>();
        final SimpleDateFormat dateTimeFormat = new SimpleDateFormat("yyyyMMddHHmmss");

        for (final Item item : elements) {
            if (item.getVR().equals("DT")) { // for date-time, these need to be converted to epoch time
                try {
                    map.put(Tag.makeMappableDicomId(item.getGroup(), item.getElement()),
                            String.valueOf(dateTimeFormat.parse(item.getValueAsString()).getTime()));
                } catch (final ParseException ex) {
                    AppLogger.warn(ex);
                }

                // other time formats can remain in string form
            } else {
                map.put(Tag.makeMappableDicomId(item.getGroup(), item.getElement()), item.getValueAsString());
            }
        }

        return map;
    }
}
