
package edu.uthscsa.ric.utilities;

public final class ByteUtilities {

	private ByteUtilities() {}



	// http://stackoverflow.com/questions/5513152/easy-way-to-concatenate-two-byte-arrays
	public static byte[] concatenateByteArrays(final byte[] a, final byte[] b) {
		final byte[] c = new byte[a.length + b.length];
		System.arraycopy(a, 0, c, 0, a.length);
		System.arraycopy(b, 0, c, a.length, b.length);
		return c;
	}



	public static byte[] getBytes(final double d) {
		return doubleToBytes(d);
	}



	public static byte[] getBytes(final float f) {
		return floatToBytes(f);
	}



	// used to convert a float[] into a byte[] (and then to hex) in AlePanel
	public static byte[] getBytes(final float[] f) {
		final byte[] b = new byte[4 * f.length];
		byte[] fl;

		// for each float
		for (int i = 0; i < f.length; i++) {
			// get the bytes
			fl = floatToBytes(f[i]);

			// put them in a byte array
			for (int j = 0; j < 4; j++) {
				b[(i * 4) + j] = fl[j];
			}
		}

		return b;
	}



	public static byte[] getBytes(final int i) {
		return intToBytes(i);
	}



	public static byte[] getBytes(final long l) {
		return longToBytes(l);
	}



	public static byte[] getBytes(final short s) { //NOPMD
		return shortToBytes(s);
	}



	public static char getChar(final byte b) {
		return byteToChar(b);
	}



	public static double getDouble(final byte[] b, final int i) {
		return bytesToDouble(b, i);
	}



	public static double getDouble2(final byte[] b) {
		final byte buf[] = { 0, 0, 0, 0, 0, 0, 0, 0 };
		buf[7] = b[3];
		buf[6] = b[2];
		buf[5] = b[1];
		buf[4] = b[0];
		buf[3] = b[7];
		buf[2] = b[6];
		buf[1] = b[5];
		buf[0] = b[4];

		int offset = 0;
		final long val = ((((long) buf[offset++]) & 0xff)) | ((((long) buf[offset++]) & 0xff) << 8) | ((((long) buf[offset++]) & 0xff) << 16)
				| ((((long) buf[offset++]) & 0xff) << 24) | ((((long) buf[offset++]) & 0xff) << 32) | ((((long) buf[offset++]) & 0xff) << 40)
				| ((((long) buf[offset++]) & 0xff) << 48) | ((((long) buf[offset++]) & 0xff) << 56);

		return Double.longBitsToDouble(val);
	}



	//public static long getLong(byte[] b, int i) { return bytesToLong(b, i); }
	public static float getFloat(final byte[] b, final int i) {
		return bytesToFloat(b, i);
	}



	public static int getInt(final byte[] b, final int i) {
		return bytesToInt(b, i);
	}



	public static long getLong(final byte[] b, final int i) {
		return bytesToLong(b, i);
	}



	public static short getShort(final byte[] b, final int i) { //NOPMD
		return bytesToShort(b, i);
	}



	public static String getString(final byte[] b, final int i, final int size) {
		return bytesToString(b, i, size);
	}



	public static byte[] intToBytes(final int a, final byte[] b) {
		b[0] = (byte) ((a >> 24) & 0xFF);
		b[1] = (byte) ((a >> 16) & 0xFF);
		b[2] = (byte) ((a >> 8) & 0xFF);
		b[3] = (byte) ((a >> 0) & 0xFF);

		return b;
	}



	public static byte[] longToBytes(final long a, final byte[] b) {
		b[0] = (byte) ((a >> 56) & 0xFF);
		b[1] = (byte) ((a >> 48) & 0xFF);
		b[2] = (byte) ((a >> 40) & 0xFF);
		b[3] = (byte) ((a >> 32) & 0xFF);
		b[4] = (byte) ((a >> 24) & 0xFF);
		b[5] = (byte) ((a >> 16) & 0xFF);
		b[6] = (byte) ((a >> 8) & 0xFF);
		b[7] = (byte) ((a >> 0) & 0xFF);

		return b;
	}



	public static byte[] shortToBytes(final short a, final byte[] b) { //NOPMD
		b[0] = (byte) ((a >> 8) & 0xFF);
		b[1] = (byte) ((a >> 0) & 0xFF);
		return b;
	}



	public static byte[] swap(final byte[] b) {
		// no point in swapping these:
		if ((b == null) || (b.length < 2)) {
			return b;
		}

		final int size = b.length;
		try {
			final byte[] temp = new byte[size];
			for (int i = 0; i < size; i++) {
				temp[i] = b[(size - 1) - i];
			}
			return temp;
		} catch (final Exception ex) {
			AppLogger.warn(ex);
			return b;
		}
	}



	//	public static String getHex(float f) { return "0x"+getHex(floatToBytes(f), 0, 4); }
	//	public static String getHex(double d) { return "0x"+getHex(doubleToBytes(d), 0, 8); }

	/**
	 * Byte swap a single double value.
	 * 
	 * @param value Value to byte swap.
	 * @return Byte swapped representation.
	 */
	public static double swap(final double value) {
		// long longValue = Double.doubleToLongBits (value);
		long longValue = Double.doubleToRawLongBits(value); // was necessary to use Raw method
		longValue = swap(longValue);
		return Double.longBitsToDouble(longValue);
	}



	/**
	 * Byte swap an array of doubles. The result of the swapping is put back into the specified array.
	 * 
	 * @param array Array of values to swap
	 */
	public static void swap(final double[] array) {
		for (int i = 0; i < array.length; i++) {
			array[i] = swap(array[i]);
		}
	}



	/**
	 * Byte swap a single float value.
	 * 
	 * @param value Value to byte swap.
	 * @return Byte swapped representation.
	 */
	public static float swap(final float value) {
		// int intValue = Float.floatToIntBits (value);
		int intValue = Float.floatToRawIntBits(value); // was necessary to use Raw method
		intValue = swap(intValue);
		return Float.intBitsToFloat(intValue);
	}



	/**
	 * Byte swap an array of floats. The result of the swapping is put back into the specified array.
	 * 
	 * @param array Array of values to swap
	 */
	public static void swap(final float[] array) {
		for (int i = 0; i < array.length; i++) {
			array[i] = swap(array[i]);
		}
	}



	/**
	 * Byte swap a single int value.
	 * 
	 * @param value Value to byte swap.
	 * @return Byte swapped representation.
	 */
	public static int swap(final int value) {
		final int b1 = (value >> 0) & 0xff;
		final int b2 = (value >> 8) & 0xff;
		final int b3 = (value >> 16) & 0xff;
		final int b4 = (value >> 24) & 0xff;

		return (b1 << 24) | (b2 << 16) | (b3 << 8) | (b4 << 0);
	}



	/**
	 * Byte swap an array of ints. The result of the swapping is put back into the specified array.
	 * 
	 * @param array Array of values to swap
	 */
	public static void swap(final int[] array) {
		for (int i = 0; i < array.length; i++) {
			array[i] = swap(array[i]);
		}
	}



	/**
	 * Byte swap a single long value.
	 * 
	 * @param value Value to byte swap.
	 * @return Byte swapped representation.
	 */
	public static long swap(final long value) {
		final long b1 = (value >> 0) & 0xff;
		final long b2 = (value >> 8) & 0xff;
		final long b3 = (value >> 16) & 0xff;
		final long b4 = (value >> 24) & 0xff;
		final long b5 = (value >> 32) & 0xff;
		final long b6 = (value >> 40) & 0xff;
		final long b7 = (value >> 48) & 0xff;
		final long b8 = (value >> 56) & 0xff;

		return (b1 << 56) | (b2 << 48) | (b3 << 40) | (b4 << 32) | (b5 << 24) | (b6 << 16) | (b7 << 8) | (b8 << 0);
	}



	/**
	 * Byte swap an array of longs. The result of the swapping is put back into the specified array.
	 * 
	 * @param array Array of values to swap
	 */
	public static void swap(final long[] array) {
		for (int i = 0; i < array.length; i++) {
			array[i] = swap(array[i]);
		}
	}



	/**
	 * Byte swap a single short value.
	 * 
	 * @param value Value to byte swap.
	 * @return Byte swapped representation.
	 */
	public static short swap(final short value) { //NOPMD
		final int b1 = value & 0xff;
		final int b2 = (value >> 8) & 0xff;

		return (short) ((b1 << 8) | (b2 << 0)); //NOPMD
	}



	/**
	 * Byte swap an array of shorts. The result of the swapping is put back into the specified array.
	 * 
	 * @param array Array of values to swap
	 */
	public static void swap(final short[] array) { //NOPMD
		for (int i = 0; i < array.length; i++) {
			array[i] = swap(array[i]);
		}
	}



	public static double swapDouble(final byte[] b, final int i) {
		return Double.longBitsToDouble(swapLong(b, i));
	}



	//public static long swapLong(byte[] b, int i) { return Long.reverseBytes(bytesToLong(b, i)); }
	public static float swapFloat(final byte[] b, final int i) {
		return Float.intBitsToFloat(swapInt(b, i));
	}



	public static int swapInt(final byte[] b, final int i) {
		return Integer.reverseBytes(bytesToInt(b, i));
	}



	public static long swapLong(final byte[] b, final int i) {
		return Long.reverseBytes(bytesToLong(b, i));
	}



	public static short swapShort(final byte[] b, final int i) { //NOPMD
		return Short.reverseBytes(bytesToShort(b, i));
	}



	private static double bytesToDouble(final byte b[], final int i) {
		// convert eight contiguous bytes to a double
		final long a = bytesToLong(b, i);
		return Double.longBitsToDouble(a);
	}



	private static float bytesToFloat(final byte b[], final int i) { //, int endian) {
		// convert four contiguous bytes to a float
		final int a = bytesToInt(b, i); //, endian);
		return Float.intBitsToFloat(a);
	}



	private static int bytesToInt(final byte[] b, final int i) {//, int endian) {
		// convert four contiguous bytes to a long
		if (b.length < (i + 4)) {
			AppLogger.warn("ByteConverter.getInt(): Reading past bounds of byte array.");
			return Integer.MIN_VALUE;
			//	System.exit(0);
		}

		final int val =

		//	if (endian == BIG_ENDIAN) val =
		((((b[i + 3]) & 0xff) << 0) | (((b[i + 2]) & 0xff) << 8) | (((b[i + 1]) & 0xff) << 16) | (((b[i + 0]) & 0xff) << 24));
		//	else val =
		//	    ( ( ( (int) b[i + 0]) & 0xff) <<  0) |
		//		( ( ( (int) b[i + 1]) & 0xff) <<  8) |
		//		( ( ( (int) b[i + 2]) & 0xff) << 16) |
		//		( ( ( (int) b[i + 3]) & 0xff) << 24);

		return val;
	}



	private static long bytesToLong(final byte[] b, final int i) {
		// convert eight contiguous bytes to a long
		if (b.length < (i + 8)) {
			AppLogger.warn("ByteConverter.getLong(): Reading past bounds of byte array.");
			return Long.MIN_VALUE;
		}

		final long val = ((((long) b[i + 7]) & 0xff) << 0) | ((((long) b[i + 6]) & 0xff) << 8) | ((((long) b[i + 5]) & 0xff) << 16)
				| ((((long) b[i + 4]) & 0xff) << 24) | ((((long) b[i + 3]) & 0xff) << 32) | ((((long) b[i + 2]) & 0xff) << 40)
				| ((((long) b[i + 1]) & 0xff) << 48) | ((((long) b[i + 0]) & 0xff) << 56);

		return val;
	}



	private static short bytesToShort(final byte[] b, final int i) { // NOPMD
		// convert two contiguous bytes into an int
		if (b.length < (i + 2)) {
			AppLogger.warn("ByteConverter.getShort(): Reading past bounds of byte array.");
			return Short.MIN_VALUE;
		}

		final short val = (short) ((((b[i + 1]) & 0xff) << 0) | (((b[i + 0]) & 0xff) << 8)); //NOPMD
		//if (endian == BIG_ENDIAN) val
		//else val = (short) (
		//	( ( ( (int) b[i + 0]) & 0xff) <<  0) |
		//	( ( ( (int) b[i + 1]) & 0xff) <<  8) );

		return val;
	}



	// used in CTI PET archive
	private static String bytesToString(final byte[] b, final int i, final int size) {
		// convert <size> contiguous bytes to a string
		if (b.length < (i + size)) {
			AppLogger.warn("ByteConverter.getString(): Reading past bounds of byte array.");
			return null;
		}

		final StringBuffer sb = new StringBuffer();
		char c;
		for (int j = 0; j < size; j++) {
			c = byteToChar(b[i + j]);
			if (c == '\u0000') {
				break; // the null character terminates strings, no more data to add
			} else {
				sb.append(c);
			}
		}

		return sb.toString();
	}



	// used in CTI PET archive
	private static char byteToChar(final byte b) {
		// converts a byte to a character
		switch (b) { // www.asciitable.com
			case 0x00:
				return '\u0000'; // NUL	null
			case 0x01:
				return '\u0001'; // SOH	start of heading
			case 0x02:
				return '\u0002'; // STX	start of text
			case 0x03:
				return '\u0003'; // ETX	end of text
			case 0x04:
				return '\u0004'; // EOT	end of transmission
			case 0x05:
				return '\u0005'; // ENQ	enquiry
			case 0x06:
				return '\u0006'; // ACK	acknowledge
			case 0x07:
				return '\u0007'; // BEL	bell
			case 0x08:
				return '\u0008'; // BS	backspace
			case 0x09:
				return '\u0009'; // TAB	horizontal tab
			case 0x0A:
				return '\n'; // LF	new line, line feed
			case 0x0B:
				return '\u000B'; // VT	vertical tab
			case 0x0C:
				return '\u000C'; // FF	new page, form feed
			case 0x0D:
				return '\r'; // CR	carriage return
			case 0x0E:
				return '\u000E'; // SO	shift out
			case 0x0F:
				return '\u000F'; // SI	shift in
			case 0x10:
				return '\u0010'; // DLE	data link escape
			case 0x11:
				return '\u0011'; // DC1	device control 1
			case 0x12:
				return '\u0012'; // DC2	device control 2
			case 0x13:
				return '\u0013'; // DC3	device control 3
			case 0x14:
				return '\u0014'; // DC4	device control 4
			case 0x15:
				return '\u0015'; // NAK	negative acknowledge
			case 0x16:
				return '\u0016'; // SYN	synchronous idle
			case 0x17:
				return '\u0017'; // ETB	end of transmission, block
			case 0x18:
				return '\u0018'; // CAN	cancel
			case 0x19:
				return '\u0019'; // EM	end of medium
			case 0x1A:
				return '\u001A'; // SUB	substitute
			case 0x1B:
				return '\u001B'; // ESC	escape
			case 0x1C:
				return '\u001C'; // FS	file separator
			case 0x1D:
				return '\u001D'; // GS	group separator
			case 0x1E:
				return '\u001E'; // RS	record separator
			case 0x1F:
				return '\u001F'; // US	unit separator
			case 0x20:
				return '\u0020'; // ' '
			case 0x21:
				return '\u0021'; // '!'
			case 0x22:
				return '\u0022'; // '"'
			case 0x23:
				return '\u0023'; // '#'
			case 0x24:
				return '\u0024'; // '$'
			case 0x25:
				return '\u0025'; // '%'
			case 0x26:
				return '\u0026'; // '&'
			case 0x27:
				return '\''; // '''
			case 0x28:
				return '\u0028'; // '('
			case 0x29:
				return '\u0029'; // ')'
			case 0x2A:
				return '\u002A'; // '*'
			case 0x2B:
				return '\u002B'; // '+'
			case 0x2C:
				return '\u002C'; // ','
			case 0x2D:
				return '\u002D'; // '-'
			case 0x2E:
				return '\u002E'; // '.'
			case 0x2F:
				return '\u002F'; // '/'
			case 0x30:
				return '\u0030'; // '0'
			case 0x31:
				return '\u0031'; // '1'
			case 0x32:
				return '\u0032'; // '2'
			case 0x33:
				return '\u0033'; // '3'
			case 0x34:
				return '\u0034'; // '4'
			case 0x35:
				return '\u0035'; // '5'
			case 0x36:
				return '\u0036'; // '6'
			case 0x37:
				return '\u0037'; // '7'
			case 0x38:
				return '\u0038'; // '8'
			case 0x39:
				return '\u0039'; // '9'
			case 0x3A:
				return '\u003A'; // ':'
			case 0x3B:
				return '\u003B'; // ';'
			case 0x3C:
				return '\u003C'; // '<'
			case 0x3D:
				return '\u003D'; // '='
			case 0x3E:
				return '\u003E'; // '>'
			case 0x3F:
				return '\u003F'; // '?'
			case 0x40:
				return '\u0040'; // '@'
			case 0x41:
				return '\u0041'; // 'A'
			case 0x42:
				return '\u0042'; // 'B'
			case 0x43:
				return '\u0043'; // 'C'
			case 0x44:
				return '\u0044'; // 'D'
			case 0x45:
				return '\u0045'; // 'E'
			case 0x46:
				return '\u0046'; // 'F'
			case 0x47:
				return '\u0047'; // 'G'
			case 0x48:
				return '\u0048'; // 'H'
			case 0x49:
				return '\u0049'; // 'I'
			case 0x4A:
				return '\u004A'; // 'J'
			case 0x4B:
				return '\u004B'; // 'K'
			case 0x4C:
				return '\u004C'; // 'L'
			case 0x4D:
				return '\u004D'; // 'M'
			case 0x4E:
				return '\u004E'; // 'N'
			case 0x4F:
				return '\u004F'; // 'O'
			case 0x50:
				return '\u0050'; // 'P'
			case 0x51:
				return '\u0051'; // 'Q'
			case 0x52:
				return '\u0052'; // 'R'
			case 0x53:
				return '\u0053'; // 'S'
			case 0x54:
				return '\u0054'; // 'T'
			case 0x55:
				return '\u0055'; // 'U'
			case 0x56:
				return '\u0056'; // 'V'
			case 0x57:
				return '\u0057'; // 'W'
			case 0x58:
				return '\u0058'; // 'X'
			case 0x59:
				return '\u0059'; // 'Y'
			case 0x5A:
				return '\u005A'; // 'Z'
			case 0x5B:
				return '\u005B'; // '['
			case 0x5C:
				return '\\'; // '\'
			case 0x5D:
				return '\u005D'; // ']'
			case 0x5E:
				return '\u005E'; // '^'
			case 0x5F:
				return '\u005F'; // '_'
			case 0x60:
				return '\u0060'; // '`'
			case 0x61:
				return '\u0061'; // 'a'
			case 0x62:
				return '\u0062'; // 'b'
			case 0x63:
				return '\u0063'; // 'c'
			case 0x64:
				return '\u0064'; // 'd'
			case 0x65:
				return '\u0065'; // 'e'
			case 0x66:
				return '\u0066'; // 'f'
			case 0x67:
				return '\u0067'; // 'g'
			case 0x68:
				return '\u0068'; // 'h'
			case 0x69:
				return '\u0069'; // 'i'
			case 0x6A:
				return '\u006A'; // 'j'
			case 0x6B:
				return '\u006B'; // 'k'
			case 0x6C:
				return '\u006C'; // 'l'
			case 0x6D:
				return '\u006D'; // 'm'
			case 0x6E:
				return '\u006E'; // 'n'
			case 0x6F:
				return '\u006F'; // 'o'
			case 0x70:
				return '\u0070'; // 'p'
			case 0x71:
				return '\u0071'; // 'q'
			case 0x72:
				return '\u0072'; // 'r'
			case 0x73:
				return '\u0073'; // 's'
			case 0x74:
				return '\u0074'; // 't'
			case 0x75:
				return '\u0075'; // 'u'
			case 0x76:
				return '\u0076'; // 'v'
			case 0x77:
				return '\u0077'; // 'w'
			case 0x78:
				return '\u0078'; // 'x'
			case 0x79:
				return '\u0079'; // 'y'
			case 0x7A:
				return '\u007A'; // 'z'
			case 0x7B:
				return '\u007B'; // '{'
			case 0x7C:
				return '\u007C'; // '|'
			case 0x7D:
				return '\u007D'; // '}'
			case 0x7E:
				return '\u007E'; // '~'
			case 0x7F:
				return '\u007F'; // DEL delete

			default:
				return '\0';
		}
	}



	// double
	private static byte[] doubleToBytes(final double f) {
		final long a = Double.doubleToLongBits(f);
		return longToBytes(a);
	}



	// float
	private static byte[] floatToBytes(final float f) {
		final int a = Float.floatToIntBits(f);
		return intToBytes(a);
	}



	// int
	private static byte[] intToBytes(final int a) {
		final byte[] b = new byte[4];
		return intToBytes(a, b);
	}



	// long
	private static byte[] longToBytes(final long a) {
		final byte[] b = new byte[8];
		return longToBytes(a, b);
	}



	// short
	private static byte[] shortToBytes(final short a) { //NOPMD
		final byte[] b = new byte[2];
		return shortToBytes(a, b);
	}



	// http://stackoverflow.com/questions/1507780/searching-for-a-sequence-of-bytes-in-a-binary-file-with-java
	/**
	 * Knuth-Morris-Pratt Algorithm for Pattern Matching
	 */
	public static int indexOf(final byte[] data, final byte[] pattern) {
		if (data.length == 0) {
			return -1;
		}

		int j = 0;
		final int[] failure = computeFailure(pattern);

		for (int i = 0; i < data.length; i++) {
			while ((j > 0) && (pattern[j] != data[i])) {
				j = failure[j - 1];
			}
			if (pattern[j] == data[i]) {
				j++;
			}
			if (j == pattern.length) {
				return (i - pattern.length) + 1;
			}
		}
		return -1;
	}



	/**
	 * Computes the failure function using a boot-strapping process, where the pattern is matched against itself.
	 */
	private static int[] computeFailure(final byte[] pattern) {
		final int[] failure = new int[pattern.length];

		int j = 0;
		for (int i = 1; i < pattern.length; i++) {
			while ((j > 0) && (pattern[j] != pattern[i])) {
				j = failure[j - 1];
			}
			if (pattern[j] == pattern[i]) {
				j++;
			}
			failure[i] = j;
		}

		return failure;
	}
}
