
package edu.uthscsa.ric.utilities;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.util.Collections;
import java.util.Vector;
import java.util.zip.GZIPInputStream;

import net.lingala.zip4j.core.ZipFile;
import net.lingala.zip4j.exception.ZipException;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;


public final class FileUtilities {

	private FileUtilities() {}



	public static boolean canRead(final URI imageFile) {
		if (imageFile != null) {
			if (uriIsFile(imageFile)) {
				if (imageFile.isAbsolute()) {
					return (new File(imageFile)).canRead();
				} else {
					return (new File(imageFile.toString())).canRead();
				}
			} else {
				return true;
			}
		} else {
			return false;
		}
	}



	public static boolean canWrite(final URI imageFile) {
		if (imageFile != null) {
			if (uriIsFile(imageFile)) {
				if (imageFile.isAbsolute()) {
					return (new File(imageFile)).canWrite();
				} else {
					return (new File(imageFile.toString())).canWrite();
				}
			} else {
				return false;
			}
		} else {
			return false;
		}
	}



	public static boolean containsFile(final File parentDir, final File fileVal) {
		File file = fileVal;

		if (file != null) {
			while (file.getParentFile() != null) {
				final File parent = file.getParentFile();
				if (parent.equals(parentDir)) {
					return true;
				}

				file = parent;
			}
		}

		return false;
	}



	public static boolean createNewFile(final File file) {
		boolean success = false;
		try {
			success = file.createNewFile();
		} catch (final IOException ex) {
			AppLogger.error(ex);
		}

		if (!success) {
			AppLogger.warn("File could not be created: " + file);
		}

		return success;
	}



	public static boolean deleteFolder(final String path) {
		final File src = new File(path);
		boolean deleted = false;

		if (src.isDirectory()) {
			final String list[] = src.list();

			for (final String element : list) {
				deleteFolder(src.getAbsolutePath() + "/" + element);
			}

			deleted = src.delete();
			src.deleteOnExit();
		} else {
			deleted = src.delete();
			src.deleteOnExit();
		}

		return deleted;
	}



	public static boolean exists(final URI imageFile) {
		if (imageFile != null) {
			if (uriIsFile(imageFile)) {
				if (imageFile.isAbsolute()) {
					return (new File(imageFile)).exists();
				} else {
					return (new File(imageFile.toString())).exists();
				}
			} else {
				String url = null;
				try {
					url = imageFile.toURL().toString();
					return URLUtilities.urlExists(url, 4000);
				} catch (final IOException ex) {
					AppLogger.error(ex);
					return false;
				}
			}
		} else {
			return false;
		}
	}



	public static boolean extensionMatches(final String ext, final String[] extensions) {
		for (final String extension : extensions) {
			if ((ext != null) && extension.equals(ext)) {
				return true;
			}
		}

		return false;
	}



	public static String getExtension(final File file) {
		final String name = file.getName();
		final int index = name.lastIndexOf('.');
		if (index != -1) {
			return name.substring(index + 1);
		}

		return null;
	}



	public static String getFullExtension(final File file) {
		final String name = file.getName();
		final int index = name.indexOf('.');
		if (index != -1) {
			return name.substring(index + 1);
		}

		return null;
	}



	public static BufferedInputStream getInputStream(final URI uri, final boolean isCompressed) throws IOException {
		BufferedInputStream input = null;

		try {
			if (uriIsFile(uri)) {
				if (isCompressed) {
					input = new BufferedInputStream(new GZIPInputStream(new FileInputStream(new File(uri))));
				} else {
					input = new BufferedInputStream(new FileInputStream(new File(uri)));
				}
			} else {
				final URL url = uri.toURL();
				final URLConnection connection = url.openConnection();
				if (isCompressed) {
					input = new BufferedInputStream(new GZIPInputStream(connection.getInputStream()));
				} else {
					input = new BufferedInputStream(connection.getInputStream());
				}
			}
		} catch (final IOException ex) {
			if (ex.getMessage().indexOf("GZIP format") != -1) {
				return getInputStream(uri, !isCompressed);
			}
		}

		return input;
	}



	public static int getLength(final URI uri) throws IOException {
		try {
			if (uriIsFile(uri)) {
				final File file = new File(uri);
				return (int) file.length();
			} else {
				final URL url = uri.toURL();
				final URLConnection connection = url.openConnection();
				return connection.getContentLength();
			}
		} catch (final IOException ex) {
			AppLogger.error(ex);
		}

		return -1;
	}



	public static String getName(final URI imageFile) {
		if (imageFile != null) {
			if (uriIsFile(imageFile)) {
				if (imageFile.isAbsolute()) {
					return (new File(imageFile)).getName();
				} else {
					return (new File(imageFile.toString())).getName();
				}
			} else {
				String path = imageFile.getPath();
				path = path.substring(path.lastIndexOf('/') + 1);

				try {
					path = URLDecoder.decode(path, "UTF-8");
				} catch (final UnsupportedEncodingException ex) {
					AppLogger.warn(ex);
				}

				return path;
			}
		} else {
			return "";
		}
	}



	public static String getParent(final URI imageFile) {
		if (imageFile != null) {
			if (uriIsFile(imageFile)) {
				if (imageFile.isAbsolute()) {
					return (new File(imageFile)).getParent();
				} else {
					return (new File(imageFile.toString())).getParent();
				}
			} else {
				String path = imageFile.toString();
				path = path.substring(0, path.lastIndexOf('/'));
				return path;
			}
		} else {
			return "";
		}
	}



	public static String[] getPathComponents(final File fileVal) {
		final Vector<String> comps = new Vector<String>();
		File file = fileVal;

		if (file != null) {
			while (file.getParentFile() != null) {
				final String name = file.getName();
				final File parent = file.getParentFile();

				comps.add(name);

				file = parent;
			}
		}

		Collections.reverse(comps);

		return comps.toArray(new String[comps.size()]);
	}



	public static BufferedReader getReader(final URI uri) throws FileNotFoundException, IOException {
		return getReader(uri, false);
	}



	public static BufferedReader getReader(final URI uri, final boolean isCompressed) throws FileNotFoundException, IOException {
		BufferedReader reader = null;

		if (uriIsFile(uri)) {
			if (isCompressed) {
				reader = new BufferedReader(new InputStreamReader(new GZIPInputStream(new FileInputStream(new File(uri)))));
			} else {
				reader = new BufferedReader(new FileReader(new File(uri)));
			}
		} else {
			final URL url = uri.toURL();
			final URLConnection connection = url.openConnection();

			if (isCompressed) {
				reader = new BufferedReader(new InputStreamReader(new GZIPInputStream(connection.getInputStream())));
			} else {
				reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));
			}
		}

		return reader;
	}



	// http://www.abeel.be/content/determine-uncompressed-size-gzip-file
	public static int getUncompressedFileSize(final File compressedFile) {
		int size = 0;
		RandomAccessFile raf = null;
		try {
			raf = new RandomAccessFile(compressedFile, "r");
			raf.seek(raf.length() - 4);
			final int b4 = raf.read();
			final int b3 = raf.read();
			final int b2 = raf.read();
			final int b1 = raf.read();
			size = (b1 << 24) | ((b2 << 16) + (b3 << 8) + b4);
		} catch (final IOException ex) {
			AppLogger.error(ex);
		} finally {
			try {
				if (raf != null) {
					raf.close();
				}
			} catch (final IOException ex) {
				AppLogger.warn(ex);
			}
		}

		return size;
	}



	public static boolean mkdir(final File file) {
		final boolean success = file.mkdir() || file.exists();
		if (!success) {
			AppLogger.warn("Directory could not be made: " + file);
		}

		return success;
	}



	public static boolean delete(final File file) {
		final boolean success = file.delete();

		if (!success) {
			AppLogger.warn("File could not be deleted: " + file);
		}

		return success;
	}



	public static boolean mkdirs(final File file) {
		final boolean success = file.mkdirs() || file.exists();
		if (!success) {
			AppLogger.warn("Directory could not be made: " + file);
		}

		return success;
	}



	public static boolean renameTo(final File from, final File to) {
		final boolean success = from.renameTo(to);
		if (!success) {
			AppLogger.warn("File could not be renamed: " + from);
		}

		return success;
	}



	public static void unzipFileToFolder(final File file, final File folder) {
		unzipFileToFolder(file, folder, null);
	}



	public static void unzipFileToFolder(final File file, final File folder, final String password) {
		try {
			final ZipFile zipFile = new ZipFile(file);
			if (zipFile.isEncrypted()) {
				zipFile.setPassword(password);
			}
			zipFile.extractAll(folder.toString());
		} catch (final ZipException ex) {
			AppLogger.error(ex);
		}
	}



	public static boolean uriIsFile(final URI uri) {
		return (uri != null) && (!uri.isAbsolute() || uri.getScheme().equals("file"));
	}



	public static void compareBytesRead(final int bytesExpected, final int bytesRead) {
		if (bytesRead < bytesExpected) {
			AppLogger.warn("Fewer bytes read (" + bytesRead + ") than expected (" + bytesExpected + ").");
		}
	}



	public static boolean readFully(final InputStream input, final byte[] buffer) throws IOException {
		boolean eof = false;

		if (input != null) {
			try {
				IOUtils.readFully(input, buffer);
			} catch (final EOFException ex) {
				eof = true;
			}
		} else {
			throw new IOException("InputStream is null!");
		}

		return eof;
	}



	public static boolean readFully(final InputStream input, final byte[] buffer, final int offset, final int length) throws IOException {
		boolean eof = false;

		if (input != null) {
			try {
				IOUtils.readFully(input, buffer, offset, length);
			} catch (final EOFException ex) {
				eof = true;
			}
		} else {
			throw new IOException("InputStream is null!");
		}

		return eof;
	}



	public static boolean skipFully(final InputStream input, final int toSkip) throws IOException {
		boolean eof = false;

		if (input != null) {
			try {
				IOUtils.skipFully(input, toSkip);
			} catch (final EOFException ex) {
				eof = true;
			}
		} else {
			throw new IOException("InputStream is null!");
		}

		return eof;
	}



	public static String getNameWithoutExtension(final File file) {
		final String name = file.getName();
		final int index = name.indexOf('.');
		if (index != -1) {
			return name.substring(0, index);
		}

		return name;
	}



	public static String removeExtension(final String filename) {
		final int index = filename.indexOf('.');
		if (index != -1) {
			return filename.substring(0, index);
		}

		return filename;
	}



	public static File replaceExtension(final File file, final String extVal) {
		String ext = extVal;
		final String parent = file.getParent();
		final String name = getNameWithoutExtension(file);

		if (ext.charAt(0) != '.') {
			ext = ("." + ext);
		}

		return new File(parent, name + ext);
	}



	public static File findUniqueFilename(final File oldFile) {
		File file = oldFile;
		final String ext = getFullExtension(file);
		final String parent = file.getParent();
		final String name = getNameWithoutExtension(file);

		int suffix = 2;

		while (file.exists()) {
			file = new File(parent, name + suffix + ((ext != null) ? "." + ext : ""));
			suffix++;
		}

		return file;
	}



	public static File findUniqueFilename(final File oldFile, final String suf, final String ext) {
		final String dir = oldFile.getParent();
		final String filename = getNameWithoutExtension(oldFile);
		final String suffix = (StringUtils.isNotBlank(suf) ? suf : "");
		final String extension = (StringUtils.isNotBlank(ext) ? ext : "");

		File selectedFile = new File(dir, filename + suffix + extension);

		for (int ctr = 2; selectedFile.exists(); ctr++) {
			selectedFile = new File(dir, filename + suffix + ctr + extension);
		}

		return selectedFile;
	}



	public static boolean writeTest(final File dir) {
		if (dir.toString().startsWith("http")) {
			return false;
		}

		if (dir.canWrite()) {
			return true;
		}

		// File.canWrite() not always working as expected for directories in Windows, try to write temp file to be sure
		final File testFile = new File(dir, ".mangotestfile" + (String.valueOf(Math.random())).substring(1));
		boolean canCreate = false;
		try {
			canCreate = testFile.createNewFile();
		} catch (final IOException ex) {
			AppLogger.warn(ex);
		}

		if (!testFile.delete()) {
			testFile.deleteOnExit();
		}

		return canCreate;
	}
}
