
package edu.uthscsa.ric.utilities;

import java.awt.Desktop;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;

import javax.swing.BorderFactory;
import javax.swing.JOptionPane;
import javax.swing.JTextArea;


public final class DesktopUtilities {

	public static final String OS = System.getProperty("os.name", "").toUpperCase();
	public static final boolean LINUX = (OS.indexOf("LINUX") != -1);
	public static final boolean MAC = (OS.indexOf("MAC") != -1);
	public static final boolean WINDOWS = (OS.indexOf("WINDOWS") != -1);



	private DesktopUtilities() {}



	public static void openURL(final String url) {
		boolean success = false;

		try {
			if (Desktop.isDesktopSupported()) {
				Desktop.getDesktop().browse(new URI(url));
				success = true;
			}

			if (!success) {
				success = openURLNonDesktop(url);
			}
		} catch (final IOException ex) {
			AppLogger.error(ex);
		} catch (final URISyntaxException ex) {
			AppLogger.error(ex);
		} catch (final UnsupportedOperationException ex) {
			AppLogger.error(ex);
		}

		if (!success) {
			final JTextArea text = new JTextArea("Browser could not be opened.  Here's the URL:\n " + url);
			text.setEditable(false);
			text.setBorder(BorderFactory.createEmptyBorder(4, 4, 4, 4));
			JOptionPane.showMessageDialog(null, text);
		}
	}



	public static void openFile(final File file) {
		boolean success = false;

		// first try using Desktop
		try {
			if (Desktop.isDesktopSupported()) {
				Desktop.getDesktop().open(file);
				success = true;
			}
		} catch (final IOException ex) {
			if (AppLogger.isDebugEnabled()) {
				AppLogger.debug(ex.getMessage());
			}
		} catch (final UnsupportedOperationException ex) {
			if (AppLogger.isDebugEnabled()) {
				AppLogger.debug(ex.getMessage());
			}
		}

		// if it's not supported, if linux, try xdg-open
		if (!success && LINUX) {
			success = openLinuxFile(file);
		}

		// still no success, try to open it in the browser, again with Desktop
		if (!success) {
			try {
				if (Desktop.isDesktopSupported()) {
					Desktop.getDesktop().browse(file.toURI());
					success = true;
				}
			} catch (final IOException ex) {
				if (AppLogger.isDebugEnabled()) {
					AppLogger.debug(ex.getMessage());
				}
			}

		}

		// ok, finally try the browser again without Desktop
		if (!success) {
			try {
				success = openURLNonDesktop(file.toURI().toURL().toString());
			} catch (final MalformedURLException ex) {
				if (AppLogger.isDebugEnabled()) {
					AppLogger.debug(ex.getMessage());
				}
			}
		}

		if (!success) {
			JOptionPane.showMessageDialog(null, "This file could not be opened.");
		}
	}



	public static void revealAndSelect(final File file) {
		try {
			if (MAC) {
				final String cmd = "tell application \"Finder\"\r reveal POSIX file \"" + file.toString() + "\"\r activate\r end tell\r";
				Runtime.getRuntime().exec(new String[] { "osascript", "-e", cmd });
			} else if (WINDOWS) {
				Runtime.getRuntime().exec("explorer /select,\"" + file.toString() + "\"");
			} else {
				final String[] cmdArray = { "xterm", "-hold", "-e", "cd " + file.getParentFile().getAbsolutePath() + "; ls -l; exec /bin/bash" };
				Runtime.getRuntime().exec(cmdArray);
			}
		} catch (final Exception ex) {
			AppLogger.error(ex);
		}
	}



	private static boolean openLinuxFile(final File file) {
		try {
			if (Runtime.getRuntime().exec(new String[] { "which", "xdg-open" }).getInputStream().read() != -1) {
				Runtime.getRuntime().exec(new String[] { "xdg-open", file.toString() });
				return true;
			}
		} catch (final IOException ex) {
			AppLogger.error(ex);
		}

		return false;
	}



	// adapted from http://www.centerkey.com/java/browser/
	private static boolean openURLNonDesktop(final String url) {
		boolean success = false;
		final String[] browsers = { "google-chrome", "firefox", "opera", "epiphany", "konqueror", "conkeror", "midori", "kazehakase", "mozilla" };

		if (MAC) {
			final Runtime runtime = Runtime.getRuntime();
			final String[] args = { "osascript", "-e", "open location \"" + url + "\"" };
			try {
				runtime.exec(args);
				success = true;
			} catch (final IOException ex) {
				if (AppLogger.isDebugEnabled()) {
					AppLogger.debug(ex.getMessage());
				}
			}
		} else if (WINDOWS) {
			try {
				Runtime.getRuntime().exec("rundll32 url.dll,FileProtocolHandler " + url);
				success = true;
			} catch (final IOException ex) {
				if (AppLogger.isDebugEnabled()) {
					AppLogger.debug(ex.getMessage());
				}
			}
		} else {
			for (final String b : browsers) {
				try {
					if (Runtime.getRuntime().exec(new String[] { "which", b }).getInputStream().read() != -1) {
						Runtime.getRuntime().exec(new String[] { b, url });
						success = true;
						break;
					}

					success = true;
				} catch (final IOException ex) {
					if (AppLogger.isDebugEnabled()) {
						AppLogger.debug(ex.getMessage());
					}
				}
			}
		}

		return success;
	}
}
