
package edu.uthscsa.ric.utilities;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.zip.GZIPInputStream;


public final class CompressionUtilities {

	private static final byte[] JPEG2000_MAGIC_NUMBER = { (byte) 0xFF, (byte) 0x4F, (byte) 0xFF, (byte) 0x51 };
	private static final byte[] JPEG_MAGIC_NUMBER = { (byte) 0xFF, (byte) 0xD8 };

	public static final List<String> COMMON_GZIP_EXTENSIONS = CollectionUtilities.immutable(".gz", ".gzip", ".z", "-gz", "-z", "_z");
	public static final String PREFERRED_GZIP_EXTENSION = ".gz";



	private CompressionUtilities() {}



	public static boolean containsJpeg2000Tag(final URI uri, final long offset) {
		boolean success = false;
		InputStream input = null;

		try {
			input = FileUtilities.getInputStream(uri, false);
			FileUtilities.skipFully(input, (int) offset);

			success = true;

			for (final byte element : JPEG2000_MAGIC_NUMBER) {
				final byte read = (byte) input.read();
				success &= (read == element);
			}
		} catch (final IOException ex) {
			AppLogger.error(ex);
		} finally {
			if (input != null) {
				try {
					input.close();
				} catch (final IOException ex) {
					AppLogger.warn(ex);
				}
			}
		}

		return success;
	}



	public static long findJpeg2000Tag(final URI uri, final long start, final int searchLength) {
		InputStream input = null;

		try {
			input = FileUtilities.getInputStream(uri, false);
			FileUtilities.skipFully(input, (int) start);

			final byte[] data = new byte[searchLength];
			FileUtilities.readFully(input, data);

			return ByteUtilities.indexOf(data, JPEG2000_MAGIC_NUMBER);
		} catch (final IOException ex) {
			AppLogger.error(ex);
		} finally {
			if (input != null) {
				try {
					input.close();
				} catch (final IOException ex) {
					AppLogger.warn(ex);
				}
			}
		}

		return -1;
	}



	public static boolean containsJpegTag(final URI uri, final long offset) {
		boolean success = false;

		InputStream input = null;
		final byte[] byteArray = new byte[JPEG_MAGIC_NUMBER.length];

		try {
			input = FileUtilities.getInputStream(uri, false);
			FileUtilities.skipFully(input, (int) offset);
			byteArray[0] = (byte) input.read();
			byteArray[1] = (byte) input.read();
			success = ((byteArray[0] == CompressionUtilities.JPEG_MAGIC_NUMBER[0]) && (byteArray[1] == CompressionUtilities.JPEG_MAGIC_NUMBER[1]));
		} catch (final IOException ex) {
			AppLogger.error(ex);
		} finally {
			if (input != null) {
				try {
					input.close();
				} catch (final IOException ex) {
					AppLogger.warn(ex);
				}
			}
		}

		return success;
	}



	public static boolean isPossibleGZIPFile(final File file) {
		if ((file == null) || !file.exists() || !file.canRead()) {
			return false;
		}

		BufferedInputStream fileIn = null;
		final byte[] byteArray = new byte[4];
		ByteBuffer byteBuffer = null;
		boolean error = false;
		boolean possibleGZIP = false;

		try {
			fileIn = new BufferedInputStream(new FileInputStream(file));
			final int bytesRead = fileIn.read(byteArray);

			if (bytesRead > 0) {
				byteBuffer = ByteBuffer.wrap(byteArray, 0, bytesRead);
			} else {
				error = true;
			}
		} catch (final Exception ex) {
			error = true;
			AppLogger.info(ex);
		} finally {
			try {
				fileIn.close();
			} catch (final Exception ex) {
				AppLogger.warn(ex);
			}
		}

		if (!error) {
			final int gzipMagic = byteBuffer.getInt(0);
			possibleGZIP = ((gzipMagic & 0xFFFF) == GZIPInputStream.GZIP_MAGIC) || ((ByteUtilities.swap(gzipMagic) & 0xFFFF) == GZIPInputStream.GZIP_MAGIC);
		}

		return possibleGZIP;
	}



	public static boolean usesGZIPFileExtension(final File file) {
		if (file != null) {
			return usesGZIPFileExtension(file.getName());
		} else {
			return false;
		}
	}



	public static boolean usesGZIPFileExtension(final String name) {
		for (final String element : COMMON_GZIP_EXTENSIONS) {
			if (name.toLowerCase().endsWith(element)) {
				return true;
			}
		}

		return false;
	}
}
