
package edu.uthscsa.ric.volume.formats.dicom;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;

import edu.uthscsa.ric.utilities.AppLogger;
import edu.uthscsa.ric.utilities.ByteUtilities;
import edu.uthscsa.ric.utilities.CollectionUtilities;
import edu.uthscsa.ric.utilities.StringUtilities;
import edu.uthscsa.ric.volume.formats.dicom.privateheader.PrivateHeader;
import edu.uthscsa.ric.volume.formats.dicom.privateheader.Siemens;


public class Item implements Comparable<Item> {

	private Item next;
	private List<Item> sublist;
	private String privateTags;
	private String vr;
	private boolean littleEndian;
	private boolean readPrivateData;
	private byte[] value;
	private long offsetEnd;
	private long offsetStart;
	private long offsetValue;
	private long valueLength;
	private long valueLengthEncapsulated;
	private static final List<PrivateHeader> PRIVATE_HEADERS;

	private final int element;
	private final int group;

	static {
		PRIVATE_HEADERS = new ArrayList<PrivateHeader>();
		PRIVATE_HEADERS.add(new Siemens());
	}



	public Item(final int group, final int element, final String vr) {
		this.group = group;
		this.element = element;

		if (vr == null) {
			this.vr = Dictionary.getVR(group, element);
			if (this.vr == null) {
				this.vr = "OB";
			}
		} else {
			this.vr = vr;
		}

		littleEndian = true;
	}



	public Item(final int group, final int element, final String vr, final byte[] value) {
		this(group, element, vr);
		this.value = value;
	}



	public Item(final int group, final int element, final String vr, final long offsetStart, final long offsetValue, final long offsetEnd,
			final boolean littleEndian) {
		this(group, element, vr);

		this.offsetStart = offsetStart;
		this.offsetValue = offsetValue;
		this.offsetEnd = offsetEnd;
		this.littleEndian = littleEndian;
	}



	public Item(final int group, final int element, final String vr, final long offsetStart, final long offsetValue, final long offsetEnd,
			final boolean little, final byte[] value) {
		this(group, element, vr, offsetStart, offsetValue, offsetEnd, little);
		this.value = value;
	}



	public Item(final int group, final int element, final String vr, final long offsetStart, final long offsetValue, final long offsetEnd,
			final boolean little, final List<Item> sublist) {
		this(group, element, vr, offsetStart, offsetValue, offsetEnd, little);
		this.sublist = sublist;
	}



	public Item(final int group, final int element, final String vr, final List<Item> sublist) {
		this(group, element, vr);
		this.sublist = sublist;
	}



	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		final Item other = (Item) obj;
		if (element != other.element) {
			return false;
		}
		if (group != other.group) {
			return false;
		}
		return true;
	}



	public int getElement() {
		return element;
	}



	public long getElementSize() {
		return offsetEnd - offsetStart;
	}



	public int getGroup() {
		return group;
	}



	public String getHexTag() {
		return "(" + StringUtilities.makeNiceHexString((short) group) + "," + StringUtilities.makeNiceHexString((short) element) + ")";
	}



	public Item getNext() {
		return next;
	}



	public long getOffsetEnd() {
		return offsetEnd;
	}



	public long getOffsetStart() {
		return offsetStart;
	}



	public String getPrivateTags() {
		return privateTags;
	}



	public List<Item> getSublist() {
		return sublist;
	}



	public boolean isGroupLength() {
		return (element == 0);
	}



	public byte[] getValue() {
		return value;
	}



	public int getValueAsInteger() {
		int valueInteger = 0;

		if (Dictionary.VR_TYPE_16_UNSIGNED.indexOf(vr) != -1) {
			valueInteger = (getShort(value, 0) & 0xFFFF);
		} else if (Dictionary.VR_TYPE_16_SIGNED.indexOf(vr) != -1) {
			valueInteger = (getShort(value, 0));
		} else if (Dictionary.VR_TYPE_32_UNSIGNED.indexOf(vr) != -1) {
			valueInteger = (getInt(value, 0));
		} else if (Dictionary.VR_TYPE_32_SIGNED.indexOf(vr) != -1) {
			valueInteger = (getInt(value, 0));
		}

		return valueInteger;
	}



	public long getValueOffset() {
		return offsetValue;
	}



	public long getValueSize() {
		if (value != null) {
			return value.length;
		} else {
			return valueLength;
		}
	}



	public String getVR() {
		return vr;
	}



	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = (prime * result) + element;
		result = (prime * result) + group;
		return result;
	}



	public boolean isMetaLength() {
		return ((group == DICOM.TAG_META_LENGTH[0]) && (element == DICOM.TAG_META_LENGTH[1]));
	}



	public boolean isPixelData() {
		return (group == DICOM.TAG_PIXEL_DATA[0]) && (element == DICOM.TAG_PIXEL_DATA[1]);
	}



	public boolean isRawData() {
		final String vr = getVR();
		if (vr != null) {
			return (Dictionary.VR_RAW_DATA.indexOf(vr) != -1);
		}

		return true;
	}



	public boolean isSequenceDelim() {
		return (group == DICOM.TAG_SUBLIST_SEQ_DELIM[0]) && (element == DICOM.TAG_SUBLIST_SEQ_DELIM[1]);
	}



	public boolean isSublist() {
		return (sublist != null);
	}



	public boolean isSpecialExplicitDataType() {
		return Dictionary.VRS_DATA.indexOf(getVR()) != -1;
	}



	public boolean isSublistItem() {
		return (group == DICOM.TAG_SUBLIST_ITEM[0]) && (element == DICOM.TAG_SUBLIST_ITEM[1]);
	}



	public boolean isSublistItemDelim() {
		return (group == DICOM.TAG_SUBLIST_ITEM_DELIM[0]) && (element == DICOM.TAG_SUBLIST_ITEM_DELIM[1]);
	}



	public boolean isTransferSyntax() {
		return ((group == DICOM.TAG_TRANSFER_SYNTAX_ID[0]) && (element == DICOM.TAG_TRANSFER_SYNTAX_ID[1]));
	}



	public boolean isValid() {
		return ((group != 0) || (element != 0));
	}



	public void setNext(final Item next) {
		this.next = next;
	}



	public void setPrivateTags(final String privateTags) {
		this.privateTags = privateTags;
	}



	@Override
	public String toString() {
		return makeFormattedString(0);
	}



	private double getDouble(final byte[] buffer, final int index) {
		if (littleEndian) {
			return ByteUtilities.swapDouble(buffer, index);
		} else {
			return ByteUtilities.getDouble(buffer, index);
		}
	}



	private List<Double> getDoubles(final byte[] buffer) {
		final List<Double> vals = new ArrayList<Double>();
		final int count = buffer.length / 8;

		if ((buffer.length % 8) == 0) {
			for (int ctr = 0; ctr < count; ctr++) {
				vals.add(getDouble(buffer, ctr * 8));
			}
		}

		return vals;
	}



	private float getFloat(final byte[] buffer, final int index) {
		if (littleEndian) {
			return ByteUtilities.swapFloat(buffer, index);
		} else {
			return ByteUtilities.getFloat(buffer, index);
		}
	}



	private List<Float> getFloats(final byte[] buffer) {
		final List<Float> vals = new ArrayList<Float>();
		final int count = buffer.length / 4;

		if ((buffer.length % 4) == 0) {
			for (int ctr = 0; ctr < count; ctr++) {
				vals.add(getFloat(buffer, ctr * 4));
			}
		}

		return vals;
	}



	private int getInt(final byte[] buffer, final int index) {
		if (littleEndian) {
			return ByteUtilities.swapInt(buffer, index);
		} else {
			return ByteUtilities.getInt(buffer, index);
		}
	}



	private List<Integer> getInts(final byte[] buffer) {
		final List<Integer> vals = new ArrayList<Integer>();
		final int count = buffer.length / 4;

		if ((buffer.length % 4) == 0) {
			for (int ctr = 0; ctr < count; ctr++) {
				vals.add(getInt(buffer, ctr * 4));
			}
		}

		return vals;
	}



	private short getShort(final byte[] buffer, final int index) {
		if (littleEndian) {
			return ByteUtilities.swapShort(buffer, index);
		} else {
			return ByteUtilities.getShort(buffer, index);
		}
	}



	private List<Integer> getShorts(final byte[] buffer, final boolean unsigned) {
		final List<Integer> vals = new ArrayList<Integer>();
		final int count = buffer.length / 2;

		if ((buffer.length % 2) == 0) {
			for (int ctr = 0; ctr < count; ctr++) {
				if (unsigned) {
					vals.add(getShort(buffer, ctr) & 0xFFFF);
				} else {
					vals.add((int) getShort(buffer, ctr * 2));
				}
			}
		}

		return vals;
	}



	public String[] getValueAsStringArray() {
		final String value = getValueAsString();
		return value.split("\\\\");
	}



	public String getValueAsString() {
		String string = null;

		if (CollectionUtilities.isNotEmpty(value)) {
			if (vr == null) {
				string = "(" + StringUtilities.makeNiceFileSizeString(value.length) + " of raw data)";
			} else if (Dictionary.VR_RAW_DATA.indexOf(vr) != -1) {
				string = "(" + StringUtilities.makeNiceFileSizeString(value.length) + " of raw data)";
			} else if (Dictionary.VR_TEXT.indexOf(vr) != -1) {
				string = StringUtilities.convertBytesToNiceString(value);
			} else if (Dictionary.VR_TAG.indexOf(vr) != -1) {
				final short group = ByteUtilities.swapShort(value, 0);
				final short item = ByteUtilities.swapShort(value, 2);
				string = ("(" + StringUtils.leftPad(Integer.toHexString(group), 4, "0") + "," + StringUtils.leftPad(Integer.toHexString(item), 4, "0") + ")");
			} else {
				if (Dictionary.VR_TYPE_16_UNSIGNED.indexOf(vr) != -1) {
					string = getShorts(value, true).toString();
				} else if (Dictionary.VR_TYPE_16_SIGNED.indexOf(vr) != -1) {
					string = getShorts(value, false).toString();
				} else if (Dictionary.VR_TYPE_32_UNSIGNED.indexOf(vr) != -1) {
					string = getInts(value).toString();
				} else if (Dictionary.VR_TYPE_32_SIGNED.indexOf(vr) != -1) {
					string = getInts(value).toString();
				} else if (Dictionary.VR_TYPE_32_FLOAT.indexOf(vr) != -1) {
					string = getFloats(value).toString();
				} else if (Dictionary.VR_TYPE_64_FLOAT.indexOf(vr) != -1) {
					string = getDoubles(value).toString();
				} else {
					string = "(" + StringUtilities.makeNiceFileSizeString(value.length) + " of raw data)";
				}
			}
		} else if (sublist != null) {
			string = "(Sublist)";
		} else {
			string = "";
		}

		return string;
	}



	public boolean isPrivateData() {
		return ((group & 1) == 1);
	}



	public boolean canReadPrivateData() {
		for (final PrivateHeader header : PRIVATE_HEADERS) {
			if (header.canRead(group, element)) {
				return true;
			}
		}

		return false;
	}



	public String readPrivateData() {
		if (!readPrivateData) {
			readPrivateData = true;

			for (final PrivateHeader header : PRIVATE_HEADERS) {
				if (header.canRead(group, element)) {
					try {
						privateTags = header.readHeader(value);
					} catch (final Exception ex) {
						AppLogger.warn("There was a problem reading private header data (" + header.getDescription() + ")");
					}

					if (StringUtils.isNotBlank(privateTags)) {
						break;
					}
				}
			}
		}

		return privateTags;
	}



	public String getPrivateDataDescription() {
		for (final PrivateHeader header : PRIVATE_HEADERS) {
			if (header.canRead(group, element)) {
				return header.getDescription();
			}
		}

		return "Private Data";
	}



	private String makeFormattedString(final int indentStart) {
		final int indent = indentStart;

		final StringBuffer indentation = new StringBuffer();
		for (int ctr = 0; ctr < indent; ctr++) {
			indentation.append("    ");
		}

		String valueStr = null;

		if (sublist != null) {
			final StringBuffer valueStrBuffer = new StringBuffer();
			for (final Item item : sublist) {
				valueStrBuffer.append('\n' + item.makeFormattedString(indent + 1));

				Item next = item.getNext();
				while (next != null) {
					valueStrBuffer.append('\n' + next.makeFormattedString(indent + 1));
					next = next.getNext();
				}
			}

			valueStr = valueStrBuffer.toString();
		} else if (vr.equals("SQ")) {
			valueStr = "";
		} else if (this.isPixelData()) {
			valueStr = "";
		} else if (value == null) {
			valueStr = "";
		} else if (isPrivateData()) {
			valueStr = readPrivateData();
			if (StringUtils.isBlank(valueStr)) {
				valueStr = '[' + getValueAsString() + ']';
			}
		} else {
			valueStr = '[' + getValueAsString() + ']';
		}

		valueStr = valueStr.replace("\\", ",");
		valueStr = valueStr.replace("[[", "[");
		valueStr = valueStr.replace("]]", "]");

		String tagStr = getHexTag();
		String des = "";

		if (this.isSublistItem()) {
			tagStr = "Sequence Item";
		} else if (this.isSublistItemDelim()) {
			tagStr = "Sequence Item Delimiter";
		} else if (this.isSequenceDelim()) {
			tagStr = "Sequence Delimiter";
		} else if (this.isPixelData()) {
			tagStr = "Pixel Data";
		} else {
			des = Dictionary.getDescription(group, element);

			if (des == null) {
				if (isPrivateData()) { // if odd, private data
					des = getPrivateDataDescription();
				} else {
					des = "Unknown Data";
				}
			}
		}

		return (indentation.toString() + ' ' + tagStr + ' ' + des + ' ' + valueStr);
	}



	public long getValueLengthEncapsulated() {
		return valueLengthEncapsulated;
	}



	public void setValueLengthEncapsulated(final long valueLengthEncapsulated) {
		this.valueLengthEncapsulated = valueLengthEncapsulated;
	}



	public void setValue(final String valueStr, final int index) {
		if (Dictionary.VR_TEXT.indexOf(vr) != -1) {
			final String[] values = getValueAsStringArray();
			values[index] = valueStr;
			value = StringUtils.join(values, '\\').getBytes();
		} else if (Dictionary.VR_TAG.indexOf(vr) != -1) {
			int startIndex = valueStr.indexOf('(') + 1;
			int endIndex = valueStr.indexOf(',', startIndex);
			final String groupStr = valueStr.substring(startIndex, endIndex);
			final int groupVal = Integer.parseInt(groupStr, 16);

			startIndex = endIndex + 1;
			endIndex = valueStr.indexOf(')');
			final String elemStr = valueStr.substring(startIndex, endIndex);
			final int elemVal = Integer.parseInt(elemStr, 16);

			byte[] groupBytes = null;
			try {
				if (littleEndian) {
					groupBytes = ByteUtilities.getBytes(ByteUtilities.swap((short) (groupVal & 0xFFFF)));
				} else {
					groupBytes = ByteUtilities.getBytes((short) (groupVal & 0xFFFF));
				}
			} catch (final NumberFormatException ex) {
				AppLogger.warn(ex);
			}

			byte[] elemBytes = null;
			try {
				if (littleEndian) {
					elemBytes = ByteUtilities.getBytes(ByteUtilities.swap((short) (elemVal & 0xFFFF)));
				} else {
					elemBytes = ByteUtilities.getBytes((short) (elemVal & 0xFFFF));
				}
			} catch (final NumberFormatException ex) {
				AppLogger.warn(ex);
			}

			value = new byte[4];
			System.arraycopy(groupBytes, 0, value, 0, 2);
			System.arraycopy(elemBytes, 0, value, 2, 2);
		} else {
			if (Dictionary.VR_TYPE_16_UNSIGNED.indexOf(vr) != -1) {
				try {
					final int valueInt = Integer.parseInt(valueStr);
					if (littleEndian) {
						value = ByteUtilities.getBytes(ByteUtilities.swap((short) (valueInt & 0xFFFF)));
					} else {
						value = ByteUtilities.getBytes((short) (valueInt & 0xFFFF));
					}
				} catch (final NumberFormatException ex) {
					AppLogger.warn(ex);
				}
			} else if (Dictionary.VR_TYPE_16_SIGNED.indexOf(vr) != -1) {
				try {
					final short valueInt = Short.parseShort(valueStr);
					if (littleEndian) {
						value = ByteUtilities.getBytes(ByteUtilities.swap(valueInt));
					} else {
						value = ByteUtilities.getBytes(valueInt);
					}
				} catch (final NumberFormatException ex) {
					AppLogger.warn(ex);
				}
			} else if (Dictionary.VR_TYPE_32_UNSIGNED.indexOf(vr) != -1) {
				try {
					final long valueInt = Long.parseLong(valueStr);
					if (littleEndian) {
						value = ByteUtilities.getBytes(ByteUtilities.swap((int) (valueInt & 0xFFFFFFFF)));
					} else {
						value = ByteUtilities.getBytes((int) (valueInt & 0xFFFFFFFF));
					}
				} catch (final NumberFormatException ex) {
					AppLogger.warn(ex);
				}
			} else if (Dictionary.VR_TYPE_32_SIGNED.indexOf(vr) != -1) {
				try {
					final int valueInt = Integer.parseInt(valueStr);
					if (littleEndian) {
						value = ByteUtilities.getBytes(ByteUtilities.swap(valueInt));
					} else {
						value = ByteUtilities.getBytes(valueInt);
					}
				} catch (final NumberFormatException ex) {
					AppLogger.warn(ex);
				}
			} else if (Dictionary.VR_TYPE_32_FLOAT.indexOf(vr) != -1) {
				try {
					final float valueFlt = Float.parseFloat(valueStr);
					if (littleEndian) {
						value = ByteUtilities.getBytes(ByteUtilities.swap(valueFlt));
					} else {
						value = ByteUtilities.getBytes(valueFlt);
					}
				} catch (final NumberFormatException ex) {
					AppLogger.warn(ex);
				}
			} else if (Dictionary.VR_TYPE_64_FLOAT.indexOf(vr) != -1) {
				try {
					final double valueFlt = Double.parseDouble(valueStr);
					if (littleEndian) {
						value = ByteUtilities.getBytes(ByteUtilities.swap(valueFlt));
					} else {
						value = ByteUtilities.getBytes(valueFlt);
					}
				} catch (final NumberFormatException ex) {
					AppLogger.warn(ex);
				}
			} else {
				AppLogger.warn("Cannot set this value: " + valueStr);
			}
		}
	}



	public boolean isLittleEndian() {
		return littleEndian;
	}



	@Override
	public int compareTo(final Item other) {
		if (other.group > group) {
			return -1;
		} else if (other.group < group) {
			return 1;
		} else {
			if (other.element > element) {
				return -1;
			} else if (other.element < element) {
				return 1;
			}
		}

		return 0;
	}



	public void setValueLength(final long valueLength) {
		this.valueLength = valueLength;
	}
}
