
package edu.uthscsa.ric.volume.formats.dicom;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.vecmath.Vector3d;

import edu.uthscsa.ric.utilities.AppLogger;
import edu.uthscsa.ric.utilities.ByteUtilities;
import edu.uthscsa.ric.utilities.CollectionUtilities;
import edu.uthscsa.ric.utilities.FileUtilities;
import edu.uthscsa.ric.volume.Coordinate;
import edu.uthscsa.ric.volume.ImageDescription;
import edu.uthscsa.ric.volume.ImageDimensions;
import edu.uthscsa.ric.volume.ImageRange;
import edu.uthscsa.ric.volume.ImageTransform;
import edu.uthscsa.ric.volume.ImageType;
import edu.uthscsa.ric.volume.VolumeIOException;
import edu.uthscsa.ric.volume.VoxelDimensions;
import edu.uthscsa.ric.volume.WritableHeader;


// ftp://medical.nema.org/medical/dicom/2009/
public class ExportableSOPClassSC extends ExportableSOPClass implements WritableHeader {

	public static final String CONVERSION_TYPE = "SYN"; // synthetic
	public static final String MODALITY = "OT"; // other
	public static final String NAME = "DICOM (SC)";
	public static final String SOP_CLASS_BYTE = "1.2.840.10008.5.1.4.1.1.7.2";
	public static final String SOP_CLASS_WORD = "1.2.840.10008.5.1.4.1.1.7.3";



	@Override
	public boolean canCompress() {
		return false;
	}



	@Override
	public File formatImageFile(final File headerFile) {
		return headerFile;
	}



	@Override
	public String getMinimumVersionSupported() {
		return null;
	}



	@Override
	public String getPluginName() {
		return NAME;
	}



	@Override
	public URL getPluginURL() {
		return null;
	}



	@Override
	public String getSOPClass(final ImageType it) {
		final boolean isWord = it.getNumBytesPerVoxel() > 1;
		if (isWord) {
			return SOP_CLASS_WORD;
		} else {
			return SOP_CLASS_BYTE;
		}
	}



	@Override
	public int[] getSupportedBytesForType(final int type) {
		if (type == ImageType.BYTE_TYPE_INTEGER) {
			return null;
		} else if (type == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) {
			return new int[] { 1, 2 };
		} else {
			return null;
		}
	}



	@Override
	public String[] getSupportedOrientations() {
		return CollectionUtilities.EMPTY_STRING_ARRAY;
	}



	@Override
	public int[] getSupportedTypesForBytes(final int bytes) {
		if (bytes == 1) {
			return new int[] { ImageType.BYTE_TYPE_INTEGER_UNSIGNED };
		} else if (bytes == 2) {
			return new int[] { ImageType.BYTE_TYPE_INTEGER_UNSIGNED };
		} else {
			return null;
		}
	}



	@Override
	public String getVersion() {
		return null;
	}



	@Override
	public boolean hasNewerVersion() {
		return false;
	}



	@Override
	public boolean isSupportedBigEndian() {
		return false;
	}



	@Override
	public boolean isSupportedDataScaleIntercept() {
		return true;
	}



	@Override
	public boolean isSupportedDataScaleSlope() {
		return true;
	}



	@Override
	public boolean isSupportedLittleEndian() {
		return true;
	}



	@Override
	public boolean isSupportedMultipleDataScales() {
		return false;
	}



	@Override
	public boolean isThisFormat(final File file) {
		if (!super.isThisFormat(file)) {
			return false;
		}

		BufferedInputStream fileIn = null;
		final byte[] data = new byte[256];

		try {
			fileIn = FileUtilities.getInputStream(file.toURI(), false);
			FileUtilities.readFully(fileIn, data);

			final byte[] sopClassByte = SOP_CLASS_BYTE.getBytes();
			final byte[] sopClassWord = SOP_CLASS_WORD.getBytes();

			if (ByteUtilities.indexOf(data, sopClassWord) != -1) {
				return true;
			}

			if (ByteUtilities.indexOf(data, sopClassByte) != -1) {
				return true;
			}

			return false;
		} catch (final IOException ex) {
			AppLogger.error(ex);
		} finally {
			try {
				if (fileIn != null) {
					fileIn.close();
				}
			} catch (final IOException ex) {
				AppLogger.warn(ex);
			}
		}

		return false;
	}



	@Override
	public boolean willWriteImage() {
		return false;
	}



	@Override
	public byte[] writeHeader(final ImageDimensions id, final VoxelDimensions vd, final ImageType it, final ImageRange ir, final ImageDescription in,
			final boolean compress, final Coordinate origin, final String orientation, final ImageTransform[] transforms, final File headerFile,
			final File imageFile, final ByteBuffer[] imageBuffer, final File tempDir) throws VolumeIOException {
		// create IDs
		final String studyUID = DICOMUtils.createUID();
		final String seriesUID = DICOMUtils.createUID();
		final String frameOfRefUID = DICOMUtils.createUID();
		final String studyId = studyNumber;
		final int seriesId = getNextSeriesNumber();
		final int imageId = getImageNumber();
		final String accessionId = getNextAccessionNumber();
		final Date now = new Date();

		return writeHeader(headerFile, id, vd, it, ir, in, origin, orientation, 0, studyUID, seriesUID, frameOfRefUID, studyId, seriesId, imageId, accessionId,
				now);
	}



	@Override
	public byte[][] writeSeriesHeaders(final ImageDimensions id, final VoxelDimensions vd, final ImageType it, final ImageRange ir, final ImageDescription in,
			final boolean compress, final Coordinate origin, final String orientation, final ImageTransform[] transforms, final File headerFile,
			final File imageFile, final ByteBuffer[] imageBuffer, final File tempDir) throws VolumeIOException {
		// create IDs
		final String studyUID = DICOMUtils.createUID();
		final String seriesUID = DICOMUtils.createUID();
		final String frameOfRefUID = DICOMUtils.createUID();
		final String studyId = studyNumber;
		final int seriesId = getNextSeriesNumber();
		int imageId = getImageNumber();
		final String accessionId = getNextAccessionNumber();
		final Date now = new Date();

		final int numSeriesPoints = id.getTimepoints();
		final byte[][] data = new byte[numSeriesPoints][];
		for (int ctr = 0; ctr < numSeriesPoints; ctr++) {
			data[ctr] = writeHeader(headerFile, id, vd, it, ir, in, origin, orientation, ctr, studyUID, seriesUID, frameOfRefUID, studyId, seriesId, imageId,
					accessionId, now);
			imageId++;
		}

		return data;
	}



	private byte[] writeHeader(final File headerFile, final ImageDimensions id, final VoxelDimensions vd, final ImageType it, final ImageRange ir,
			final ImageDescription in, final Coordinate origin, final String orientation, final int seriesIndex, final String studyUID, final String seriesUID,
			final String frameOfRefUID, final String studyId, final int seriesId, final int imageId, final String accessionId, final Date now)
			throws VolumeIOException {
		final Vector<Item> tags = new Vector<Item>();
		final Map<Long, String> mappedDicomTags = in.getDicomItems();

		// Multi-frame Grayscale Byte SC Image Information Object Definition (A.8.3)
		// Multi-frame Grayscale Word SC Image Information Object Definition (A.8.4)

		// create IDs
		final String sopInstanceUID = DICOMUtils.createUID();

		try {
			// Meta Header
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_FILE_META_INFO_VERSION, new byte[] { 0, 1 }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_MEDIA_STORED_SOP_CLASS_UID, getSOPClass(it)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_MEDIA_STORED_SOP_INSTANCE_UID, sopInstanceUID));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_TRANSFER_SYNTAX_UID, XFER_SYNTAX_EXPLICIT_VT_LITLE_ENDIAN));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_IMPLEMENTATION_CLASS_UID, IMPLEMENTATION_CLASS_UID + ".1." + VERSION));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_IMPLEMENTATION_VERSION_NAME, IMPLEMENTATION_VERSION_NAME + " " + VERSION));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_GROUP_LENGTH, countSize(tags)));

			// SOP Common Module (C.12.1)
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_SOP_CLASS_UID, getSOPClass(it)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_SOP_INSTANCE_UID, sopInstanceUID));

			// Patient Module (C.7.1.1)
			addTag(tags, createTag(Dictionary.TAG_0010_PATIENT_NAME, mappedDicomTags, makePatientName(in, headerFile)));
			addTag(tags, createTag(Dictionary.TAG_0010_PATIENT_ID, mappedDicomTags, makePatientId(in, headerFile)));
			addTag(tags, createTag(Dictionary.TAG_0010_PATIENT_BIRTH_DATE, mappedDicomTags));
			addTag(tags, createTagType2(Dictionary.TAG_0010_PATIENT_SEX, mappedDicomTags));

			// General Study Module (C.7.2.1)
			addTag(tags, createTag(Dictionary.TAG_0020_STUDY_INSTANCE_ID, mappedDicomTags, studyUID));
			addTag(tags, createTag(Dictionary.TAG_0008_STUDY_DATE, mappedDicomTags, DATE_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0008_STUDY_TIME, mappedDicomTags, TIME_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0008_STUDY_DESCRIPTION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_REFERRING_PHYSICIAN_NAME, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0020_STUDY_ID, mappedDicomTags, studyId));
			addTag(tags, createTag(Dictionary.TAG_0008_ACCESSION_NUMBER, mappedDicomTags, accessionId));

			// General Equipment (C.7.5.1)
			addTag(tags, createTag(Dictionary.TAG_0008_MANUFACTURER, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_INSTITUTION_NAME, mappedDicomTags));

			// Patient Study Module (C.7.2.2)
			addTag(tags, createTag(Dictionary.TAG_0010_PATIENT_WEIGHT, mappedDicomTags));

			// General Series Module (C.7.3.1)
			addTag(tags, createTag(Dictionary.TAG_0008_MODALITY, mappedDicomTags, MODALITY));
			addTag(tags, createTag(Dictionary.TAG_0020_SERIES_INSTANCE_ID, mappedDicomTags, seriesUID));
			addTag(tags, createTag(Dictionary.TAG_0020_SERIES_ID, mappedDicomTags, new int[] { seriesId }));
			addTag(tags, createTag(Dictionary.TAG_0020_IMAGE_LATERALITY, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_SERIES_DESCRIPTION, mappedDicomTags));

			// Frame of Reference Module (C.7.4.1)
			addTag(tags, createTag(Dictionary.TAG_0020_FRAME_OF_REF_ID, mappedDicomTags, frameOfRefUID));
			addTag(tags, createTag(Dictionary.TAG_0020_POS_REF_INDICATOR, mappedDicomTags));

			// SC Equipment Module (C.8.6.1)
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_CONVERSION_TYPE, CONVERSION_TYPE));

			// General Image Module (C.7.6.1)
			addTag(tags, createTag(Dictionary.TAG_0020_IMAGE_NUMBER, mappedDicomTags, new int[] { imageId }));
			addTag(tags, createTagType2(Dictionary.TAG_0020_PATIENT_ORIENTATION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_CONTENT_DATE, mappedDicomTags, DATE_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0008_CONTENT_TIME, mappedDicomTags, TIME_FORMAT.format(now)));

			// Image Pixel Module (C.7.6.3)
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SAMPLES_PER_PIXEL, SAMPLES_PER_PIXEL));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PHOTOMETRIC_INTERP, PHOTOMETRIC_INTERP));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_COLUMNS, id.getCols()));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_ROWS, id.getRows()));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_BITS_ALLOCATED, it.getNumBytesPerVoxel() * 8));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_BITS_STORED, it.getNumBytesPerVoxel() * 8));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_HIGH_BIT, (it.getNumBytesPerVoxel() * 8) - 1));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PIXEL_REP, PIXEL_REPRESENTATION));

			// Multi-frame Module (C.7.6.6)
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_NUMBER_OF_FRAMES, new int[] { id.getSlices() }));
			addTag(tags,
					createExplicitTag(Dictionary.TAG_0028_FRAME_INCREMENT_POINTER, new short[] { Dictionary.TAG_0018_SLICE_LOCATION_VECTOR.getGroupId(),
							Dictionary.TAG_0018_SLICE_LOCATION_VECTOR.getElementId() }));

			// SC Multi-frame Image Module (C.8.6.3)
			addTag(tags, createTag(Dictionary.TAG_0028_BURNED_IN_ANNOTATION, mappedDicomTags));
			addTag(tags, createExplicitTag(Dictionary.TAG_2050_PRESENTATION_LUT_SHAPE, PRESENTATION_LUT_SHAPE));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_RESCALE_INTERCEPT, new double[] { ir.getDataScaleIntercepts()[seriesIndex * id.getSlices()] }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_RESCALE_SLOPE, new double[] { ir.getDataScaleSlopes()[seriesIndex * id.getSlices()] }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_RESCALE_TYPE, RESCALE_TYPE));
			addTag(tags,
					createExplicitTag(Dictionary.TAG_0028_FRAME_INCREMENT_POINTER, new short[] { Dictionary.TAG_0018_SLICE_LOCATION_VECTOR.getGroupId(),
							Dictionary.TAG_0018_SLICE_LOCATION_VECTOR.getElementId() }));

			// SC Multi-frame Vector Module (C.8.6.4)
			addTag(tags,
					createExplicitTag(Dictionary.TAG_0018_SLICE_LOCATION_VECTOR,
							makeSliceLocationVector(vd.getSliceThickness(true), id.getSlices(), 1, id, vd, orientation, origin)));

			// Pixel Measures Functional Group (C.7.6.16.2.1)
			final Item sliceThickness = createExplicitTag(Dictionary.TAG_0018_SLICE_THICKNESS, new double[] { vd.getSliceThickness(true) });
			final Item pixelSpacing = createExplicitTag(Dictionary.TAG_0028_PIXEL_SPACING, new double[] { vd.getRowSize(true), vd.getColSize(true) });
			sliceThickness.setNext(pixelSpacing);
			final Item pixelMeasuresSequence = createSequenceTag(Dictionary.TAG_0028_PIXEL_MEASURES_SQ, createListFromItem(sliceThickness));

			// Plane Orientation Functional Group (C.7.6.16.2.4)
			final Coordinate lastPixelFirstRow = makeRealWorldCoordinateAt(id.getCols() - 1, 0, 0, orientation, id, vd, origin);
			final Coordinate firstPixelFirstRow = makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin);
			final Vector3d vecRow = new Vector3d(lastPixelFirstRow.xDbl - firstPixelFirstRow.xDbl, lastPixelFirstRow.yDbl - firstPixelFirstRow.yDbl,
					lastPixelFirstRow.zDbl - firstPixelFirstRow.zDbl);
			vecRow.normalize();

			final Coordinate lastPixelFirstCol = makeRealWorldCoordinateAt(0, id.getRows() - 1, 0, orientation, id, vd, origin);
			final Coordinate firstPixelFirstCol = makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin);
			final Vector3d vecCol = new Vector3d(lastPixelFirstCol.xDbl - firstPixelFirstCol.xDbl, lastPixelFirstCol.yDbl - firstPixelFirstCol.yDbl,
					lastPixelFirstCol.zDbl - firstPixelFirstCol.zDbl);
			vecCol.normalize();

			final Item imageOrientation = createExplicitTag(Dictionary.TAG_0020_IMAGE_ORIENTATION, new double[] { vecRow.x, vecRow.y, vecRow.z, vecCol.x,
					vecCol.y, vecCol.z });
			final Item planeOrientationSequence = createExplicitTag(Dictionary.TAG_0020_PLANE_ORIENTATION_SQ, createListFromItem(imageOrientation));

			// Multi-frame Functional Groups Module (C.7.6.16)
			addTag(tags, createTag(Dictionary.TAG_0020_IMAGE_NUMBER, mappedDicomTags, new int[] { imageId }));
			addTag(tags, createTag(Dictionary.TAG_0008_CONTENT_DATE, mappedDicomTags, DATE_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0008_CONTENT_TIME, mappedDicomTags, TIME_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0028_NUMBER_OF_FRAMES, mappedDicomTags, new int[] { id.getSlices() }));

			{
				// Shared
				{
					planeOrientationSequence.setNext(pixelMeasuresSequence);
					addTag(tags, createSequenceTag(Dictionary.TAG_5200_SHARED_FUNCTIONAL_GROUPS_SQ, createListFromItem(planeOrientationSequence)));
				}

				// Per-frame
				{
					final List<Item> perFrameItems = new ArrayList<Item>();
					final int numSlices = id.getSlices();

					for (int ctr = 0; ctr < numSlices; ctr++) {
						// Plane Position Functional Group (C.7.6.16.2.3)
						final Item imagePos = createExplicitTag(Dictionary.TAG_0020_IMAGE_POSITION,
								makeRealWorldCoordinateAt(0, 0, ctr, orientation, id, vd, origin).getValues());
						final Item planePositionSequence = createSequenceTag(Dictionary.TAG_0020_PLANE_POSITION_SQ, createListFromItem(imagePos));
						perFrameItems.add(planePositionSequence);
					}

					addTag(tags, createSequenceTag(Dictionary.TAG_5200_PER_FRAME_FUNCTIONAL_GROUPS_SQ, perFrameItems));
				}
			}

			// Additional Tags (for better compatibility)
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SMALLEST_IMAGE_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMin(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_LARGEST_IMAGE_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMax(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SMALLEST_SERIES_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMin(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_LARGEST_SERIES_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMax(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_WINDOW_WIDTH, new double[] { ir.getDisplayMax() - ir.getDisplayMin() }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_WINDOW_CENTER, new double[] { (ir.getDisplayMax() + ir.getDisplayMin()) / 2.0 }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0020_IMAGE_POSITION, makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin).getValues()));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PIXEL_SPACING, new double[] { vd.getRowSize(true), vd.getColSize(true) }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0018_SLICE_THICKNESS, new double[] { vd.getSliceThickness(true) }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0020_IMAGE_ORIENTATION, new double[] { vecRow.x, vecRow.y, vecRow.z, vecCol.x, vecCol.y, vecCol.z }));
		} catch (final IOException ex) {
			throw new VolumeIOException(ex.getMessage());
		}

		try {
			return makeHeader(tags, id, it, XFER_SYNTAX_EXPLICIT_VT_LITLE_ENDIAN, 1);
		} catch (final UnsupportedEncodingException ex) {
			throw new VolumeIOException(ex.getMessage());
		}
	}
}
