
package edu.uthscsa.ric.volume.formats.dicom;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.vecmath.Vector3d;

import edu.uthscsa.ric.utilities.AppLogger;
import edu.uthscsa.ric.utilities.ByteUtilities;
import edu.uthscsa.ric.utilities.CollectionUtilities;
import edu.uthscsa.ric.utilities.FileUtilities;
import edu.uthscsa.ric.volume.Coordinate;
import edu.uthscsa.ric.volume.ImageDescription;
import edu.uthscsa.ric.volume.ImageDimensions;
import edu.uthscsa.ric.volume.ImageRange;
import edu.uthscsa.ric.volume.ImageTransform;
import edu.uthscsa.ric.volume.ImageType;
import edu.uthscsa.ric.volume.VolumeIOException;
import edu.uthscsa.ric.volume.VoxelDimensions;
import edu.uthscsa.ric.volume.WritableHeader;


// ftp://medical.nema.org/medical/dicom/2009/
public class ExportableSOPClassPET extends ExportableSOPClass implements WritableHeader {

	public static final String MODALITY = "PT";
	public static final String NAME = "DICOM (PET)";
	public static final String SOP_CLASS_ENHANCED_PET = "1.2.840.10008.5.1.4.1.1.130";



	@Override
	public boolean canCompress() {
		return false;
	}



	@Override
	public File formatImageFile(final File headerFile) {
		return headerFile;
	}



	@Override
	public String getMinimumVersionSupported() {
		return null;
	}



	@Override
	public String getPluginName() {
		return NAME;
	}



	@Override
	public URL getPluginURL() {
		return null;
	}



	@Override
	public String getSOPClass(final ImageType it) {
		return SOP_CLASS_ENHANCED_PET;
	}



	@Override
	public int[] getSupportedBytesForType(final int type) {
		if (type == ImageType.BYTE_TYPE_INTEGER) {
			return null;
		} else if (type == ImageType.BYTE_TYPE_INTEGER_UNSIGNED) {
			return new int[] { 2 };
		} else {
			return null;
		}
	}



	@Override
	public String[] getSupportedOrientations() {
		return CollectionUtilities.EMPTY_STRING_ARRAY;
	}



	@Override
	public int[] getSupportedTypesForBytes(final int bytes) {
		if (bytes == 1) {
			return null;
		} else if (bytes == 2) {
			return new int[] { ImageType.BYTE_TYPE_INTEGER_UNSIGNED };
		} else {
			return null;
		}
	}



	@Override
	public String getVersion() {
		return null;
	}



	@Override
	public boolean hasNewerVersion() {
		return false;
	}



	@Override
	public boolean isSupportedBigEndian() {
		return false;
	}



	@Override
	public boolean isSupportedDataScaleIntercept() {
		return true;
	}



	@Override
	public boolean isSupportedDataScaleSlope() {
		return true;
	}



	@Override
	public boolean isSupportedLittleEndian() {
		return true;
	}



	@Override
	public boolean isSupportedMultipleDataScales() {
		return false;
	}



	@Override
	public boolean isThisFormat(final File file) {
		if (!super.isThisFormat(file)) {
			return false;
		}

		BufferedInputStream fileIn = null;
		final byte[] data = new byte[256];

		try {
			fileIn = FileUtilities.getInputStream(file.toURI(), false);
			FileUtilities.readFully(fileIn, data);

			final byte[] sopClass = SOP_CLASS_ENHANCED_PET.getBytes();

			if (ByteUtilities.indexOf(data, sopClass) != -1) {
				return true;
			}

			return false;
		} catch (final IOException ex) {
			AppLogger.error(ex);
		} finally {
			try {
				if (fileIn != null) {
					fileIn.close();
				}
			} catch (final IOException ex) {
				AppLogger.warn(ex);
			}
		}

		return false;
	}



	@Override
	public boolean willWriteImage() {
		return false;
	}



	@Override
	public byte[] writeHeader(final ImageDimensions id, final VoxelDimensions vd, final ImageType it, final ImageRange ir, final ImageDescription in,
			final boolean compress, final Coordinate origin, final String orientation, final ImageTransform[] transforms, final File headerFile,
			final File imageFile, final ByteBuffer[] imageBuffer, final File tempDir) throws VolumeIOException {
		// create IDs
		final String studyUID = DICOMUtils.createUID();
		final String seriesUID = DICOMUtils.createUID();
		final String frameOfRefUID = DICOMUtils.createUID();
		final String studyId = studyNumber;
		final int seriesId = getNextSeriesNumber();
		final int imageId = getImageNumber();
		final String accessionId = getNextAccessionNumber();
		final String dimOrgUID = DICOMUtils.createUID();
		final Date now = new Date();

		return writeHeader(headerFile, id, vd, it, ir, in, origin, orientation, 0, studyUID, seriesUID, frameOfRefUID, studyId, seriesId, imageId, dimOrgUID,
				accessionId, now);
	}



	@Override
	public byte[][] writeSeriesHeaders(final ImageDimensions id, final VoxelDimensions vd, final ImageType it, final ImageRange ir, final ImageDescription in,
			final boolean compress, final Coordinate origin, final String orientation, final ImageTransform[] transforms, final File headerFile,
			final File imageFile, final ByteBuffer[] imageBuffer, final File tempDir) throws VolumeIOException {
		// create IDs
		final String studyUID = DICOMUtils.createUID();
		final String seriesUID = DICOMUtils.createUID();
		final String frameOfRefUID = DICOMUtils.createUID();
		final String studyId = studyNumber;
		final int seriesId = getNextSeriesNumber();
		int imageId = getImageNumber();
		final String accessionId = getNextAccessionNumber();
		final String dimOrgUID = DICOMUtils.createUID();
		final Date now = new Date();

		final int numSeriesPoints = id.getTimepoints();
		final byte[][] data = new byte[numSeriesPoints][];
		for (int ctr = 0; ctr < numSeriesPoints; ctr++) {
			data[ctr] = writeHeader(headerFile, id, vd, it, ir, in, origin, orientation, ctr, studyUID, seriesUID, frameOfRefUID, studyId, seriesId, imageId,
					dimOrgUID, accessionId, now);
			imageId++;
		}

		return data;
	}



	private byte[] writeHeader(final File headerFile, final ImageDimensions id, final VoxelDimensions vd, final ImageType it, final ImageRange ir,
			final ImageDescription in, final Coordinate origin, final String orientation, final int seriesIndex, final String studyUID, final String seriesUID,
			final String frameOfRefUID, final String studyId, final int seriesId, final int imageId, final String dimOrgUID, final String accessionId,
			final Date now) throws VolumeIOException {
		final Vector<Item> tags = new Vector<Item>();

		final Map<Long, String> mappedDicomTags = in.getDicomItems();

		// Enhanced PET Image Information Object Definition (A.56.1)

		try {

			// create IDs
			final String sopInstanceUID = DICOMUtils.createUID();

			// Meta Header
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_FILE_META_INFO_VERSION, new byte[] { 0, 1 }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_MEDIA_STORED_SOP_CLASS_UID, getSOPClass(it)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_MEDIA_STORED_SOP_INSTANCE_UID, sopInstanceUID));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_TRANSFER_SYNTAX_UID, XFER_SYNTAX_EXPLICIT_VT_LITLE_ENDIAN));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_IMPLEMENTATION_CLASS_UID, IMPLEMENTATION_CLASS_UID + ".1." + VERSION));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_IMPLEMENTATION_VERSION_NAME, IMPLEMENTATION_VERSION_NAME + " " + VERSION));
			addTag(tags, createExplicitTag(Dictionary.TAG_0002_GROUP_LENGTH, countSize(tags)));

			// SOP Common Module (C.12.1)
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_SOP_CLASS_UID, getSOPClass(it)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_SOP_INSTANCE_UID, sopInstanceUID));

			// Patient Module (C.7.1.1)
			addTag(tags, createTag(Dictionary.TAG_0010_PATIENT_NAME, mappedDicomTags, makePatientName(in, headerFile)));
			addTag(tags, createTag(Dictionary.TAG_0010_PATIENT_ID, mappedDicomTags, makePatientId(in, headerFile)));
			addTag(tags, createTag(Dictionary.TAG_0010_PATIENT_BIRTH_DATE, mappedDicomTags));
			addTag(tags, createTagType2(Dictionary.TAG_0010_PATIENT_SEX, mappedDicomTags));

			// General Study Module (C.7.2.1)
			addTag(tags, createTag(Dictionary.TAG_0020_STUDY_INSTANCE_ID, mappedDicomTags, studyUID));
			addTag(tags, createTag(Dictionary.TAG_0008_STUDY_DATE, mappedDicomTags, DATE_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0008_STUDY_TIME, mappedDicomTags, TIME_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0008_REFERRING_PHYSICIAN_NAME, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0020_STUDY_ID, mappedDicomTags, studyId));
			addTag(tags, createTag(Dictionary.TAG_0008_ACCESSION_NUMBER, mappedDicomTags, accessionId));
			addTag(tags, createTag(Dictionary.TAG_0008_STUDY_DESCRIPTION, mappedDicomTags));

			// General Series Module (C.7.3.1)
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_MODALITY, MODALITY));
			addTag(tags, createTag(Dictionary.TAG_0020_SERIES_INSTANCE_ID, mappedDicomTags, seriesUID));
			addTag(tags, createTag(Dictionary.TAG_0020_SERIES_ID, mappedDicomTags, new int[] { seriesId }));
			addTag(tags, createTag(Dictionary.TAG_0008_SERIES_DESCRIPTION, mappedDicomTags));

			// Enhanced PET Series Module (C.8.22.1)
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_MODALITY, MODALITY));

			// Frame of Reference Module (C.7.4.1)
			addTag(tags, createExplicitTag(Dictionary.TAG_0020_FRAME_OF_REF_ID, frameOfRefUID));
			addTag(tags, createExplicitTag(Dictionary.TAG_0020_POS_REF_INDICATOR, null));

			// General Equipment Module (C.7.5.1)
			addTag(tags, createTag(Dictionary.TAG_0008_MANUFACTURER, mappedDicomTags));

			// Enhanced General Equipment Module (C.7.5.2)
			addTag(tags, createTag(Dictionary.TAG_0008_MANUFACTURER, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_MANUFACTURER_MODEL_NAME, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_DEVICE_SERIAL_NUMBER, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_SOFTWARE_VERIONS, mappedDicomTags));

			// Image Pixel Module (C.7.6.3)
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SAMPLES_PER_PIXEL, SAMPLES_PER_PIXEL));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PHOTOMETRIC_INTERP, PHOTOMETRIC_INTERP));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_COLUMNS, id.getCols()));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_ROWS, id.getRows()));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_BITS_ALLOCATED, it.getNumBytesPerVoxel() * 8));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_BITS_STORED, it.getNumBytesPerVoxel() * 8));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_HIGH_BIT, (it.getNumBytesPerVoxel() * 8) - 1));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PIXEL_REP, PIXEL_REPRESENTATION));

			// Acquisition Context Module (C.7.6.14)
			addTag(tags, createSequenceTag(Dictionary.TAG_0040_ACQUISITION_CONTEXT_SQ, createListFromItem(null)));

			// SC Multi-frame Vector Module (C.8.6.4)
			addTag(tags,
					createExplicitTag(Dictionary.TAG_0018_SLICE_LOCATION_VECTOR,
							makeSliceLocationVector(vd.getSliceThickness(true), id.getSlices(), 1, id, vd, orientation, origin)));

			// Multi-frame Functional Groups (C.7.6.16)
			addTag(tags, createTag(Dictionary.TAG_0020_IMAGE_NUMBER, mappedDicomTags, new int[] { imageId }));
			addTag(tags, createTag(Dictionary.TAG_0008_CONTENT_DATE, mappedDicomTags, DATE_FORMAT.format(now)));
			addTag(tags, createTag(Dictionary.TAG_0008_CONTENT_TIME, mappedDicomTags, TIME_FORMAT.format(now)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_NUMBER_OF_FRAMES, new int[] { id.getSlices() }));

			{
				// Shared
				{
					// Pixel Measures Functional Group (C.7.6.16.2.1)
					final Item sliceThickness = createExplicitTag(Dictionary.TAG_0018_SLICE_THICKNESS, new double[] { vd.getSliceThickness(true) });
					final Item pixelSpacing = createExplicitTag(Dictionary.TAG_0028_PIXEL_SPACING, new double[] { vd.getRowSize(true), vd.getColSize(true) });
					sliceThickness.setNext(pixelSpacing);

					final Item pixelMeasuresSequence = createSequenceTag(Dictionary.TAG_0028_PIXEL_MEASURES_SQ, createListFromItem(sliceThickness));

					// Plane Orientation Functional Group (C.7.6.16.2.4)
					final Coordinate lastPixelFirstRow = makeRealWorldCoordinateAt(id.getCols() - 1, 0, 0, orientation, id, vd, origin);
					final Coordinate firstPixelFirstRow = makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin);
					final Vector3d vecRow = new Vector3d(lastPixelFirstRow.xDbl - firstPixelFirstRow.xDbl, lastPixelFirstRow.yDbl - firstPixelFirstRow.yDbl,
							lastPixelFirstRow.zDbl - firstPixelFirstRow.zDbl);
					vecRow.normalize();

					final Coordinate lastPixelFirstCol = makeRealWorldCoordinateAt(0, id.getRows() - 1, 0, orientation, id, vd, origin);
					final Coordinate firstPixelFirstCol = makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin);
					final Vector3d vecCol = new Vector3d(lastPixelFirstCol.xDbl - firstPixelFirstCol.xDbl, lastPixelFirstCol.yDbl - firstPixelFirstCol.yDbl,
							lastPixelFirstCol.zDbl - firstPixelFirstCol.zDbl);
					vecCol.normalize();

					final Item imageOrientation = createExplicitTag(Dictionary.TAG_0020_IMAGE_ORIENTATION, new double[] { vecRow.x, vecRow.y, vecRow.z,
							vecCol.x, vecCol.y, vecCol.z });
					final Item planeOrientationSequence = createSequenceTag(Dictionary.TAG_0020_PLANE_ORIENTATION_SQ, createListFromItem(imageOrientation));

					// Frame Anatomy Functional Group (C.7.6.16.2.8)
					final Item anatomicRegionSequence = makeCodeSequence(Dictionary.TAG_0008_ANATOMIC_REGION_SQ, mappedDicomTags);
					final Item frameLaterality = createTag(Dictionary.TAG_0020_FRAME_LATERALITY, mappedDicomTags);

					anatomicRegionSequence.setNext(frameLaterality);
					final Item frameAnatomySequence = createSequenceTag(Dictionary.TAG_0020_FRAME_ANATOMY_SQ, createListFromItem(anatomicRegionSequence));

					// Pixel Transformation Functional Group (C.7.6.16.2.9)
					final Item rescaleIntercept = createExplicitTag(Dictionary.TAG_0028_RESCALE_INTERCEPT,
							new double[] { ir.getDataScaleIntercepts()[seriesIndex * id.getSlices()] });
					final Item rescaleSlope = createExplicitTag(Dictionary.TAG_0028_RESCALE_SLOPE,
							new double[] { ir.getDataScaleSlopes()[seriesIndex * id.getSlices()] });
					final Item rescaleType = createExplicitTag(Dictionary.TAG_0028_RESCALE_TYPE, RESCALE_TYPE);

					rescaleIntercept.setNext(rescaleSlope);
					rescaleSlope.setNext(rescaleType);

					final Item pixelTransSequence = createSequenceTag(Dictionary.TAG_0028_PIXEL_TRANSFORMATION_SQ, createListFromItem(rescaleIntercept));

					// Frame VOI LUT Functional Group (C.7.6.16.2.10)
					final Item windowWidth = createExplicitTag(Dictionary.TAG_0028_WINDOW_WIDTH, new double[] { ir.getDisplayMax() - ir.getDisplayMin() });
					final Item windowCenter = createExplicitTag(Dictionary.TAG_0028_WINDOW_CENTER,
							new double[] { (ir.getDisplayMax() + ir.getDisplayMin()) / 2.0 });
					windowCenter.setNext(windowWidth);
					final Item frameVoiLutSequence = createSequenceTag(Dictionary.TAG_0028_FRAME_VOI_LUT_SQ, createListFromItem(windowCenter));

					// Real World Value Mapping Functional Group (C.7.6.16.2.11)
					final Item lutExplanation = createTag(Dictionary.TAG_0028_LUT_EXPLANATION, mappedDicomTags);
					final Item measurementUnitsCodeSequence = makeCodeSequence(Dictionary.TAG_0040_MEASUREMENT_UNITS_CODE_SQ, mappedDicomTags);
					final Item lutLabel = createTag(Dictionary.TAG_0040_LUT_LABEL, mappedDicomTags);
					final Item realWorldLastValueMapped = createExplicitTag(Dictionary.TAG_0040_REAL_WORLD_LAST_VALUE_MAPPED, (int) it.getTypeMax(true));
					final Item realWorldFirstValueMapped = createExplicitTag(Dictionary.TAG_0040_REAL_WORLD_FIRST_VALUE_MAPPED, (int) it.getTypeMin(true));
					final Item realWorldIntercept = createExplicitTag(Dictionary.TAG_0040_REAL_WORLD_VALUE_INTERCEPT, (double) ir.getDataScaleIntercepts()[0]);
					final Item realWorldSlope = createExplicitTag(Dictionary.TAG_0040_REAL_WORLD_VALUE_SLOPE, (double) ir.getDataScaleSlopes()[0]);

					lutExplanation.setNext(measurementUnitsCodeSequence);
					measurementUnitsCodeSequence.setNext(lutLabel);
					lutLabel.setNext(realWorldLastValueMapped);
					realWorldLastValueMapped.setNext(realWorldFirstValueMapped);
					realWorldFirstValueMapped.setNext(realWorldIntercept);
					realWorldIntercept.setNext(realWorldSlope);

					final Item realWorldMappingSequence = createSequenceTag(Dictionary.TAG_0040_REAL_WORLD_MAPPING_SQ, createListFromItem(lutExplanation));

					// Radiopharmaceutical Usage Functional Group (C.7.6.16.2.19)
					final Item radiopharmaAgentNumber = createTag(Dictionary.TAG_0018_RADIOPHARMA_AGENT_NUMBER, mappedDicomTags);
					final Item radiopharmaUsageSequence = createSequenceTag(Dictionary.TAG_0018_RADIOPHARMA_USAGE_SQ,
							createListFromItem(radiopharmaAgentNumber));

					// PET Frame Type Functional Group (C.8.22.5.1)
					final Item frameType = createExplicitTag(Dictionary.TAG_0008_FRAME_TYPE, CollectionUtilities.array(FRAME_TYPE));
					final Item pixelPresentation = createExplicitTag(Dictionary.TAG_0008_PIXEL_PRESENTATION, PIXEL_PRESENTATION);
					final Item volumetricProperties = createTag(Dictionary.TAG_0008_VOLUMETRIC_PROPERTIES, mappedDicomTags);
					final Item volBasedCalcTech = createTag(Dictionary.TAG_0008_VOLUME_BASED_CALCULATION_TECH, mappedDicomTags);

					frameType.setNext(pixelPresentation);
					pixelPresentation.setNext(volumetricProperties);
					volumetricProperties.setNext(volBasedCalcTech);

					final Item petFrameTypeSequence = createSequenceTag(Dictionary.TAG_0018_PET_FRAME_TYPE_SQ, createListFromItem(frameType));

					// PET Frame Acquisition Functional Group (C.8.22.5.2)
					final Item dataCollectionDiameter = createTag(Dictionary.TAG_0018_DATA_COLLECTION_DIAMETER, mappedDicomTags);
					final Item gantryDetectorTilt = createTag(Dictionary.TAG_0018_GANTRY_DETECTOR_TILT, mappedDicomTags);
					final Item gantryDetectorSkew = createTag(Dictionary.TAG_0018_GANTRY_DETECTOR_SLEW, mappedDicomTags);
					final Item tableHeight = createTag(Dictionary.TAG_0018_TABLE_HIEGHT, mappedDicomTags);

					dataCollectionDiameter.setNext(gantryDetectorTilt);
					gantryDetectorTilt.setNext(gantryDetectorSkew);
					gantryDetectorSkew.setNext(tableHeight);
					final Item petFrameAcquisitionSequence = createSequenceTag(Dictionary.TAG_0018_PET_FRAME_ACQUISITION_SQ,
							createListFromItem(dataCollectionDiameter));

					// PET Position Functional Group (C.8.22.5.4)
					final Item dataCollectionCenterPatient = createTag(Dictionary.TAG_0018_DATA_COLLECTION_CENTER_PATIENT, mappedDicomTags);
					final Item reconstructionTargetCenterPatient = createTag(Dictionary.TAG_0018_RECONSTUCTION_TARGET_CENTER_PATIENT, mappedDicomTags);
					final Item tablePosition = createTag(Dictionary.TAG_0018_TABLE_POSITION, mappedDicomTags);

					dataCollectionCenterPatient.setNext(reconstructionTargetCenterPatient);
					reconstructionTargetCenterPatient.setNext(tablePosition);
					final Item petPositionSequence = createSequenceTag(Dictionary.TAG_0018_PET_POSITION_SQ, createListFromItem(dataCollectionCenterPatient));

					// PET Frame Correction Factors Functional Group (C.8.22.5.5)
					final Item primaryCountsAccumulated = createTag(Dictionary.TAG_0054_PRIMARY_COUNTS_ACCUMULATED, mappedDicomTags);
					final Item sliceSensitivityFactor = createTag(Dictionary.TAG_0054_SLICE_SENSITIVITY_FACTOR, mappedDicomTags);
					final Item scatterFractionFactor = createTag(Dictionary.TAG_0054_SCATTER_FRACTION_FACTOR, mappedDicomTags);
					final Item deadTimeFactor = createTag(Dictionary.TAG_0054_DEAD_TIME_FACTOR, mappedDicomTags);

					primaryCountsAccumulated.setNext(sliceSensitivityFactor);
					sliceSensitivityFactor.setNext(scatterFractionFactor);
					scatterFractionFactor.setNext(deadTimeFactor);
					final Item perFrameCorrectionFactorsSequence = createSequenceTag(Dictionary.TAG_0018_PET_FRAME_CORRECTION_FACTORS_SQ,
							createListFromItem(primaryCountsAccumulated));

					// PET Reconstruction Functional Group Macro (C.8.22.5.6)
					final Item reconstructionDiameter = createTag(Dictionary.TAG_0018_RECONSTRUCTION_DIAMETER, mappedDicomTags);
					final Item reconstructionAlgorithm = createTag(Dictionary.TAG_0018_RECONSTRUCTION_ALGORITHM, mappedDicomTags);
					final Item reconstructionType = createTag(Dictionary.TAG_0018_RECONSTRUCTION_TYPE, mappedDicomTags);
					final Item iterativeReconstructionMethod = createTag(Dictionary.TAG_0018_ITERATIVE_RECONSTRUCTION_METHOD, mappedDicomTags);

					reconstructionDiameter.setNext(reconstructionAlgorithm);
					reconstructionAlgorithm.setNext(reconstructionType);
					reconstructionType.setNext(iterativeReconstructionMethod);
					final Item petReconstructionSequence = createSequenceTag(Dictionary.TAG_0018_PET_RECONSTRUCTION_SQ,
							createListFromItem(reconstructionDiameter));

					// collect shared macros
					petFrameAcquisitionSequence.setNext(petPositionSequence);
					petPositionSequence.setNext(perFrameCorrectionFactorsSequence);
					perFrameCorrectionFactorsSequence.setNext(radiopharmaUsageSequence);
					radiopharmaUsageSequence.setNext(petReconstructionSequence);
					petReconstructionSequence.setNext(petFrameTypeSequence);
					petFrameTypeSequence.setNext(frameAnatomySequence);
					frameAnatomySequence.setNext(planeOrientationSequence);
					planeOrientationSequence.setNext(pixelMeasuresSequence);
					pixelMeasuresSequence.setNext(frameVoiLutSequence);
					frameVoiLutSequence.setNext(pixelTransSequence);
					pixelTransSequence.setNext(realWorldMappingSequence);

					addTag(tags, createSequenceTag(Dictionary.TAG_5200_SHARED_FUNCTIONAL_GROUPS_SQ, createListFromItem(petFrameAcquisitionSequence)));
				}

				// Per-frame
				final List<Item> perFrameItems = new ArrayList<Item>();
				final int numSlices = id.getSlices();
				for (int ctr = 0; ctr < numSlices; ctr++) {
					// Frame Content Functional Group (C.7.6.16.2.2)
					final Item stackId = createExplicitTag(Dictionary.TAG_0020_STACK_ID, "1");
					final Item stackPosition = createExplicitTag(Dictionary.TAG_0020_IN_STACK_POSITION, ctr + 1);
					final Item temporalPosition = createExplicitTag(Dictionary.TAG_0020_TEMPORAL_POSITION_INDEX, 1);
					final Item dimIndexValues = createExplicitTag(Dictionary.TAG_0020_DIMENSION_INDEX_VALUES, ctr + 1);

					stackId.setNext(stackPosition);
					stackPosition.setNext(temporalPosition);
					temporalPosition.setNext(dimIndexValues);

					final Item frameContentSequence = createSequenceTag(Dictionary.TAG_0020_FRAME_CONTENT_SQ, createListFromItem(stackId));

					// Plane Position Functional Group (C.7.6.16.2.3)
					final Item imagePos = createExplicitTag(Dictionary.TAG_0020_IMAGE_POSITION,
							makeRealWorldCoordinateAt(0, 0, ctr, orientation, id, vd, origin).getValues());
					final Item planePositionSequence = createSequenceTag(Dictionary.TAG_0020_PLANE_POSITION_SQ, createListFromItem(imagePos));

					frameContentSequence.setNext(planePositionSequence);

					perFrameItems.add(frameContentSequence);
				}

				addTag(tags, createSequenceTag(Dictionary.TAG_5200_PER_FRAME_FUNCTIONAL_GROUPS_SQ, perFrameItems));
			}

			// Multi-frame Dimension Module (C.7.6.17)
			final Item dimOrgId = createExplicitTag(Dictionary.TAG_0020_DIMENSION_ORGANIZATION_ID, dimOrgUID);
			addTag(tags, createSequenceTag(Dictionary.TAG_0020_DIMENSION_ORGANIZATION_SQ, createListFromItem(dimOrgId)));

			final Item dimOrgId2 = createExplicitTag(Dictionary.TAG_0020_DIMENSION_ORGANIZATION_ID, dimOrgUID);
			final Item dimensionIndexPointer = createExplicitTag(Dictionary.TAG_0020_DIMENSION_INDEX_POINTER,
					new short[] { Dictionary.TAG_0020_IMAGE_POSITION.getGroupId(), Dictionary.TAG_0020_IMAGE_POSITION.getElementId() });
			final Item functionalGroupPointer = createExplicitTag(Dictionary.TAG_0020_FUNCTIONAL_GROUP_POINTER, new short[] {
					Dictionary.TAG_0020_PLANE_POSITION_SQ.getGroupId(), Dictionary.TAG_0020_PLANE_POSITION_SQ.getElementId() });

			dimOrgId2.setNext(dimensionIndexPointer);
			dimensionIndexPointer.setNext(functionalGroupPointer);
			addTag(tags, createSequenceTag(Dictionary.TAG_0020_DIMENSION_INDEX_SQ, createListFromItem(dimOrgId2)));

			// Enhanced PET Isotope Module (C.8.22.4)
			final Item radionuclideTotalDose = createTag(Dictionary.TAG_0018_RADIONUCLIDE_TOTAL_DOSE, mappedDicomTags);
			final Item radionuclideHalflife = createTag(Dictionary.TAG_0018_RADIONUCLIDE_HALF_LIFE, mappedDicomTags);
			final Item radionuclidePositronFraction = createTag(Dictionary.TAG_0018_RADIONUCLIDE_POSITRON_FRACTION, mappedDicomTags);
			final Item radiopharmaStartTime = createTag(Dictionary.TAG_0018_RADIOPHARMA_START_TIME, mappedDicomTags);
			final Item radiopharmaAgentNumber2 = createTag(Dictionary.TAG_0018_RADIOPHARMA_AGENT_NUMBER, mappedDicomTags);
			final Item radionuclideCodeSequence = makeCodeSequence(Dictionary.TAG_0054_RADIONUCLIDE_CODE_SQ, mappedDicomTags);
			final Item administrationRouteCodeSequence = makeCodeSequence(Dictionary.TAG_0054_ADMINISTRATION_ROUTE_CODE_SQ, mappedDicomTags);
			final Item radiopharmaCodeSequence = makeCodeSequence(Dictionary.TAG_0054_RADIOPHARMA_CODE_SQ, mappedDicomTags);

			radionuclideTotalDose.setNext(radionuclideHalflife);
			radionuclideHalflife.setNext(radionuclidePositronFraction);
			radionuclidePositronFraction.setNext(radiopharmaStartTime);
			radiopharmaStartTime.setNext(radiopharmaAgentNumber2);
			radiopharmaAgentNumber2.setNext(radionuclideCodeSequence);
			radionuclideCodeSequence.setNext(administrationRouteCodeSequence);
			administrationRouteCodeSequence.setNext(radiopharmaCodeSequence);

			addTag(tags, createSequenceTag(Dictionary.TAG_0054_RADIOPHARMA_INFORMATION_SQ, createListFromItem(radionuclideTotalDose)));

			// Enhanced PET Acquisition Module (C.8.22.2)
			addTag(tags, createTag(Dictionary.TAG_0018_ACQUISITION_START_CONDITON, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_ACQUISITION_TERM_CONDITON, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0054_TYPE_OF_DETECTOR_MOTION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_DETECTOR_GEOMETRY, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_TRANSVERSE_DETECTION_SEPARATION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_AXIAL_DETECTOR_DIMENSION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_COLLIMATOR_TYPE, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0054_COINCIDENCE_WINDOW_WIDTH, mappedDicomTags));

			final Item energyWindowLowerLimit = createTag(Dictionary.TAG_0054_ENERGY_WINDOW_LOWER_LIMIT, mappedDicomTags);
			final Item energyWindowUpperLimit = createTag(Dictionary.TAG_0054_ENERGY_WINDOW_UPPER_LIMIT, mappedDicomTags);

			energyWindowLowerLimit.setNext(energyWindowUpperLimit);
			addTag(tags, createSequenceTag(Dictionary.TAG_0054_ENERGY_WINDOW_RANGE_SQ, createListFromItem(energyWindowLowerLimit)));

			addTag(tags, createTag(Dictionary.TAG_0018_TABLE_MOTION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_TIME_OF_FLIGHT_INFO_USED, mappedDicomTags));
			addTag(tags, makeCodeSequence(Dictionary.TAG_0054_VIEW_CODE_SQ, mappedDicomTags));

			// Enhanced PET Image Module (C.8.22.3)
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_IMAGE_TYPE, CollectionUtilities.array(FRAME_TYPE)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0008_PIXEL_PRESENTATION, PIXEL_PRESENTATION));
			addTag(tags, createTag(Dictionary.TAG_0008_VOLUMETRIC_PROPERTIES, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_VOLUME_BASED_CALCULATION_TECH, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0008_ACQUISITION_DATETIME, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_ACQUISITION_DURATION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0054_COUNTS_SOURCE, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_DECAY_CORRECTED, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_ATTENUATION_CORRECTED, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_SCATTER_CORRECTED, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_DEAD_TIME_CORRECTED, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_GANTRY_MOTION_CORRECTED, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_PATIENT_MOTION_CORRECTED, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_COUNT_LOSS_NORMALIZATION_CORRECTED, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_RANDOMS_CORRECTED, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_NONUNIFORM_RADIO_SAMPLING_CORRECTED, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_SENSITIVITY_CALIBRATED, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0018_DETECTOR_NORMALIZATION_CORRECTION, mappedDicomTags));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SAMPLES_PER_PIXEL, SAMPLES_PER_PIXEL));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PHOTOMETRIC_INTERP, PHOTOMETRIC_INTERP));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_BITS_ALLOCATED, it.getNumBytesPerVoxel() * 8));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_BITS_STORED, it.getNumBytesPerVoxel() * 8));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_HIGH_BIT, (it.getNumBytesPerVoxel() * 8) - 1));
			addTag(tags, createTag(Dictionary.TAG_0018_CONTENT_QUALIFICATION, mappedDicomTags));
			addTag(tags, createTag(Dictionary.TAG_0028_BURNED_IN_ANNOTATION, mappedDicomTags));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_LOSSY_IMAGE_COMPRESSION, LOSSY_IMAGE_COMPRESSION));
			addTag(tags, createExplicitTag(Dictionary.TAG_2050_PRESENTATION_LUT_SHAPE, PRESENTATION_LUT_SHAPE));

			// Additional Tags
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SMALLEST_IMAGE_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMin(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_LARGEST_IMAGE_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMax(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_SMALLEST_SERIES_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMin(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_LARGEST_SERIES_PIXEL_VALUE, convertToActualPixelValue(ir.getImageMax(), ir)));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_WINDOW_WIDTH, new double[] { ir.getDisplayMax() - ir.getDisplayMin() }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_WINDOW_CENTER, new double[] { (ir.getDisplayMax() + ir.getDisplayMin()) / 2.0 }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0028_PIXEL_SPACING, new double[] { vd.getRowSize(true), vd.getColSize(true) }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0018_SLICE_THICKNESS, new double[] { vd.getSliceThickness(true) }));
			addTag(tags, createExplicitTag(Dictionary.TAG_0020_IMAGE_POSITION, makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin).getValues()));

			final Coordinate lastPixelFirstRow = makeRealWorldCoordinateAt(id.getCols() - 1, 0, 0, orientation, id, vd, origin);
			final Coordinate firstPixelFirstRow = makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin);
			final Vector3d vecRow = new Vector3d(lastPixelFirstRow.xDbl - firstPixelFirstRow.xDbl, lastPixelFirstRow.yDbl - firstPixelFirstRow.yDbl,
					lastPixelFirstRow.zDbl - firstPixelFirstRow.zDbl);
			vecRow.normalize();

			final Coordinate lastPixelFirstCol = makeRealWorldCoordinateAt(0, id.getRows() - 1, 0, orientation, id, vd, origin);
			final Coordinate firstPixelFirstCol = makeRealWorldCoordinateAt(0, 0, 0, orientation, id, vd, origin);
			final Vector3d vecCol = new Vector3d(lastPixelFirstCol.xDbl - firstPixelFirstCol.xDbl, lastPixelFirstCol.yDbl - firstPixelFirstCol.yDbl,
					lastPixelFirstCol.zDbl - firstPixelFirstCol.zDbl);
			vecCol.normalize();

			addTag(tags, createExplicitTag(Dictionary.TAG_0020_IMAGE_ORIENTATION, new double[] { vecRow.x, vecRow.y, vecRow.z, vecCol.x, vecCol.y, vecCol.z }));
		} catch (final IOException ex) {
			throw new VolumeIOException(ex.getMessage());
		}

		try {
			return makeHeader(tags, id, it, XFER_SYNTAX_EXPLICIT_VT_LITLE_ENDIAN, 1);
		} catch (final UnsupportedEncodingException ex) {
			throw new VolumeIOException(ex.getMessage());
		}
	}
}
