
package edu.uthscsa.ric.volume.formats.dicom;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;

import Jama.Matrix;
import edu.uthscsa.ric.utilities.AppLogger;
import edu.uthscsa.ric.utilities.ByteUtilities;
import edu.uthscsa.ric.utilities.CollectionUtilities;
import edu.uthscsa.ric.volume.Coordinate;
import edu.uthscsa.ric.volume.ImageDescription;
import edu.uthscsa.ric.volume.ImageDimensions;
import edu.uthscsa.ric.volume.ImageRange;
import edu.uthscsa.ric.volume.ImageType;
import edu.uthscsa.ric.volume.VoxelDimensions;


public abstract class ExportableSOPClass extends Series {

	public static final String TEXT_UNKNOWN = "UNKNOWN";

	public static final String TEXT_ASCII = "ascii";

	private static final byte[] MAGIC_NUMBER = { (byte) 68, (byte) 73, (byte) 67, (byte) 77 };

	private static int accessionNumber = 1;
	private static int seriesNumber = 1001;

	protected final DecimalFormat formatter = new DecimalFormat("#.##########################");
	protected final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyyMMdd");
	protected final SimpleDateFormat DATE_FORMAT_ALT = new SimpleDateFormat("yyyy.MM.dd");
	protected final SimpleDateFormat DATE_TIME_FORMAT = new SimpleDateFormat("yyyyMMddHHmmss");
	protected final SimpleDateFormat DATE_TIME_FORMAT_SHORT = new SimpleDateFormat("yyMMddHHmm");
	protected final SimpleDateFormat TIME_FORMAT = new SimpleDateFormat("HHmmss");
	protected final SimpleDateFormat TIME_FORMAT_ALT = new SimpleDateFormat("HH:mm:ss");
	protected final String ACCESSION_NUMBER_PREFIX;
	protected final String studyNumber = DATE_TIME_FORMAT_SHORT.format(new Date());

	public static final String EXT = ".dcm";
	public static final String IMPLEMENTATION_CLASS_UID = "1.2.826.0.1.3680043.9.3468";
	public static final String IMPLEMENTATION_VERSION_NAME = "RII-DICOM";
	public static final String LOSSY_IMAGE_COMPRESSION = "00";
	public static final String PHOTOMETRIC_INTERP = "MONOCHROME2";
	public static final String PIXEL_PRESENTATION = "MONOCHROME";
	public static final String PRESENTATION_LUT_SHAPE = "IDENTITY";
	public static final String RESCALE_TYPE = "US";
	public static final String XFER_SYNTAX_EXPLICIT_VT_LITLE_ENDIAN = "1.2.840.10008.1.2.1";
	public static final List<String> FRAME_TYPE = CollectionUtilities.immutable("ORIGINAL", "PRIMARY", "VOLUME", "NONE");
	public static final int IMAGE_NUMBER = 101;
	public static final int OB_TAG_HEADER_SIZE = 12; // group (2), item (2), vr (2), empty (2), size (4)
	public static final int PIXEL_DATA_TAG_HEADER_SIZE = 12; // group (2), item (2), vr (2), empty (2), size (4)
	public static final int PIXEL_REPRESENTATION = 0; // unsigned
	public static final int PREAMBLE_SIZE = 128;
	public static final int SAMPLES_PER_PIXEL = 1;
	public static final int STANDARD_TAG_HEADER_SIZE = 8; // group (2), item (2), vr (2), size (2)
	public static final int SUBLIST_ITEM_TAG_HEADER_SIZE = 8; // group (2), item (2), size (4)
	public static final int SUBLIST_TAG_HEADER_SIZE = 20; // group (2), item (2), vr (2), empty (2), size (4) + seq. delim (4), size (4)
	public static final String VERSION = "1.0";

	static final short[] TAG_SUBLIST_ITEM = { (short) 0xFFFE, (short) 0xE000 };
	static final short[] TAG_SUBLIST_ITEM_DELIM = { (short) 0xFFFE, (short) 0xE00D };
	static final short[] TAG_SUBLIST_SEQ_DELIM = { (short) 0xFFFE, (short) 0xE0DD };



	public static ImageType getPreferredImageType() {
		return new ImageType(2, ImageType.BYTE_TYPE_INTEGER_UNSIGNED, 2 * 8, true);
	}



	private static byte[] convertDouble(final double val) {
		final long valLong = Double.doubleToLongBits(val);

		final byte[] b = new byte[8];

		b[0] = (byte) ((valLong >> 0) & 0xFF);
		b[1] = (byte) ((valLong >> 8) & 0xFF);
		b[2] = (byte) ((valLong >> 16) & 0xFF);
		b[3] = (byte) ((valLong >> 24) & 0xFF);
		b[4] = (byte) ((valLong >> 32) & 0xFF);
		b[5] = (byte) ((valLong >> 40) & 0xFF);
		b[6] = (byte) ((valLong >> 48) & 0xFF);
		b[7] = (byte) ((valLong >> 56) & 0xFF);

		return b;
	}



	public ExportableSOPClass() {
		ACCESSION_NUMBER_PREFIX = "M" + studyNumber;
	}



	public abstract String getSOPClass(ImageType it);



	protected void addTag(final Vector<Item> tags, final Item tag) {
		if (!tags.contains(tag)) {
			tags.add(tag);
		}
	}



	protected int convertToActualPixelValue(final double val, final ImageRange ir) {
		return (int) Math.round((val - ir.getDataScaleIntercepts()[0]) / ir.getDataScaleSlopes()[0]);
	}



	protected static int countSize(final List<Item> tags) {
		int tagsTotalSize = 0;
		final Iterator<Item> iterator = tags.iterator();
		while (iterator.hasNext()) {
			final Item item = iterator.next();
			if (!item.isPixelData()) {
				tagsTotalSize += countItem(item);
			}
		}

		return tagsTotalSize;
	}



	protected static List<Item> getTagsInGroup(final List<Item> tags, final int group, final boolean excludeSize) {
		final List<Item> tagsInGroup = new ArrayList<Item>();
		for (final Item item : tags) {
			if (item.getGroup() == group) {
				if (!excludeSize || (item.getElement() != 0)) {
					tagsInGroup.add(item);
				}
			}
		}

		return tagsInGroup;
	}



	protected Item createExplicitTag(final Tag tag, final Object data) throws IOException {
		return createTag(tag, data);
	}



	protected Item createSequenceTag(final Tag tag, final List<Item> items) throws IOException {
		return createTag(tag, items);
	}



	protected Item createTag(final Tag tag, final Map<Long, String> mappedData) throws IOException {
		return createTag(tag, prepareData(tag, mappedData, null));
	}



	protected Item createTag(final Tag tag, final Tag[] tags, final Map<Long, String> mappedData) throws IOException {
		Object data = null;
		Tag tag2 = null;

		for (int ctr = 0; ctr < tags.length; ctr++) {
			tag2 = tags[ctr];
			data = prepareData(tag2, mappedData, null, (ctr == (tags.length - 1)));
			if (data != null) {
				break;
			}
		}

		return createTag(tag, data);
	}



	protected Item createTag(final Tag tag, final Map<Long, String> mappedData, final Object data) throws IOException {
		return createTag(tag, prepareData(tag, mappedData, data));
	}



	protected Item createTagType2(final Tag tag, final Map<Long, String> mappedData) throws IOException {
		return createTag(tag, prepareData(tag, mappedData, null, false));
	}



	protected int getImageNumber() {
		return IMAGE_NUMBER;
	}



	protected String getNextAccessionNumber() {
		return (ACCESSION_NUMBER_PREFIX + StringUtils.leftPad("" + accessionNumber++, 4, "0"));
	}



	protected int getNextSeriesNumber() {
		return seriesNumber++;
	}



	protected short getShortFromHex(final String str) {
		return (short) (0xFFFF & Integer.parseInt(str.substring(2), 16));
	}



	protected Item makeCodeSequence(final Tag tag, final Map<Long, String> dicomMap) throws IOException {
		return makeCodeSequence(tag, dicomMap, null);
	}



	protected static byte[] makeHeader(final List<Item> tags, final ImageDimensions id, final ImageType it, final String transferSyntax, final int numFiles)
			throws UnsupportedEncodingException {
		Collections.sort(tags, new DICOMComparator());

		final int tagsTotalSize = countSize(tags);
		final int dicomHeaderTotalSize = PREAMBLE_SIZE + MAGIC_NUMBER.length + tagsTotalSize + PIXEL_DATA_TAG_HEADER_SIZE;

		final byte[] header = new byte[dicomHeaderTotalSize];
		final ByteBuffer buffer = ByteBuffer.wrap(header);
		buffer.order(ByteOrder.LITTLE_ENDIAN);

		final boolean explicit = !transferSyntax.equals(Constants.TRANSFER_SYNTAX_ID_IMPLICIT);

		buffer.position(PREAMBLE_SIZE);
		buffer.put(MAGIC_NUMBER);

		final Iterator<Item> iterator = tags.iterator();
		while (iterator.hasNext()) {
			final Item item = iterator.next();

			if (!item.isPixelData()) {
				writeItem(buffer, item, tags, explicit);
			}
		}

		buffer.putShort(Dictionary.TAG_PIXEL_DATA.getGroupId());
		buffer.putShort(Dictionary.TAG_PIXEL_DATA.getElementId());

		if (it.getNumBytesPerVoxel() == 1) {
			buffer.put("OB".getBytes(TEXT_ASCII));
		} else if (it.getNumBytesPerVoxel() == 2) {
			buffer.put("OW".getBytes(TEXT_ASCII));
		} else if (it.getByteType() == ImageType.BYTE_TYPE_FLOAT) {
			buffer.put("OF".getBytes(TEXT_ASCII));
		}

		buffer.putShort((short) 0);

		int size = (id.getNumVoxelsVolume() / numFiles) * it.getNumBytesPerVoxel();

		if ((size % 2) != 0) {
			size += 1;
			id.setImageTrailer(1);
		}

		buffer.putInt(size);

		return header;
	}



	protected String makePatientId(final ImageDescription in, final File headerFile) {
		String patientId = in.getID();
		if (StringUtils.isBlank(patientId)) {
			String headerFileName = headerFile.getName();
			if (headerFileName.indexOf('.') != -1) {
				headerFileName = headerFileName.substring(0, headerFileName.indexOf('.'));
			}

			patientId = headerFileName.replaceAll("[^A-Za-z0-9]", "");
			if (patientId.length() > 32) {
				patientId = patientId.substring(0, 32);
			}

			patientId = patientId.toUpperCase();
			patientId += "01";
		}

		return patientId;
	}



	protected String makePatientName(final ImageDescription in, final File headerFile) {
		String patientName = in.getName();
		if (StringUtils.isBlank(patientName)) {
			String headerFileName = headerFile.getName();
			if (headerFileName.indexOf('.') != -1) {
				headerFileName = headerFileName.substring(0, headerFileName.indexOf('.'));
			}

			patientName = headerFileName.replaceAll("[^A-Za-z0-9]", " ");
			if (patientName.length() > 32) {
				patientName = patientName.substring(0, 32);
			}

			patientName = patientName.toUpperCase();
		}

		return patientName;
	}



	protected Coordinate makeRealWorldCoordinateAt(final int xLoc, final int yLoc, final int zLoc, final String orientation, final ImageDimensions id,
			final VoxelDimensions vd, final Coordinate origin) {
		final Coordinate ijk = new Coordinate(xLoc, yLoc, zLoc);
		final Coordinate xyz = convertIJKtoXYZ(ijk, orientation, origin, vd);
		final Coordinate world = convertXYZtoWorld(xyz);

		return world;
	}



	protected double[] makeSliceLocationVector(final double sliceThickness, final int numSlices, final int numSeriesPoints, final ImageDimensions id,
			final VoxelDimensions vd, final String orientation, final Coordinate origin) {
		final double[] sliceLocations = new double[numSlices * numSeriesPoints];

		int orientationSliceIndex;
		if (orientation.charAt(2) == 'X') {
			orientationSliceIndex = 0;
		} else if (orientation.charAt(2) == 'Y') {
			orientationSliceIndex = 1;
		} else {
			orientationSliceIndex = 2;
		}

		for (int ctrS = 0; ctrS < numSeriesPoints; ctrS++) {
			for (int ctr = 0; ctr < numSlices; ctr++) {
				final double[] values = makeRealWorldCoordinateAt(0, 0, ctr, orientation, id, vd, origin).getValues();
				sliceLocations[(ctrS * numSlices) + ctr] = values[orientationSliceIndex];
			}
		}

		return sliceLocations;
	}



	private Coordinate convertIJKtoXYZ(final Coordinate ijk, final String orientation, final Coordinate origin, final VoxelDimensions vd) {
		final double[][] mat = (new Matrix(DICOMUtils.convertNEMAToNiftiSForm(orientation, origin.xDbl, origin.yDbl, origin.zDbl, vd.getXSize(), vd.getYSize(),
				vd.getZSize()))).getArray();
		final double xLoc = (ijk.xDbl * mat[0][0]) + (ijk.yDbl * mat[0][1]) + (ijk.zDbl * mat[0][2]) + (mat[0][3]);
		final double yLoc = (ijk.xDbl * mat[1][0]) + (ijk.yDbl * mat[1][1]) + (ijk.zDbl * mat[1][2]) + (mat[1][3]);
		final double zLoc = (ijk.xDbl * mat[2][0]) + (ijk.yDbl * mat[2][1]) + (ijk.zDbl * mat[2][2]) + (mat[2][3]);

		return new Coordinate(xLoc, yLoc, zLoc);
	}



	private byte[] convertInteger(final int val) {
		final byte[] b = new byte[4];

		b[0] = (byte) ((val >> 0) & 0xFF);
		b[1] = (byte) ((val >> 8) & 0xFF);
		b[2] = (byte) ((val >> 16) & 0xFF);
		b[3] = (byte) ((val >> 24) & 0xFF);

		return b;
	}



	private String formatDateFromEpochString(final String str) {
		// first check if it already represents a formatted string
		if (DATE_FORMAT.toPattern().length() == str.length()) {
			return str;
		}

		long date = 0;

		try {
			date = Long.parseLong(str);
		} catch (final NumberFormatException ex) {
			if (DATE_FORMAT_ALT.toPattern().length() == str.length()) {
				return str.replace(".", "");
			}
		}

		return DATE_FORMAT.format(new Date(date));
	}



	private String formatTimeFromEpochString(final String str) {
		String time = str;

		if (time.indexOf('.') != -1) {
			time = time.substring(0, time.indexOf('.'));
		}

		if (TIME_FORMAT.toPattern().length() == time.length()) {
			return time;
		}

		long date = 0;

		try {
			date = Long.parseLong(time);
		} catch (final NumberFormatException ex) {
			if (TIME_FORMAT_ALT.toPattern().length() == time.length()) {
				return time.replace(":", "");
			}
		}

		return TIME_FORMAT.format(new Date(date));
	}



	private Object convertMappedData(final Tag tag, final String data) {
		final String vr = tag.getVr();

		try {
			if (StringUtils.isNotBlank(data)) {
				if (vr.equals("DA")) { // date string
					return formatDateFromEpochString(data);
				} else if (vr.equals("DT")) { // date time string
					return DATE_TIME_FORMAT.format(new Date(Long.parseLong(data)));
				} else if (vr.equals("TM")) { // time string
					return formatTimeFromEpochString(data);
				} else if (vr.equals("US")) { // short
					return Integer.parseInt(data);
				} else if (vr.equals("SS")) { // short
					return Integer.parseInt(data);
				} else if (vr.equals("UL")) { // integer
					return Integer.parseInt(data);
				} else if (vr.equals("FD")) { // double
					return Double.parseDouble(data);
				} else if (vr.equals("IS")) { // integer string
					final StringTokenizer tokenizer = new StringTokenizer(data);
					final int numTokens = tokenizer.countTokens();
					final int[] values = new int[numTokens];
					int index = 0;

					while (tokenizer.hasMoreTokens()) {
						values[index++] = Integer.parseInt(tokenizer.nextToken());
					}

					return values;
				} else if (vr.equals("DS")) { // decimal string
					final StringTokenizer tokenizer = new StringTokenizer(data);
					final int numTokens = tokenizer.countTokens();
					final double[] values = new double[numTokens];
					int index = 0;

					while (tokenizer.hasMoreTokens()) {
						values[index++] = Double.parseDouble(tokenizer.nextToken());
					}

					return values;
				}
			}
		} catch (final Exception ex) {
			AppLogger.error(ex);
		}

		return data;
	}



	private byte[] convertShort(final int val) {
		final byte[] b = new byte[2];

		b[0] = (byte) ((val >> 0) & 0xFF);
		b[1] = (byte) ((val >> 8) & 0xFF);

		return b;
	}



	private Coordinate convertXYZtoWorld(final Coordinate xyz) {
		final double xLoc = (0 - xyz.xDbl);
		final double yLoc = (0 - xyz.yDbl);
		final double zLoc = (xyz.zDbl - 0);

		return new Coordinate(xLoc, yLoc, zLoc);
	}



	private static int countItem(final Item item) {
		int size = 0;

		final boolean hasSublist = (item.getSublist() != null);
		final boolean isOB = item.isSpecialExplicitDataType();

		if (hasSublist) {
			size += countSizeSublist(item);
		} else if (isOB) {
			size += countSizeOB(item);
		} else {
			size += countSizeStandard(item);
		}

		final Item nextItem = item.getNext();

		if (nextItem != null) {
			size += countItem(nextItem);
		}

		return size;
	}



	private static int countSizeOB(final Item item) {
		int size = OB_TAG_HEADER_SIZE;
		final byte[] value = item.getValue();

		if (value != null) {
			size += value.length;
		}

		return size;
	}



	private static int countSizeStandard(final Item item) {
		int size = STANDARD_TAG_HEADER_SIZE;
		final byte[] value = item.getValue();

		if (value != null) {
			size += value.length;
		}

		return size;
	}



	private static int countSizeSublist(final Item item) {
		int size = SUBLIST_TAG_HEADER_SIZE;

		final List<Item> sublist = item.getSublist();

		for (final Item sublistItem : sublist) {
			size += SUBLIST_ITEM_TAG_HEADER_SIZE;
			size += countItem(sublistItem);
		}

		return size;
	}



	@SuppressWarnings("unchecked")
	private Item createTag(final Tag tag, final Object data) throws IOException {
		final int groupCode = tag.getGroup();
		final int itemCode = tag.getElement();
		final String vr = tag.getVr();
		byte[] value = null;
		List<Item> sublist = null;
		boolean padNull = false;
		boolean padString = false;

		if (data != null) {
			if (vr.equals("OB")) { // other string
				value = (byte[]) data;
				padNull = ((value.length % 2) != 0);
			} else if (vr.equals("UI")) { // uid string
				String str = (String) data;

				if (str.length() > 64) {
					str = str.substring(0, 64);
				}

				value = str.getBytes(TEXT_ASCII);
				padNull = ((value.length % 2) != 0);
			} else if (vr.equals("SH")) { // short string
				String str = (String) data;
				if (str.length() > 16) {
					str = str.substring(0, 16);
				} else if (str.length() < 16) {
					str = StringUtils.rightPad(str, 16);
				}

				value = str.getBytes(TEXT_ASCII);
			} else if (vr.equals("PN")) { // name string
				String str = (String) data;
				if (str.length() > 64) {
					str = str.substring(0, 64);
				}

				value = str.getBytes(TEXT_ASCII);
				padString = ((value.length % 2) != 0);
			} else if (vr.equals("LO")) { // long string
				String str = (String) data;
				if (str.length() > 64) {
					str = str.substring(0, 64);
				}

				value = str.getBytes(TEXT_ASCII);
				padString = ((value.length % 2) != 0);
			} else if (vr.equals("DA")) { // date string
				String str = (String) data;
				if (str.length() > 8) {
					str = str.substring(0, 8);
				} else if (str.length() < 8) {
					str = StringUtils.rightPad(str, 8);
				}

				value = str.getBytes(TEXT_ASCII);
			} else if (vr.equals("DT")) { // date time string
				String str = (String) data;
				if (str.length() > 26) {
					str = str.substring(0, 26);
				} else if (str.length() < 26) {
					str = StringUtils.rightPad(str, 26);
				}

				value = str.getBytes(TEXT_ASCII);
			} else if (vr.equals("TM")) { // time string
				String str = (String) data;
				if (str.length() > 16) {
					str = str.substring(0, 16);
				}

				value = str.getBytes(TEXT_ASCII);
				padString = ((value.length % 2) != 0);
			} else if (vr.equals("CS")) { // code string
				if (data instanceof String[]) {
					final StringBuffer sb = new StringBuffer();
					final String[] codes = (String[]) data;
					for (int ctr = 0; ctr < codes.length; ctr++) {
						if (ctr > 0) {
							sb.append('\\');
						}

						String code = codes[ctr];
						if (code.length() > 16) {
							code = code.substring(0, 16);
						}

						sb.append(code);
					}

					value = sb.toString().getBytes(TEXT_ASCII);
					padString = ((value.length % 2) != 0);
				} else {
					String str = (String) data;
					if (str.length() > 16) {
						str = str.substring(0, 16);
					}

					value = str.getBytes(TEXT_ASCII);
					padString = ((value.length % 2) != 0);
				}
			} else if (vr.equals("IS")) { // integer string
				if (data instanceof String) {
					value = data.toString().getBytes(TEXT_ASCII);
					padString = ((value.length % 2) != 0);
				} else {
					final int[] intArray = (int[]) data;
					final StringBuffer sb = new StringBuffer();

					for (int ctr = 0; ctr < intArray.length; ctr++) {
						if (ctr > 0) {
							sb.append('\\');
						}

						String intStr = "" + intArray[ctr];
						if (intStr.length() > 12) {
							intStr = intStr.substring(0, 12);
							AppLogger.warn("DicomWriter: Truncating integer! " + intArray[ctr]);
						}

						sb.append(intStr);
					}

					value = sb.toString().getBytes(TEXT_ASCII);
					padString = ((value.length % 2) != 0);
				}
			} else if (vr.equals("DS")) { // decimal string
				final double[] doubleArray = (double[]) data;
				final StringBuffer sb = new StringBuffer();

				for (int ctr = 0; ctr < doubleArray.length; ctr++) {
					if (ctr > 0) {
						sb.append('\\');
					}

					String doubleStr = formatter.format(doubleArray[ctr]);
					if (doubleStr.length() > 16) {
						doubleStr = doubleStr.substring(0, 16);
					}

					sb.append(doubleStr);
				}

				value = sb.toString().getBytes(TEXT_ASCII);
				padString = ((value.length % 2) != 0);
			} else if (vr.equals("AT")) { // tag
				final short[] dataShortArray = (short[]) data;
				value = new byte[4];

				byte[] temp = convertShort(dataShortArray[0]);
				value[0] = temp[0];
				value[1] = temp[1];

				temp = convertShort(dataShortArray[1]);
				value[2] = temp[0];
				value[3] = temp[1];
			} else if (vr.equals("US")) { // short
				final int val = (Integer) data;
				value = convertShort(val);
			} else if (vr.equals("SS")) { // short
				final int val = (Integer) data;
				value = convertShort(val);
			} else if (vr.equals("UL")) { // integer
				final int val = (Integer) data;
				value = convertInteger(val);
			} else if (vr.equals("FD")) { // double
				if (data instanceof Double) {
					final double val = (Double) data;
					value = convertDouble(val);
				} else if (data instanceof double[]) {
					final double[] array = (double[]) data;
					final byte[] doubleData = new byte[8 * array.length];

					for (int ctr = 0; ctr < array.length; ctr++) {
						final byte[] doubleAsByteArray = convertDouble(array[ctr]);
						System.arraycopy(doubleAsByteArray, 0, doubleData, ctr * 8, 8);
					}

					value = doubleData;
				}
			} else if (vr.equals("SQ")) {
				sublist = (List<Item>) data;
			}
		}

		if (padNull) {
			value = padArray(value, 0);
		} else if (padString) {
			value = padArray(value, 32);
		}

		if (value != null) {
			return new Item(groupCode, itemCode, vr, value);
		} else if (sublist != null) {
			return new Item(groupCode, itemCode, vr, sublist);
		} else {
			return new Item(groupCode, itemCode, vr);
		}
	}



	private Item makeCodeSequence(final Tag tag, final Map<Long, String> dicomMap, final String codeVal) throws IOException {
		String code = codeVal;
		final Object data = prepareData(tag, dicomMap, code);

		if (data != null) {
			code = data.toString();
		} else {
			code = TEXT_UNKNOWN;
		}

		final Item codingSchemeDesignator = createTag(Dictionary.TAG_0008_CODING_SCHEME_DESIGNATOR, "UCUM");
		final Item codeValue = createTag(Dictionary.TAG_0008_CODE_VALUE, code);
		final Item codeMeaning = createTag(Dictionary.TAG_0008_CODE_MEANING, code);

		codeValue.setNext(codingSchemeDesignator);
		codingSchemeDesignator.setNext(codeMeaning);
		return createTag(tag, createListFromItem(codeValue));
	}



	private Object makeDefaultData(final Tag tag) {
		final String vr = tag.getVr();

		if (vr.equals("OB")) { // other string
			return TEXT_UNKNOWN;
		} else if (vr.equals("UI")) { // uid string
			return "0";
		} else if (vr.equals("SH")) { // short string
			return TEXT_UNKNOWN;
		} else if (vr.equals("PN")) { // name string
			return TEXT_UNKNOWN;
		} else if (vr.equals("LO")) { // long string
			return TEXT_UNKNOWN;
		} else if (vr.equals("DA")) { // date string
			return DATE_TIME_FORMAT.format(new Date(0));
		} else if (vr.equals("DT")) { // date time string
			return DATE_TIME_FORMAT.format(new Date(0));
		} else if (vr.equals("TM")) { // time string
			return TIME_FORMAT.format(new Date(0));
		} else if (vr.equals("CS")) { // code string
			return TEXT_UNKNOWN;
		} else if (vr.equals("IS")) { // integer string
			return new int[] { 0 };
		} else if (vr.equals("DS")) { // decimal string
			return new double[] { 0 };
		} else if (vr.equals("AT")) { // tag
			return new short[2];
		} else if (vr.equals("US")) { // short
			return 0;
		} else if (vr.equals("SS")) { // short
			return 0;
		} else if (vr.equals("UL")) { // integer
			return 0;
		} else if (vr.equals("FD")) { // double
			return 0d;
		}

		return null;
	}



	private byte[] padArray(final byte[] copyFrom, final int padVal) {
		final byte[] copyTo = new byte[copyFrom.length + 1];
		System.arraycopy(copyFrom, 0, copyTo, 0, copyFrom.length);
		copyTo[copyTo.length - 1] = (byte) padVal;
		return copyTo;
	}



	private Object prepareData(final Tag tag, final Map<Long, String> dicomMap, final Object data) {
		return prepareData(tag, dicomMap, data, true);
	}



	private Object prepareData(final Tag tag, final Map<Long, String> dicomMap, final Object dataVal, final boolean useDefault) {
		Object data = dataVal;
		final long id = Tag.makeMappableDicomId(tag);

		// first, see if there is any overriding data from the read file format
		if (dicomMap != null) {
			final Object mappedData = convertMappedData(tag, dicomMap.get(id));
			if (mappedData != null) {
				return mappedData;
			}
		}

		// if not, was there data for this tag?
		if (data != null) {
			return data;
		}

		if (useDefault) {
			// ok, then is the default data mapped?
			data = Dictionary.DICOM_DEFAULT_STRING_MAP.get(Tag.makeMappableDicomId(tag));
			if (data != null) {
				return data;
			}

			// no, ok make some default data then
			return makeDefaultData(tag);
		} else {
			return null;
		}
	}



	private static void writeEmpty(final ByteBuffer buffer, final Item item, final boolean explicit) throws UnsupportedEncodingException {
		buffer.putShort((short) (0xFFFF & item.getGroup()));
		buffer.putShort((short) (0xFFFF & item.getElement()));

		if (explicit) {
			buffer.put(item.getVR().getBytes(TEXT_ASCII));
			buffer.putShort((short) 0);
		} else {
			buffer.putInt(0);
		}
	}



	private static void writeItem(final ByteBuffer buffer, final Item item, final List<Item> tags, final boolean explicitVal)
			throws UnsupportedEncodingException {
		final boolean hasSublist = (item.getSublist() != null);
		final boolean hasValue = (item.getValue() != null);
		final boolean isOB = item.isSpecialExplicitDataType();
		final boolean isGroupLength = item.isGroupLength() && (item.getGroup() != DICOM.TAG_PIXEL_DATA[0]);
		boolean explicit = explicitVal;

		if (item.getGroup() <= 0x0002) {
			buffer.order(ByteOrder.LITTLE_ENDIAN);
			explicit = true;
		} else if (item.isLittleEndian()) {
			buffer.order(ByteOrder.LITTLE_ENDIAN);
		} else {
			buffer.order(ByteOrder.BIG_ENDIAN);
		}

		if (hasSublist) {
			writeSublist(buffer, item, tags, explicit);
		} else if (isOB) {
			writeOB(buffer, item, explicit);
		} else if (isGroupLength) {
			writeGroupLength(buffer, item, tags, explicit);
		} else if (hasValue) {
			writeStandard(buffer, item, explicit);
		} else {
			writeEmpty(buffer, item, explicit);
		}

		final Item nextItem = item.getNext();

		if (nextItem != null) {
			writeItem(buffer, nextItem, tags, explicit);
		}
	}



	private static void writeOB(final ByteBuffer buffer, final Item item, final boolean explicit) throws UnsupportedEncodingException {
		final byte[] value = item.getValue();
		buffer.putShort((short) (0xFFFF & item.getGroup()));
		buffer.putShort((short) (0xFFFF & item.getElement()));

		if (explicit) {
			buffer.put(item.getVR().getBytes(TEXT_ASCII));
			buffer.putShort((short) 0);
		}

		buffer.putInt((int) item.getValueSize());

		if (value != null) {
			buffer.put(value);
		}
	}



	private static void writeStandard(final ByteBuffer buffer, final Item item, final boolean explicit) throws UnsupportedEncodingException {
		final byte[] value = item.getValue();
		buffer.putShort((short) (0xFFFF & item.getGroup()));
		buffer.putShort((short) (0xFFFF & item.getElement()));

		if (explicit) {
			buffer.put(item.getVR().getBytes(TEXT_ASCII));
			buffer.putShort((short) (0xFFFF & item.getValueSize()));
		} else {
			buffer.putInt((int) (0xFFFFFFFF & item.getValueSize()));
		}

		if (value != null) {
			buffer.put(value);
		}
	}



	private static void writeGroupLength(final ByteBuffer buffer, final Item item, final List<Item> tags, final boolean explicit)
			throws UnsupportedEncodingException {
		buffer.putShort((short) (0xFFFF & item.getGroup()));
		buffer.putShort((short) (0xFFFF & item.getElement()));

		if (explicit) {
			buffer.put(item.getVR().getBytes(TEXT_ASCII));
			buffer.putShort((short) (0xFFFF & item.getValueSize()));
		} else {
			buffer.putInt((int) (0xFFFFFFFF & item.getValueSize()));
		}

		final List<Item> groupTags = getTagsInGroup(tags, item.getGroup(), true);
		final int size = countSize(groupTags);

		if (item.isLittleEndian()) {
			buffer.put(ByteUtilities.getBytes(ByteUtilities.swap(size)));
		} else {
			buffer.put(ByteUtilities.getBytes(size));
		}
	}



	private static void writeSublist(final ByteBuffer buffer, final Item item, final List<Item> tags, final boolean explicit)
			throws UnsupportedEncodingException {
		buffer.putShort((short) (0xFFFF & item.getGroup()));
		buffer.putShort((short) (0xFFFF & item.getElement()));

		if (explicit) {
			buffer.put(item.getVR().getBytes(TEXT_ASCII));
			buffer.putShort((short) 0); // empty
		}

		buffer.putInt(-1); // undefined length

		final List<Item> sublist = item.getSublist();

		for (final Item sublistItem : sublist) {
			buffer.putShort((short) (0xFFFF & TAG_SUBLIST_ITEM[0]));
			buffer.putShort((short) (0xFFFF & TAG_SUBLIST_ITEM[1]));
			buffer.putInt(countItem(sublistItem));
			writeItem(buffer, sublistItem, tags, explicit);
		}

		buffer.putShort((short) (0xFFFF & TAG_SUBLIST_SEQ_DELIM[0]));
		buffer.putShort((short) (0xFFFF & TAG_SUBLIST_SEQ_DELIM[1]));
		buffer.putInt(0);
	}



	protected List<Item> createListFromItem(final Item item) {
		final List<Item> list = new ArrayList<Item>();

		if (item != null) {
			list.add(item);
		}

		return list;
	}
}
