
package edu.uthscsa.ric.volume.formats.dicom;

import java.math.BigInteger;
import java.util.UUID;


public class DICOMUtils {

	// https://groups.google.com/forum/#!topic/comp.protocols.dicom/r7xKburURFY
	public static String createUID() {
		return createOIDFromUUIDCanonicalHexString(UUID.randomUUID().toString());
	}



	private static String createOIDFromUUIDCanonicalHexString(final String hexString) throws IllegalArgumentException {
		final String OID_PREFIX = "2.25"; // {joint-iso-itu-t uuid(25) <uuid-single-integer-value>} 
		final UUID uuid = UUID.fromString(hexString);
		final long leastSignificantBits = uuid.getLeastSignificantBits();
		final long mostSignificantBits = uuid.getMostSignificantBits();
		BigInteger decimalValue = makeBigIntegerFromUnsignedLong(mostSignificantBits);
		decimalValue = decimalValue.shiftLeft(64);
		final BigInteger bigValueOfLeastSignificantBits = makeBigIntegerFromUnsignedLong(leastSignificantBits);
		// not add() ... do not want to introduce question of signedness of long 
		decimalValue = decimalValue.or(bigValueOfLeastSignificantBits);
		return OID_PREFIX + "." + decimalValue.toString();
	}



	private static BigInteger makeBigIntegerFromUnsignedLong(final long unsignedLongVal) {
		long unsignedLong = unsignedLongVal;

		BigInteger bigValue;
		if (unsignedLong < 0) {
			unsignedLong = unsignedLong & Long.MAX_VALUE;
			bigValue = BigInteger.valueOf(unsignedLong);
			bigValue = bigValue.setBit(63);
		} else {
			bigValue = BigInteger.valueOf(unsignedLong);
		}

		return bigValue;
	}



	public static boolean isValidOrientationString(final String string) {
		if ((string == null) || (string.length() != 6)) {
			return false;
		}

		int temp = string.toUpperCase().indexOf("X");
		if ((temp == -1) || (temp > 2) || (string.toUpperCase().lastIndexOf("X") != temp)) {
			return false;
		}

		temp = string.toUpperCase().indexOf("Y");
		if ((temp == -1) || (temp > 2) || (string.toUpperCase().lastIndexOf("Y") != temp)) {
			return false;
		}

		temp = string.toUpperCase().indexOf("Z");
		if ((temp == -1) || (temp > 2) || (string.toUpperCase().lastIndexOf("Z") != temp)) {
			return false;
		}

		if ((string.charAt(3) != '+') && (string.charAt(3) != '-')) {
			return false;
		}

		if ((string.charAt(4) != '+') && (string.charAt(4) != '-')) {
			return false;
		}

		if ((string.charAt(5) != '+') && (string.charAt(5) != '-')) {
			return false;
		}

		return true;
	}



	public static double[][] convertNEMAToNiftiSForm(final String string, final double xOrigin, final double yOrigin, final double zOrigin, final double xDim,
			final double yDim, final double zDim) {
		final double[][] affine = new double[4][4];

		if (!isValidOrientationString(string)) {
			for (int ctr = 0; ctr < 4; ctr++) {
				affine[ctr][ctr] = 1.0;
			}
		} else {
			final String xyz = string.substring(0, 3).toUpperCase();
			final String sense = string.substring(3);
			final int xIndex = xyz.indexOf('X');
			final int yIndex = xyz.indexOf('Y');
			final int zIndex = xyz.indexOf('Z');

			final double[] dims = new double[] { xDim, yDim, zDim };
			final boolean xFlip = (sense.charAt(xIndex) == '-');
			final boolean yFlip = (sense.charAt(yIndex) == '-');
			final boolean zFlip = (sense.charAt(zIndex) == '-');

			// rotation and scale
			affine[0][xIndex] = dims[xIndex] * (xFlip ? -1 : 1);
			affine[1][yIndex] = dims[yIndex] * (yFlip ? -1 : 1);
			affine[2][zIndex] = dims[zIndex] * (zFlip ? -1 : 1);

			// translation
			affine[0][3] = xOrigin * dims[xIndex] * (xFlip ? 1 : -1);
			affine[1][3] = yOrigin * dims[yIndex] * (yFlip ? 1 : -1);
			affine[2][3] = zOrigin * dims[zIndex] * (zFlip ? 1 : -1);

			affine[3][3] = 1;
		}

		return affine;
	}
}
