
package edu.uthscsa.ric.volume;

import java.io.File;
import java.net.URI;
import java.nio.ByteBuffer;


/**
 * ImageVolume is an interface that allows access to the image volume, with methods to get and put voxel values. It also provides access to the original header
 * of the volume as ReadableHeader as well as some convenience methods to retrieve image and voxel dimensions.
 */
public interface ImageVolume {

	@Deprecated
	int HEADER_TYPE_AVW = 1;
	@Deprecated
	int HEADER_TYPE_NEMA = 8;
	@Deprecated
	int HEADER_TYPE_NIFTI = 3;
	@Deprecated
	int HEADER_TYPE_WRITABLE = 1024;

	String HEADER_FORMAT_ANALYZE = "Analyze";
	String HEADER_FORMAT_DES = "DES";
	String HEADER_FORMAT_DICOM_MR = "DICOM (MR)";
	String HEADER_FORMAT_DICOM_PET = "DICOM (PET)";
	String HEADER_FORMAT_DICOM_SC = "DICOM (SC)";
	String HEADER_FORMAT_NIFTI = "NIFTI";



	/**
	 * Converts an XYZ index into an array offset based on image orientation.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @return image offset
	 */
	long convertIndexToOffset(int xLoc, int yLoc, int zLoc);



	/**
	 * Returns the center coordinate.
	 * 
	 * @return the center coordinate
	 */
	Coordinate getCenter();



	/**
	 * Returns the current series point.
	 * 
	 * @return the current series point
	 */
	int getCurrentSeriesPoint();



	/**
	 * Returns the ImageDescription.
	 * 
	 * @return the ImageDescription
	 */
	ImageDescription getImageDescription();



	/**
	 * Returns the ImageDimensions.
	 * 
	 * @return the ImageDimensions
	 */
	ImageDimensions getImageDimensions();



	/**
	 * Returns the ImageRange.
	 * 
	 * @return the ImageRange
	 */
	ImageRange getImageRange();



	/**
	 * Returns the ImageTransform.
	 * 
	 * @return the ImageTransform
	 */
	ImageTransform getImageTransform();



	/**
	 * Returns the ImageType.
	 * 
	 * @return the ImageType
	 */
	ImageType getImageType();



	/**
	 * Returns the series label for the specified index.
	 * 
	 * @param seriesIndex the series index
	 * @return the series label
	 */
	String getLabel(int seriesIndex);



	/**
	 * Returns the native integer value at a specified image index. An index of (0, 0, 0) corresponds to the LAS corner.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param seriesIndex series index
	 * @return integer value at specified image index
	 */
	long getNativeIntegerValueAtIndex(int xLoc, int yLoc, int zLoc, int seriesIndex);



	/**
	 * Returns a string representation of this volume's data orientation (e.g., XYZ+--).
	 * 
	 * @return a string representation of this volume's data orientation
	 */
	String getOrientationString();



	/**
	 * Returns the origin voxel location relative to the left-anterior-superior edge of the image.
	 * 
	 * @return the origin as a Coordinate
	 */
	Coordinate getOrigin();



	/**
	 * Returns the original offsets.
	 * 
	 * @return the offests as a Coordinate
	 */
	Coordinate getOriginNativeOffsets();



	/**
	 * Returns the raw (without data scaling) voxel value at a specified image index. An index of (0, 0, 0) corresponds to the LAS corner.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @return value at specified image index
	 */
	double getRawVoxelValueAtIndex(int xLoc, int yLoc, int zLoc);



	/**
	 * Returns the raw (without data scaling) voxel value at a specified image index and timepoint. An index of (0, 0, 0) corresponds to the LAS corner..
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param tLoc timepoint
	 * @return value at specified image index and timepoint
	 */
	double getRawVoxelValueAtIndex(int xLoc, int yLoc, int zLoc, int tLoc);



	/**
	 * Returns the raw (without data scaling) voxel value at a specified image index and timepoint in transformed volume. An index of (0, 0, 0) corresponds to
	 * the LAS corner.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param tLoc timepoint
	 * @param transform image transform to use.
	 * @return value at a specified image index and timepoint in transformed volume
	 */
	double getRawVoxelValueAtIndex(int xLoc, int yLoc, int zLoc, int tLoc, ImageTransform transform);



	/**
	 * Returns the raw (without data scaling) voxel value at a specified offset. The offset refers to the offset into the data array as it is arranged on disk.
	 * 
	 * @param offset image array offset
	 * @return value at specified offset
	 */
	double getRawVoxelValueForOffset(long offset);



	/**
	 * Returns the ReadableHeader of this ImageVolume.
	 * 
	 * @return the ReadableHeader of this ImageVolume
	 */
	ReadableHeader getReadableHeader();



	/**
	 * Returns the series length.
	 * 
	 * @return the series length
	 */
	int getSeriesLength();



	/**
	 * Returns the intercept value of the image [(0 - intercept) / scale] -- the value that should be on disk so that the screen value is 0.
	 * 
	 * @return the intercept value of the image
	 */
	double getVolumeIntercept();



	/**
	 * Returns the intercept value of the image [(intercept) / scale] -- if the value on disk is 0, this is the screen value.
	 * 
	 * @return the intercept value of the image
	 */
	double getVolumeInterceptScreen();



	/**
	 * Returns the VoxelDimensions.
	 * 
	 * @return the VoxelDimensions
	 */
	VoxelDimensions getVoxelDimensions();



	/**
	 * Returns the voxel value at a specified coordinate in real world space.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @return value at specified world space coordinate
	 */
	double getVoxelValueAtCoordinate(double xLoc, double yLoc, double zLoc);



	/**
	 * Returns the voxel value at a specified coordinate and timepoint in real world space.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param tLoc timepoint
	 * @return value at specified world space coordinate and timepoint
	 */
	double getVoxelValueAtCoordinate(double xLoc, double yLoc, double zLoc, int tLoc);



	/**
	 * Returns the voxel value at a specified coordinate and timepoint in a transformed world space.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param tLoc timepoint
	 * @param transform world space transform to use.
	 * @return value at a specified coordinate and timepoint in a transformed world space
	 */
	double getVoxelValueAtCoordinate(double xLoc, double yLoc, double zLoc, int tLoc, ImageTransform transform);



	/**
	 * Returns the voxel value at a specified image index. An index of (0, 0, 0) corresponds to the LAS corner.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @return value at specified image index
	 */
	double getVoxelValueAtIndex(int xLoc, int yLoc, int zLoc);



	/**
	 * Returns the voxel value at a specified image index and timepoint. An index of (0, 0, 0) corresponds to the LAS corner.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param tLoc timepoint
	 * @return value at specified image index and timepoint
	 */
	double getVoxelValueAtIndex(int xLoc, int yLoc, int zLoc, int tLoc);



	/**
	 * Returns the voxel value at a specified image index and timepoint in transformed volume. An index of (0, 0, 0) corresponds to the LAS corner.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param tLoc timepoint
	 * @param transform transform to apply to the image
	 * @return value at a specified image index and timepoint in transformed volume
	 */
	double getVoxelValueAtIndex(int xLoc, int yLoc, int zLoc, int tLoc, ImageTransform transform);



	/**
	 * Returns the voxel value at a specified offset. The offset refers to the offset into the data array as it is arranged on disk.
	 * 
	 * @param offset image array offset
	 * @return value at specified offset
	 */
	double getVoxelValueForOffset(long offset);



	/**
	 * Returns the work buffer (a float buffer), for use with advanced methods. Buffer data is stored in the image's natural data order. The method
	 * convertIndexToOffset() can be used to access the buffer using XYZ coordinates. Buffer data does not include data scales; it stores raw data values.<br>
	 * <br>
	 * Data scales can be fetched from ImageRange using getDataScaleSlopes() and getDataScaleIntercepts().<br>
	 * The data scale index can be calculated as (offset / (numRows * numCols)). The number of rows and columns can be gotten from ImageDimensions. If you know
	 * that the data scales are global (e.g., see hasGlobalDataScaleSlope()), than it would be unnecessary to use the array and index method.<br>
	 * <br>
	 * The method getRawVoxelValueAtOffset(offset) will give the same result as getWorkBuffer().getFloat(offset * 4).<br>
	 * <br>
	 * The method getVoxelValueAtOffset(offset) will give the same result as (getWorkBuffer().getFloat(offset * 4) * dataScaleSlope[dataScaleIndex]) +
	 * dataScaleIntercept[dataScaleIndex].
	 * 
	 * @deprecated
	 * @return image work buffer
	 */
	@Deprecated
	ByteBuffer getWorkBuffer();



	/**
	 * Returns the work buffer (a float buffer), for use with advanced methods. Buffer data is stored in the image's natural data order. The method
	 * convertIndexToOffset() can be used to access the buffer using XYZ coordinates. Buffer data does not include data scales; it stores raw data values.<br>
	 * <br>
	 * Data scales can be fetched from ImageRange using getDataScaleSlopes() and getDataScaleIntercepts().<br>
	 * The data scale index can be calculated as (offset / (numRows * numCols)). The number of rows and columns can be gotten from ImageDimensions. If you know
	 * that the data scales are global (e.g., see hasGlobalDataScaleSlope()), than it would be unnecessary to use the array and index method.<br>
	 * <br>
	 * The method getRawVoxelValueAtOffset(offset) will give the same result as getWorkBuffer().getFloat(offset * 4).<br>
	 * <br>
	 * The method getVoxelValueAtOffset(offset) will give the same result as (getWorkBuffer().getFloat(offset * 4) * dataScaleSlope[dataScaleIndex]) +
	 * dataScaleIntercept[dataScaleIndex].
	 * 
	 * @return image work buffer
	 */
	ByteBuffer[] getWorkBuffers();



	/**
	 * Returns x image dimension.
	 * 
	 * @return x image dimension
	 */
	int getXDim();



	/**
	 * Returns x voxel size.
	 * 
	 * @return x voxel size
	 */
	double getXSize();



	/**
	 * Returns y image dimension.
	 * 
	 * @return y image dimension
	 */
	int getYDim();



	/**
	 * Returns y voxel size.
	 * 
	 * @return y voxel size
	 */
	double getYSize();



	/**
	 * Returns z image dimension.
	 * 
	 * @return z image dimension
	 */
	int getZDim();



	/**
	 * Returns z voxel size.
	 * 
	 * @return z voxel size
	 */
	double getZSize();



	/**
	 * 
	 * Replaces a voxel value at a specified coordinate in real world space.
	 * 
	 * @deprecated
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param value replacement value
	 */
	@Deprecated
	void putVoxelValueAtCoordinate(double xLoc, double yLoc, double zLoc, double value);



	/**
	 * Replaces a voxel value at a specified coordinate and timepoint in real world space.
	 * 
	 * @deprecated
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param tLoc timepoint
	 * @param value replacement value
	 */
	@Deprecated
	void putVoxelValueAtCoordinate(double xLoc, double yLoc, double zLoc, int tLoc, double value);



	/**
	 * Replaces a voxel value at a specified coordinate and timepoint in transformed world space.
	 * 
	 * @deprecated
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param tLoc timepoint
	 * @param transform world space transform to use
	 * @param value replacement value
	 */
	@Deprecated
	void putVoxelValueAtCoordinate(double xLoc, double yLoc, double zLoc, int tLoc, double value, ImageTransform transform);



	/**
	 * Replaces a voxel value at a specified image index. An index of (0, 0, 0) corresponds to the LAS corner.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param value replacement value
	 */
	void putVoxelValueAtIndex(int xLoc, int yLoc, int zLoc, double value);



	/**
	 * Replaces a voxel value at a specified image index and timepoint. An index of (0, 0, 0) corresponds to the LAS corner.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param tLoc timepoint
	 * @param value replacement value
	 */
	void putVoxelValueAtIndex(int xLoc, int yLoc, int zLoc, int tLoc, double value);



	/**
	 * Replaces a voxel value at a specified image index and timepoint in transformed volume. An index of (0, 0, 0) corresponds to the LAS corner.
	 * 
	 * @param xLoc X index
	 * @param yLoc Y index
	 * @param zLoc Z index
	 * @param tLoc timepoint
	 * @param value replacement value
	 * @param transform image transform to use
	 * @deprecated
	 */
	@Deprecated
	void putVoxelValueAtIndex(int xLoc, int yLoc, int zLoc, int tLoc, double value, ImageTransform transform);



	/**
	 * Replaces a voxel value at a specified offset.
	 * 
	 * @param offset image array offset
	 * @param value replacement value
	 */
	void putVoxelValueAtOffset(int offset, double value);



	/**
	 * Reads a file into an ImageVolume.
	 * 
	 * @deprecated
	 * @param file a file to read
	 * @return the header file that was found (could be same as parameter)
	 * @throws InvalidHeaderException invalid header exception
	 * @throws Exception exception
	 */
	@Deprecated
	File readFiles(File file) throws InvalidHeaderException, Exception; // NOPMD



	/**
	 * Reads a file into an ImageVolume.
	 * 
	 * @param file a file to read
	 * @return the header file that was found (could be same as parameter)
	 * @throws InvalidHeaderException invalid header exception
	 * @throws Exception exception
	 */
	URI readFiles(URI file) throws InvalidHeaderException, Exception; // NOPMD



	/**
	 * Reads a file into an ImageVolume.
	 * 
	 * @param file a file to read
	 * @param readOnly true to load in read-only mode (will use less memory if datatype is byte or short)
	 * @return the header file that was found (could be same as parameter)
	 * @throws InvalidHeaderException invalid header exception
	 * @throws Exception exception
	 */
	URI readFiles(URI file, boolean readOnly) throws InvalidHeaderException, Exception; // NOPMD



	/**
	 * Sets the current series point.
	 * 
	 * @param point the current series point
	 */
	void setCurrentSeriesPoint(int point);



	/**
	 * Set to true to force loading this image in the current thread, instead of a background thread.
	 * 
	 * @param forceLoadInThread true to force loading this image in the current thread, false otherwise
	 */
	void setForceLoadInThead(boolean forceLoadInThread);



	/**
	 * Flag the image as dirty (needs to be saved).
	 */
	void setImageAsDirty();



	/**
	 * Sets the origin of this ImageVolume.
	 * 
	 * @param origin the new origin
	 */
	void setOrigin(Coordinate origin);



	/**
	 * Set this image as read-only. This flag is only applicable to integer images. When true, the backing data store will use the native bit depth of the
	 * image, which may be less than the default 4-byte floating point backing data store. To access read-only data use getNativeIntegerValueAtIndex().
	 * 
	 * @param readOnly true to set image as read-only, false otherwise
	 */
	void setReadOnly(boolean readOnly);



	/**
	 * Returns a string representation of this ImageVolume.
	 * 
	 * @return a string representation
	 */
	@Override
	String toString();



	/**
	 * Notifies volume that an image operation has been performed.
	 */
	void volumeOperationCompleted();



	/**
	 * Notifies volume that an image operation has been performed.
	 * 
	 * @param des a short description of the operation
	 */
	void volumeOperationCompleted(String des);



	/**
	 * Write file.
	 * 
	 * @deprecated
	 * @param headerFile file to save
	 * @param headerType header type (see header type constants such as ImageVolume.HEADER_TYPE_NIFTI)
	 * @param otherFormatName if headerType is ImageVolume.HEADER_TYPE_WRITABLE, then otherFormatName must equal a loaded WritableHeader plugin name
	 * @param it the ImageType of the output image
	 * @param applyDataScales true to multiply data scales into the image, false to leave data scale information in header
	 * @param fitPrecision true to fit the image data range to the data type range, false otherwise
	 * @throws Exception exception
	 */
	@Deprecated
	void writeFilesAs(final File headerFile, final int headerType, final String otherFormatName, final ImageType it, final boolean applyDataScales,
			final boolean fitPrecision) throws Exception; // NOPMD



	/**
	 * Write file.
	 * 
	 * @param headerFile file to save
	 * @param formatName string representation of header format (e.g., HEADER_FORMAT_NIFTI)
	 * @param it the ImageType of the output image
	 * @param orientation the orientation string (e.g., "XYZ+++") or null to use current
	 * @param applyDataScales true to multiply data scales into the image, false to leave data scale information in header
	 * @param fitPrecision true to fit the image data range to the data type range, false otherwise
	 * @throws Exception exception
	 */
	void writeFilesAs(final File headerFile, final String formatName, final ImageType it, final String orientation, final boolean applyDataScales,
			final boolean fitPrecision) throws Exception; // NOPMD
}
