
package edu.uthscsa.ric.volume;

import edu.uthscsa.ric.mango.viewerslice.VolumeManager;


/**
 * ImageDimensions contains information related to image size in columns/rows/slices and x/y/z orientations, as well as timepoints and offset.
 */
public class ImageDimensions implements Cloneable {

	private boolean tiled;
	private int cols;
	private int imageOffset;
	private int imageTrailer;
	private int rows;
	private int slices;
	private int timepoints;
	private int x;
	private int y;
	private int z;
	private int[] imageOffsets;

	public static final int SLICE_DIRECTION_AXIAL = VolumeManager.SLICE_DIRECTION_AXIAL;
	public static final int SLICE_DIRECTION_CORONAL = VolumeManager.SLICE_DIRECTION_CORONAL;
	public static final int SLICE_DIRECTION_SAGITTAL = VolumeManager.SLICE_DIRECTION_SAGITTAL;



	/**
	 * The only class constructor.
	 * 
	 * @param cols number of columns
	 * @param rows number of rows
	 * @param slices number of slices
	 * @param seriesPoints number of timepoints
	 */
	public ImageDimensions(final int cols, final int rows, final int slices, final int seriesPoints) {
		this.cols = cols;
		this.rows = rows;
		this.slices = slices;
		this.timepoints = seriesPoints;

		// sanity checks
		if (this.rows == 0) {
			this.rows = 1;
		}

		if (this.cols == 0) {
			this.cols = 1;
		}

		if (this.slices == 0) {
			this.slices = 1;
		}

		if (this.timepoints == 0) {
			this.timepoints = 1;
		}
	}



	/**
	 * Returns a copy of this object.
	 * 
	 * @return clone object
	 * @throws CloneNotSupportedException clone not supported exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}



	/**
	 * Tests equality. This method ignores offset, timepoints, and imageTrailer.
	 * 
	 * @param obj object to compare
	 * @return true if the objects are equals, false otherwise
	 * @deprecated
	 */
	@Deprecated
	public boolean equals(final ImageDimensions obj) { // NOPMD
		return equals((Object) obj);
	}



	/**
	 * Tests equality. This method ignores offset, timepoints, and imageTrailer.
	 * 
	 * @param obj object to compare
	 * @return true if the objects are equals, false otherwise
	 */
	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		final ImageDimensions other = (ImageDimensions) obj;
		if (this.cols != other.cols) {
			return false;
		}
		if (this.rows != other.rows) {
			return false;
		}
		if (this.slices != other.slices) {
			return false;
		}
		if (this.x != other.x) {
			return false;
		}
		if (this.y != other.y) {
			return false;
		}

		return (this.z == other.z);
	}



	/**
	 * Gets the number of columns in a volume of this image.
	 * 
	 * @return number of columns in a volume of this image
	 */
	public int getCols() {
		return this.cols;
	}



	/**
	 * Gets the image offset.
	 * 
	 * @return size of image offset
	 */
	public int getImageOffset() {
		return this.imageOffset;
	}



	/**
	 * Returns an array of image offsets for multi-file images.
	 * 
	 * @return an array of image ofsets
	 */
	public int[] getImageOffsets() {
		return this.imageOffsets;
	}



	/**
	 * Returns the number of bytes in the image trailer.
	 * 
	 * @return the number of bytes in the image trailer
	 */
	public int getImageTrailer() {
		return this.imageTrailer;
	}



	/**
	 * Gets the number of voxels in this image.
	 * 
	 * @deprecated
	 * @return number of voxels in this image
	 */
	@Deprecated
	public long getNumVoxels() {
		return this.rows * this.cols * this.slices * this.timepoints;
	}



	/**
	 * Gets the number of voxels in a slice of this image.
	 * 
	 * @return number of voxels in a slice of this image
	 */
	public int getNumVoxelsSlice() {
		return this.rows * this.cols;
	}



	/**
	 * Gets the number of voxels in a slice of this image.
	 * 
	 * @param sliceDirection axial, coronal, or sagittal
	 * @return number of voxels in a slice of this image
	 */
	public int getNumVoxelsSlice(final int sliceDirection) {
		if (sliceDirection == 0) {
			return this.x * this.y;
		} else if (sliceDirection == 1) {
			return this.x * this.z;
		} else if (sliceDirection == 2) {
			return this.y * this.z;
		} else {
			return getNumVoxelsSlice();
		}
	}



	/**
	 * Gets the number of voxels in this image.
	 * 
	 * @return number of voxels in this image
	 */
	public long getNumVoxelsTimeseries() {
		return this.rows * this.cols * this.slices * (long) this.timepoints;
	}



	/**
	 * Gets the number of voxels in a volume of this image.
	 * 
	 * @return number of voxels in a volume of this image
	 */
	public int getNumVoxelsVolume() {
		return this.rows * this.cols * this.slices;
	}



	/**
	 * Gets the number of rows in a volume of this image.
	 * 
	 * @return number of rows in a volume of this image
	 */
	public int getRows() {
		return this.rows;
	}



	/**
	 * Gets the number of slices in a volume of this image.
	 * 
	 * @return number of slices in a volume of this image
	 */
	public int getSlices() {
		return this.slices;
	}



	/**
	 * Gets the number of timepoints in this image.
	 * 
	 * @return number of timepoints in this image
	 */
	public int getTimepoints() {
		return this.timepoints;
	}



	/**
	 * Gets the X dimension.
	 * 
	 * @return size of X dimension
	 */
	public int getX() {
		return this.x;
	}



	/**
	 * Gets the Y dimension.
	 * 
	 * @return size of Y dimension
	 */
	public int getY() {
		return this.y;
	}



	/**
	 * Gets the Z dimension.
	 * 
	 * @return size of Z dimension
	 */
	public int getZ() {
		return this.z;
	}



	/**
	 * Computes hash code.
	 * 
	 * @return the hash code
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = (prime * result) + this.cols;
		result = (prime * result) + this.rows;
		result = (prime * result) + this.slices;
		result = (prime * result) + this.x;
		result = (prime * result) + this.y;
		result = (prime * result) + this.z;
		return result;
	}



	/**
	 * Returns true if the specified coordinate is within the range of these dimensions.
	 * 
	 * @param coord the coordinate to test
	 * @return true if the specified coordinage is within the range of these dimensions
	 */
	public boolean inRange(final Coordinate coord) {
		return ((coord.xInt >= 0) && (coord.xInt < this.x) && (coord.yInt >= 0) && (coord.yInt < this.y) && (coord.zInt >= 0) && (coord.zInt < this.z));
	}



	/**
	 * Returns true if the data is tiled (e.g., DICOM mosaic data).
	 * 
	 * @return true if the data is tiled
	 */
	public boolean isTiled() {
		return this.tiled;
	}



	/**
	 * Use to verify that columns, rows, slices, and timepoints have valid values.
	 * 
	 * @return true, if valid
	 */
	public boolean isValid() {
		return isValid(null);
	}



	/**
	 * Use to verify that columns, rows, slices, and timepoints have valid values.
	 * 
	 * @param buf stores any error messages
	 * @return true, if valid
	 */
	public boolean isValid(final StringBuffer buf) {
		boolean valid = true;

		if (!(this.cols > 0)) {
			if (buf != null) {
				buf.append("Columns =" + this.cols + "\n");
			}

			valid = false;
		}

		if (!(this.rows > 0)) {
			if (buf != null) {
				buf.append("Rows =" + this.rows + "\n");
			}

			valid = false;
		}

		if (!(this.slices > 0)) {
			if (buf != null) {
				buf.append("Slices =" + this.slices + "\n");
			}

			valid = false;
		}

		if (!(this.timepoints >= 0)) {
			if (buf != null) {
				buf.append("Series Points =" + this.timepoints + "\n");
			}

			valid = false;
		}

		return valid;
	}



	/**
	 * Sets the image offset.
	 * 
	 * @param num size of image offset
	 */
	public void setImageOffset(final int num) {
		this.imageOffset = num;
	}



	/**
	 * Sets an array of image offsets for multi-file images.
	 * 
	 * @param imageOffsets an array of image offsets
	 */
	public void setImageOffsets(final int imageOffsets[]) {
		this.imageOffsets = imageOffsets;
	}



	/**
	 * Sets the number of bytes in the image trailer
	 * 
	 * @param imageTrailer number of bytes
	 */
	public void setImageTrailer(final int imageTrailer) {
		this.imageTrailer = imageTrailer;
	}



	/**
	 * Set to true if the data is tiled.
	 * 
	 * @param tiled true if the data is tiles, false otherwise
	 */
	public void setTiled(final boolean tiled) {
		this.tiled = tiled;
	}



	/**
	 * Sets the number of timepoints in this image.
	 * 
	 * @param num number of timepoints in this image
	 */
	public void setTimepoints(final int num) {
		this.timepoints = num;
	}



	/**
	 * Set the values of this ImageDimensions to that of another.
	 * 
	 * @param id replacement ImageDimensions object
	 */
	public void setValues(final ImageDimensions id) {
		setValues(id, true);
	}



	/**
	 * Set the values of this ImageDimensions to that of another.
	 * 
	 * @param id replacement ImageDimensions object
	 * @param updateXYZ xyz values will be replaced, if true
	 */
	public void setValues(final ImageDimensions id, final boolean updateXYZ) {
		this.rows = id.rows;
		this.cols = id.cols;
		this.slices = id.slices;
		this.timepoints = id.timepoints;
		this.imageOffset = id.imageOffset;

		if (updateXYZ) {
			this.x = id.x;
			this.y = id.y;
			this.z = id.z;
		}
	}



	/**
	 * Sets the X dimension.
	 * 
	 * @param num size of X dimension
	 */
	public void setX(final int num) {
		this.x = num;
	}



	/**
	 * Sets the Y dimension.
	 * 
	 * @param num size of Y dimension
	 */
	public void setY(final int num) {
		this.y = num;
	}



	/**
	 * Sets the Z dimension.
	 * 
	 * @param num size of Z dimension
	 */
	public void setZ(final int num) {
		this.z = num;
	}



	/**
	 * Returns a string representation of this object.
	 * 
	 * @return a string representation of this object
	 */
	@Override
	public String toString() {
		return "ImageDimensions [x=" + this.x + ", y=" + this.y + ", z=" + this.z + ", rows=" + this.rows + ", cols=" + this.cols + ", slices=" + this.slices
				+ ", imageOffset=" + this.imageOffset + ", timepoints=" + this.timepoints + ", imageTrailer=" + this.imageTrailer + "]";
	}
}
