
package edu.uthscsa.ric.volume;

import java.util.List;
import java.util.Map;

import edu.uthscsa.ric.mango.MangoPluginType;


/**
 * An EditableHeader is a header with editable fields.
 * 
 * @author michaelmartinez
 *
 */
public interface EditableHeader extends MangoPluginType {

	/**
	 * A CodeOption consists of a code string and human-readable name.
	 * 
	 * @author michaelmartinez
	 *
	 */
	class CodeOption implements Comparable<EditableHeader.CodeOption> {

		private final String code;
		private final String name;



		/**
		 * Constructor.
		 * 
		 * @param code the code string
		 * @param name the human-readable name
		 */
		public CodeOption(final String code, final String name) {
			this.code = code;
			this.name = name;
		}



		/**
		 * Compare two CodeOptions.
		 * 
		 * @param codeOption the CodeOption to compare
		 * @return the comparison value
		 */
		@Override
		public int compareTo(final CodeOption codeOption) {
			return toString().compareTo(codeOption.toString());
		}



		/**
		 * Returns the code string.
		 * 
		 * @return the code string
		 */
		public String getCode() {
			return code;
		}



		/**
		 * Returns the human-readable name.
		 * 
		 * @return the human-readable name
		 */
		public String getName() {
			return name;
		}



		/**
		 * Returns a string representation.
		 * 
		 * @return a string representation
		 */
		@Override
		public String toString() {
			return getName() + " (" + getCode() + ")";
		}
	}



	/**
	 * Returns a map of field names to field values. For example, one map element may be "Datatype" : List [datatype]. Another example, "Dimensions" : List
	 * [xDim, yDim, zDim].
	 * 
	 * @return a map of field names to lists of field values
	 */
	Map<String, List<String>> getEditableFieldValues();



	/**
	 * Returns a map of field names to lists of human readable code names. For example, one map element may be "Datatype" : List[CodeOption(0,"Unknown"),
	 * CodeOption(1,"8-bit Integer"), CodeOption(2,"32-bit Float")].
	 * 
	 * @return a map of field names to lists of human readable code names
	 */
	Map<String, List<CodeOption>> getHumanReadableFieldOptions();



	/**
	 * Returns true if this header is text-editable. If so, implementation of getEditableFieldValues() is not necessary.
	 * 
	 * @return true if this header is text-editable, false otherwise
	 */
	boolean isTextEditableHeader();



	/**
	 * Sets a header field.
	 * 
	 * @param name the field name
	 * @param value the field value
	 * @param index the field index [this would correspond to the list index used in getEditableFieldValues()]
	 */
	void setEditableFieldValue(final String name, final String value, final int index);



	/**
	 * Returns a list of anonymizable fields or null if anonymization is not supported.
	 * 
	 * @return a list of anonymizable fields or null
	 */
	List<String> getAnonymizableFields();



	/**
	 * Perform anonymization. Returns a map of field names to field values.
	 * 
	 * @return a map of field names to field values
	 */
	Map<String, List<String>> anonymize();



	/**
	 * Write out the header.
	 */
	void writeHeader();
}
