
package edu.uthscsa.ric.volume;

import java.io.Serializable;


/**
 * Coordinate stores information about a point in space.
 */
public class Coordinate implements Cloneable, Serializable {

	public double xDbl;
	public double yDbl;
	public double zDbl;
	public int xInt;
	public int yInt;
	public int zInt;

	public static final double OUT_OF_RANGE = Double.NaN;
	public static final int COORDINATE_TYPE_ATLAS = 8;
	public static final int COORDINATE_TYPE_INDEX = 0;
	public static final int COORDINATE_TYPE_MM = 2;
	public static final int COORDINATE_TYPE_WORLD = 4;

	private static final long serialVersionUID = 1L;



	/**
	 * Default constructor.
	 */
	public Coordinate() {}



	/**
	 * Copy constructor.
	 * 
	 * @param coor The Coordinate object to copy.
	 */
	public Coordinate(final Coordinate coor) {
		setValues(coor);
	}



	/**
	 * Constructor.
	 * 
	 * @param x The X value.
	 * @param y The Y value.
	 * @param z The Z value.
	 */
	public Coordinate(final double x, final double y, final double z) {
		setValues(x, y, z);
	}



	/**
	 * Returns a copy of this object.
	 * 
	 * @return clone object
	 * @throws CloneNotSupportedException clone not supported exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}



	/**
	 * Returns the distance between this Coordinate and another.
	 * 
	 * @param coor the other coordinate
	 * @return The distance between this Coordinate and another.
	 */
	public double distance(final Coordinate coor) {
		final double disX = this.xDbl - coor.xDbl;
		final double disY = this.yDbl - coor.yDbl;
		final double disZ = this.zDbl - coor.zDbl;

		return Math.sqrt((disX * disX) + (disY * disY) + (disZ * disZ));
	}



	/**
	 * Tests equality.
	 * 
	 * @return true if the objects are equals, false otherwise.
	 */
	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof Coordinate)) {
			return false;
		}
		final Coordinate other = (Coordinate) obj;
		if (Double.doubleToLongBits(this.xDbl) != Double.doubleToLongBits(other.xDbl)) {
			return false;
		}
		if (this.xInt != other.xInt) {
			return false;
		}
		if (Double.doubleToLongBits(this.yDbl) != Double.doubleToLongBits(other.yDbl)) {
			return false;
		}
		if (this.yInt != other.yInt) {
			return false;
		}
		if (Double.doubleToLongBits(this.zDbl) != Double.doubleToLongBits(other.zDbl)) {
			return false;
		}

		return (this.zInt == other.zInt);
	}



	/**
	 * Returns an array of coordinate values.
	 * 
	 * @return an array of coordinate values
	 */
	public double[] getValues() {
		return new double[] { this.xDbl, this.yDbl, this.zDbl };
	}



	/**
	 * Compute hash code.
	 * 
	 * @return the hash code
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		long temp;
		temp = Double.doubleToLongBits(this.xDbl);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		result = (prime * result) + this.xInt;
		temp = Double.doubleToLongBits(this.yDbl);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		result = (prime * result) + this.yInt;
		temp = Double.doubleToLongBits(this.zDbl);
		result = (prime * result) + (int) (temp ^ (temp >>> 32));
		result = (prime * result) + this.zInt;
		return result;
	}



	/**
	 * This method can be used to check if all values are zero.
	 * 
	 * @return Returns true if all values are zero; false otherwise.
	 */
	public boolean isAllZeros() {
		return ((this.xDbl == 0) && (this.yDbl == 0) && (this.zDbl == 0));
	}



	/**
	 * Returns true if any coordinate value is NaN.
	 * 
	 * @return true if any coordinate value is NaN
	 */
	public boolean isOutOfRange() {
		return (Double.isNaN(this.xDbl) || Double.isNaN(this.yDbl) || Double.isNaN(this.zDbl));
	}



	/**
	 * Sets the values of this Coordinate to that of another Coordinate
	 * 
	 * @param coor The replacement Coordinate.
	 */
	public final void setValues(final Coordinate coor) {
		this.xDbl = coor.xDbl;
		this.yDbl = coor.yDbl;
		this.zDbl = coor.zDbl;
		this.xInt = coor.xInt;
		this.yInt = coor.yInt;
		this.zInt = coor.zInt;
	}



	/**
	 * Sets the values of this Coordinate.
	 * 
	 * @param x The X value.
	 * @param y The Y value.
	 * @param z The Z value.
	 */
	public void setValues(final double x, final double y, final double z) {
		this.xDbl = x;
		this.yDbl = y;
		this.zDbl = z;
		this.xInt = (int) x;
		this.yInt = (int) y;
		this.zInt = (int) z;
	}



	/**
	 * Sets the values of this Coordinate and rounds the integer values.
	 * 
	 * @param x The X value.
	 * @param y The Y value.
	 * @param z The Z value.
	 */
	public void setValuesRound(final double x, final double y, final double z) {
		this.xDbl = x;
		this.yDbl = y;
		this.zDbl = z;
		this.xInt = (int) Math.round(x);
		this.yInt = (int) Math.round(y);
		this.zInt = (int) Math.round(z);
	}



	/**
	 * Returns a string representation of this object.
	 * 
	 * @return a string representation of this object
	 */
	@Override
	public String toString() {
		return "Coordinate [xDbl=" + this.xDbl + ", yDbl=" + this.yDbl + ", zDbl=" + this.zDbl + ", xInt=" + this.xInt + ", yInt=" + this.yInt + ", zInt="
				+ this.zInt + "]";
	}
}
