
package edu.uthscsa.ric.roi.mask;

import edu.uthscsa.ric.volume.ImageBounds;


/**
 * A ROI data buffer.
 * 
 * @author michaelmartinez
 *
 */
public interface ROIMaskBuffer {

	/**
	 * Returns the ROI mask value at the specified offset and series index.
	 * 
	 * @param offset volume offset
	 * @param index series index
	 * @return the mask value
	 */
	long get(int offset, int index);



	/**
	 * Returns the ROI mask value at the specified offset for the current series index.
	 * 
	 * @param offset volume offset
	 * @return the mask value
	 */
	long getCurrent(int offset);



	/**
	 * Returns the maximum number of ROI colors that this buffer supports.
	 * 
	 * @return the maximum number of ROI colors that this buffer supports
	 */
	int getMaximumColors();



	/**
	 * Returns the ImageBounds of the specified mask and series index.
	 * 
	 * @param mask a bit mask
	 * @param index series index
	 * @return the ImageBounds the mask
	 */
	ImageBounds getROIBounds(long mask, int index);



	/**
	 * Returns the ImageBounds of the specified mask for the current series index.
	 * 
	 * @param mask a bit mask
	 * @return the ImageBounds of the mask
	 */
	ImageBounds getROIBoundsCurrent(long mask);



	/**
	 * Returns the current ROI series index.
	 * 
	 * @return the current ROI series index
	 */
	int getSeriesIndex();



	/**
	 * Returns the length of the ROI series.
	 * 
	 * @return the length of the ROI series
	 */
	int getSeriesLength();



	/**
	 * Returns true if the ROI color index is not found.
	 * 
	 * @param roiNum color index
	 * @return true if the ROI color index is not found
	 */
	boolean isEmptyOf(int roiNum);



	/**
	 * Sets a mask value at a specified offset and series index.
	 * 
	 * @param offset volume offset
	 * @param index series index
	 * @param val the mask value
	 */
	void put(int offset, int index, long val);



	/**
	 * Sets a mask value at a specified offset for the current series index.
	 * 
	 * @param offset volume offset
	 * @param val the mask value
	 */
	void putCurrent(int offset, long val);
}
