
package edu.uthscsa.ric.mango.viewerslice;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.image.BufferedImage;
import java.net.URI;
import java.util.List;

import javax.swing.JFrame;

import edu.uthscsa.ric.cli.ParameterNameAnnotation;
import edu.uthscsa.ric.mango.MangoData;
import edu.uthscsa.ric.mango.ProgressMeter;
import edu.uthscsa.ric.mango.SliceListener;
import edu.uthscsa.ric.mango.viewerprojection.ProjectionManager;
import edu.uthscsa.ric.mango.viewersurface.SurfaceManager;
import edu.uthscsa.ric.roi.Line;
import edu.uthscsa.ric.roi.ROI;
import edu.uthscsa.ric.roi.ROIData;
import edu.uthscsa.ric.volume.Analysis;
import edu.uthscsa.ric.volume.Cluster;
import edu.uthscsa.ric.volume.Coordinate;
import edu.uthscsa.ric.volume.Histogram;
import edu.uthscsa.ric.volume.ImageVolume;


/**
 * This class manages ImageVolume data.
 * 
 * @author michaelmartinez
 *
 */
public interface VolumeManager {

	int FILTER_TYPE_SLICE = 2;
	int FILTER_TYPE_TIME = 1;
	int FILTER_TYPE_VOLUME = 4;
	int INTERPOLATION_TYPE_LINEAR = 1;
	int INTERPOLATION_TYPE_NN = 0;
	int INTERPOLATION_TYPE_SINC = 2;
	int RANK_TYPE_MAX = 0;
	int RANK_TYPE_MEDIAN = 1;
	int RANK_TYPE_MIN = 2;
	int SELECTION_TYPE_ROI_INSIDE = 2;
	int SELECTION_TYPE_ROI_OUTSIDE = 3;
	int SELECTION_TYPE_SLICE = 1;
	int SELECTION_TYPE_VOLUME = 0;
	int SERIES_TYPE_MAX = 2;
	int SERIES_TYPE_MEAN = 1;
	int SERIES_TYPE_MIN = 3;
	int SERIES_TYPE_STDEV = 4;
	int SERIES_TYPE_SUM = 0;
	int SLICE_DIRECTION_AXIAL = 0;
	int SLICE_DIRECTION_CORONAL = 1;
	int SLICE_DIRECTION_SAGITTAL = 2;
	int SLICE_DIRECTION_TEMPORAL = 3;
	int VIDEO_FORMAT_AVI_DIB = 2;
	int VIDEO_FORMAT_AVI_MJPG = 0;
	int VIDEO_FORMAT_AVI_PNG = 1;
	int VIDEO_FORMAT_QUICKTIME_ANIMATION = 5;
	int VIDEO_FORMAT_QUICKTIME_JPEG = 3;
	int VIDEO_FORMAT_QUICKTIME_PNG = 4;
	int VIDEO_FORMAT_QUICKTIME_RAW = 6;
	int VIDEO_QUALITY_BEST = 0;
	int VIDEO_QUALITY_FASTEST = 2;
	int VIDEO_QUALITY_GOOD = 1;



	/**
	 * Adds a bookmark.
	 * 
	 * @param coord the bookmark coordinate
	 * @param label the bookmark label
	 */
	void addBookmark(Coordinate coord, String label);



	/**
	 * Adds more ROI colors (color count starts at 8, then 16, 32, 64).
	 */
	void addMoreROIColors();



	/**
	 * Adds an overlay.
	 * 
	 * @param openLocation the file location
	 * @param parametric true to open both positives and negatives, false to only open positives
	 */
	@ParameterNameAnnotation(names = { "openLocation", "parametric" })
	void addOverlay(String openLocation, boolean parametric);



	/**
	 * Overlays the base volume onto another VolumeManager.
	 * 
	 * @param manager the other VolumeManager
	 * @param colorTable the overlay color table
	 */
	@ParameterNameAnnotation(names = { "manager", "colorTable" })
	void addOverlayOnto(VolumeManager manager, String colorTable);



	/**
	 * Adds an overlay from a URL.
	 * 
	 * @param url the URL
	 */
	@ParameterNameAnnotation(names = { "url" })
	void addOverlayURL(String url);



	/**
	 * Adds a slice listener, which will be notified anytime a slice's position has changed.
	 * 
	 * @param listener the SliceListener to add
	 * @param sliceDirection the slice direction
	 */
	void addSliceListener(SliceListener listener, int sliceDirection);



	/**
	 * Associates a plugin window with this VolumeManager.
	 * 
	 * @param title the title of this plugin
	 * @param window the window
	 */
	@ParameterNameAnnotation(names = { "title", "window" })
	void associatePluginWindow(final String title, final JFrame window);



	/**
	 * Creates a ProjectionManager.
	 * 
	 * @param rankType the rank type (see ProjectionManager constants)
	 * @param sliceDirection the slice direction
	 * @param resX the X resolution
	 * @param resY the Y resolution
	 * @param resZ the Z resolution
	 * @param rotationAxis the rotation axis (the ProjectionManager constants)
	 * @param projectionQuality the projection quality (see ProjectionManager constants)
	 * @return the ProjectionManager
	 */
	@ParameterNameAnnotation(names = { "rankType", "sliceDirection", "resX", "resY", "resZ", "rotationAxis", "projectionQuality" })
	ProjectionManager buildProjection(int rankType, int sliceDirection, double resX, double resY, double resZ, int rotationAxis, int projectionQuality);



	/**
	 * Creates a SurfaceManager.
	 * 
	 * @param threshold the build threshold
	 * @param percentMax true if the threshold values represent percent of image maximum, false if they refer to actual range values
	 * @param shrinkWrap true to use shrinkwrap, false otherwise
	 * @param maximum true if the threshold represents a maximum, false otherwise
	 * @param imageSmoothing true to use image smoothing, false otherwise
	 * @param width FWHM of the image smoothing filter
	 * @param kernelSize kernel size of the image smoothing filter
	 * @param resX the output X resolution
	 * @param resY the output Y resolution
	 * @param resZ the output Z resolution
	 * @param surfaceSmoothing true to use surface smoothing, false otherwise
	 * @param pointError the point error threshold
	 * @param featureAngle the feature angle threshold
	 * @param iterations number of iterations of surface smoothing
	 * @return the SurfaceManager
	 */
	@ParameterNameAnnotation(names = { "threshold", "percentMax", "shrinkWrap", "maximum", "imageSmoothing", "width", "kernelSize", "resX", "resY", "resZ",
			"surfaceSmoothing", "pointError", "featureAngle", "iterations" })
	SurfaceManager buildSurface(double threshold, boolean percentMax, boolean shrinkWrap, boolean maximum, boolean imageSmoothing, double width, int kernelSize,
			double resX, double resY, double resZ, boolean surfaceSmoothing, double pointError, double featureAngle, int iterations);



	/**
	 * Captures a multi-slice layout.
	 * 
	 * @param saveLocation the location to save the layout image file
	 * @param rows the number of layout rows
	 * @param columns the number of layout cols
	 * @param startSlice the starting slice
	 * @param skipSlices the number of slices to skip between each slice captured
	 * @param series true to capture along a series, false to capture along the current slice direction
	 * @param useLabels true to embed slice labels, false otherwise
	 * @param useColorBars true to embed color bars, false otherwise
	 * @param useText true to embed text, false otherwise
	 * @param title the layout title
	 * @param footnote the layout footnote
	 */
	@ParameterNameAnnotation(names = { "saveLocation", "rows", "columns", "startSlice", "skipSlices", "slices/series", "useLabels", "useColorBars", "useText",
			"title", "footnote" })
	void captureLayout(String saveLocation, int rows, int columns, int startSlice, int skipSlices, boolean series, boolean useLabels, boolean useColorBars,
			boolean useText, String title, String footnote);



	/**
	 * Captures a single-slice.
	 * 
	 * @param saveLocation the location to save the image file
	 */
	@ParameterNameAnnotation(names = { "saveLocation" })
	void captureSnapshot(String saveLocation);



	/**
	 * Starts video capture.
	 * 
	 * @param saveLocation the location to save the video file
	 * @param videoFormat the video format code (see constants)
	 * @param videoQuality the video quality code (see constants)
	 * @param fps the frames per second
	 * @param animate true to automatically animate the projection, false otherwise
	 */
	@ParameterNameAnnotation(names = { "saveLocation", "videoFormat", "videoQuality", "fps", "record/animate" })
	void captureVideoStart(String saveLocation, int videoFormat, int videoQuality, int fps, boolean animate);



	/**
	 * Stops the video capture.
	 */
	void captureVideoStop();



	/**
	 * Continue a paused script.
	 */
	void continueScript();



	/**
	 * Converts index coordinate to world coordinate.
	 * 
	 * @param coor the index coordinate
	 */
	void convertIndexToWorldCoordinate(final Coordinate coor);



	/**
	 * Converts world coordinate to index coordinate.
	 * 
	 * @param coor the world coordinate
	 */
	void convertWorldToIndexCoordinate(final Coordinate coor);



	/**
	 * Creates a webpage.
	 * 
	 * @param saveLocation the location to save the webpage files
	 * @param title the title of the webpage
	 * @param useOverlays true to include overlay data in the webpage, false to only include the base volume
	 * @param useAtlas true to include atlas data in the webpage, false otherwise
	 * @param useKioskMode true to use "kiosk mode", false otherwise
	 * @param outputSingleFile true to output a single HTML file, false to output separate HTML, CSS, and JS files
	 * @deprecated
	 */
	@Deprecated
	@ParameterNameAnnotation(names = { "saveLocation", "title", "useOverlays", "useAtlas", "useKioskMode", "outputSingleFile" })
	void createWebpage(final String saveLocation, final String title, final boolean useOverlays, final boolean useAtlas, final boolean useKioskMode,
			final boolean outputSingleFile);



	/**
	 * Creates a webpage.
	 * 
	 * @param saveLocation the location to save the webpage files
	 * @param title the title of the webpage
	 * @param useOverlays true to include overlay data in the webpage, false to only include the base volume
	 * @param useAtlas true to include atlas data in the webpage, false otherwise
	 * @param useSurfaces true to include surface data, false otherwise
	 * @param useKioskMode true to use "kiosk mode", false otherwise
	 * @param outputSingleFile true to output a single HTML file, false to output separate HTML, CSS, and JS files
	 */
	@ParameterNameAnnotation(names = { "saveLocation", "title", "useOverlays", "useAtlas", "useSurfaces", "useKioskMode", "outputSingleFile" })
	void createWebpage(final String saveLocation, final String title, final boolean useOverlays, final boolean useAtlas, final boolean useSurfaces,
			final boolean useKioskMode, final boolean outputSingleFile);



	/**
	 * Decrements the slice in the specified direction.
	 * 
	 * @param direction the slice direction (see constants)
	 * @return true if the slice was able to be decremented in that direction, false otherwise
	 */
	@ParameterNameAnnotation(names = { "direction" })
	boolean decrement(int direction);



	/**
	 * Deletes the specified ROI.
	 * 
	 * @param roi the ROI to delete
	 */
	@ParameterNameAnnotation(names = { "roi" })
	void deleteROI(final ROI roi);



	/**
	 * Call to remove this VolumeManager from the application.
	 */
	void disposeVolumeManager();



	/**
	 * Returns a point to display a plugin window next to, but not in front of, the viewer window.
	 * 
	 * @param dimension the size of the plugin window
	 * @return the point to display the plugin window
	 */
	@ParameterNameAnnotation(names = { "dimension" })
	Point findGoodPluginWindowLocation(final Dimension dimension);



	/**
	 * Flips the orientation of the data in the specified dimension.
	 * 
	 * @param volume the volume to flip
	 * @param sliceDirection the slice direction to flip about (see constants)
	 */
	@ParameterNameAnnotation(names = { "volume", "sliceDirection" })
	void flipOrientation(ImageVolume volume, int sliceDirection);



	/**
	 * Returns the base volume.
	 * 
	 * @return the base volume
	 */
	ImageVolume getBaseVolume();



	/**
	 * Returns the currently selected position.
	 * 
	 * @return the currently selected position
	 */
	Coordinate getCurrentPosition();



	/**
	 * Returns the currently selected volume.
	 * 
	 * @return the currently selected volume
	 */
	ImageVolume getCurrentVolume();



	/**
	 * Returns the currently selected position in world space.
	 * 
	 * @return the currently selected position in world space
	 */
	Coordinate getCurrentWorldPosition();



	/**
	 * Returns the parent window of this viewer.
	 * 
	 * @return the parent window of this viewer
	 */
	JFrame getFrame();



	/**
	 * Returns the result of the last Histogram analysis.
	 * 
	 * @return the result of the last Histogram
	 */
	Histogram getHistogramAnalysis();



	/**
	 * Returns the URI of the file used to load the base volume of this viewer.
	 * 
	 * @return the URI
	 */
	URI getLoadedURI();



	/**
	 * Returns the URI of the file used to load the specified volume.
	 * 
	 * @param volume the ImageVolume
	 * @return the URI
	 */
	URI getLoadedURI(ImageVolume volume);



	/**
	 * Returns a rendering of the main slice view. If operating in a new thread, you may want to call setStopPaintingState() and setNavigationDisabledState()
	 * first. If several slices are needed, successive calls can use getMainImage(BufferedImage bi, int slice).
	 * 
	 * @return the rendered image
	 */
	BufferedImage getMainImage();



	/**
	 * Returns a rendering of the main slice view. If operating in a new thread, you may want to call setStopPaintingState() and setNavigationDisabledState()
	 * first. If several slices are needed, successive calls can use getMainImage(BufferedImage bi, int slice, boolean exclude).
	 * 
	 * @param exclude exclude non-image information (e.g., ROIs)
	 * @return the rendered image
	 */
	BufferedImage getMainImage(boolean exclude);



	/**
	 * Returns a rendering of the main slice view.
	 * 
	 * @param bi the image in which to store the image (typical to call after calling the no-arg method first)
	 * @param slice the slice to render
	 * @return the rendered image
	 */
	BufferedImage getMainImage(BufferedImage bi, int slice);



	/**
	 * Returns a rendering of the main slice view.
	 * 
	 * @param bi the image in which to store the image (typical to call after calling the no-arg method first)
	 * @param slice the slice to render
	 * @param exclude exclude non-image information (e.g., ROIs)
	 * @return the rendered image
	 */
	BufferedImage getMainImage(BufferedImage bi, int slice, boolean exclude);



	/**
	 * Returns a rendering of the main slice view.
	 * 
	 * @param bi the image in which to store the image (typical to call after calling the no-arg method first)
	 * @param slice the slice to render
	 * @param timeIndex the timepoint to render (note, you must manually set the timepoint back to the correct position after this method is called)
	 * @param exclude exclude non-image information (e.g., ROIs)
	 * @return the rendered image
	 */
	BufferedImage getMainImage(BufferedImage bi, int slice, int timeIndex, boolean exclude);



	/**
	 * Returns a nicely formatted title for this VolumeManager
	 * 
	 * @return a nicely formatted title for this VolumeManager
	 */
	String getManagerTitle();



	/**
	 * Returns a MangoData object.
	 * 
	 * @return a MangoData object
	 * @deprecated
	 */
	@Deprecated
	MangoData getMangoData();



	/**
	 * Returns a nice name for this volume;
	 * 
	 * @param volume the ImageVolume
	 * @return the nice name
	 */
	String getName(ImageVolume volume);



	/**
	 * Returns the origin of the base volume.
	 * 
	 * @return the origin of the base volume
	 */
	Coordinate getOrigin();



	/**
	 * Returns a list of overlays volumes.
	 * 
	 * @return a list of overlay volumes
	 */
	List<ImageVolume> getOverlays();



	/**
	 * Returns the ProjectionManager.
	 * 
	 * @return the ProjectionManager
	 */
	ProjectionManager getProjectionManager();



	/**
	 * Returns the currently selected ROI color index.
	 * 
	 * @return the currently selected ROI color index
	 */
	int getRoiColor();



	/**
	 * Returns a ROIData object.
	 * 
	 * @return a ROIData object
	 */
	ROIData getRoiData();



	/**
	 * Returns the name reference of this viewer during the recording of a script.
	 * 
	 * @return the name reference of this viewer
	 */
	String getScriptObjectName();



	/**
	 * Returns the currently selected slice direction.
	 * 
	 * @return the currently selected slice direction
	 */
	int getSliceDirection();



	/**
	 * Returns the currently selected slice number.
	 * 
	 * @return the currently selected slice number
	 */
	int getSliceNumber();



	/**
	 * Returns the SurfaceManager
	 * 
	 * @return the SurfaceManager
	 */
	SurfaceManager getSurfaceManager();



	/**
	 * Returns the Mango version.
	 * 
	 * @return the Mango version
	 */
	String getVersion();



	/**
	 * Gets the ImageVolume at the specified index.
	 * 
	 * @param index the index
	 * @return the ImageVolume
	 */
	@ParameterNameAnnotation(names = { "index" })
	ImageVolume getVolume(int index);



	/**
	 * Returns the screen display alpha value of the specified ImageVolume.
	 * 
	 * @param volume the ImageVolume
	 * @return the screen display alpha value
	 */
	double getVolumeDisplayAlpha(final ImageVolume volume);



	/**
	 * Returns the screen display color table for the specified ImageVolume.
	 * 
	 * @param volume the ImageVolume
	 * @return the screen display color table
	 */
	String getVolumeDisplayColorTable(final ImageVolume volume);



	/**
	 * Returns the screen display range maximum for the specified ImageVolume.
	 * 
	 * @param volume the ImageVolume
	 * @return the screen display range maximum
	 */
	double getVolumeDisplayRangeMax(final ImageVolume volume);



	/**
	 * Returns the screen display range minimum for the specified ImageVolume.
	 * 
	 * @param volume the ImageVolume
	 * @return the screen display range minimum
	 */
	double getVolumeDisplayRangeMin(final ImageVolume volume);



	/**
	 * Returns the index of the specified volume.
	 * 
	 * @param volume the ImageVolume
	 * @return the index
	 */
	@ParameterNameAnnotation(names = { "volume" })
	int getVolumeIndex(ImageVolume volume);



	/**
	 * Returns a list of all loaded ImageVolumes.
	 * 
	 * @return a list of all loaded ImageVolumes
	 */
	List<ImageVolume> getVolumes();



	/**
	 * Increments the slice in the specified direction.
	 * 
	 * @param direction the slice direction (see constants)
	 * @return true if the slice was able to be incremented in that direction, false otherwise
	 */
	@ParameterNameAnnotation(names = { "direction" })
	boolean increment(int direction);



	/**
	 * Returns true if the currently selected volume is an overlay.
	 * 
	 * @return true if the currently selected volume is an overlay, false if it is the base volume
	 */
	boolean isCurrentVolumeOverlay();



	/**
	 * Returns true if the currently selected volume is using a transform.
	 * 
	 * @return true if the currently selected volume is using a transform.
	 */
	boolean isCurrentVolumeUsingTransform();



	/**
	 * Returns true if the application is in multi-slice ROI mode.
	 * 
	 * @return true if the application is in multi-slice ROI mode, false otherwise
	 */
	boolean isMultiSliceMode();



	/**
	 * Returns true if the screen display of the specified ImageVolume is hidden
	 * 
	 * @param volume the ImageVolume
	 * @return true if the screen display is hidden, false if it is shown
	 */
	boolean isVolumeDisplayHidden(final ImageVolume volume);



	/**
	 * Returns true if the screen display range is negative.
	 * 
	 * @param volume the ImageVolume
	 * @return true of the screen display range is negative
	 */
	boolean isVolumeDisplayNegative(final ImageVolume volume);



	/**
	 * Returns true if the specified ImageVolume is an overlay.
	 * 
	 * @param volume the ImageVolume
	 * @return true if the specified ImageVolume is an overlay, false otherwise
	 */
	@ParameterNameAnnotation(names = { "volume" })
	boolean isVolumeOverlay(ImageVolume volume);



	/**
	 * Returns true if the specified ImageVolume is using a transform.
	 * 
	 * @param volume the ImageVolume
	 * @return true if the specified ImageVolume is using a transform, false otherwise
	 */
	@ParameterNameAnnotation(names = { "volume" })
	boolean isVolumeUsingTransform(ImageVolume volume);



	/**
	 * Returns true if the application is using world mode.
	 * 
	 * @return true if the application is using world mode
	 */
	boolean isWorldMode();



	/**
	 * Loads an ROI.
	 * 
	 * @param openLocation the location of the file to load
	 */
	@ParameterNameAnnotation(names = { "openLocation" })
	void loadROI(String openLocation);



	/**
	 * Loads an ROI from a URL.
	 * 
	 * @param url the URL
	 * @param forceImport true to force loading this in import mode, false otherwise
	 * @param forceWorld true to force loading this in world mode, false otherwise
	 */
	@ParameterNameAnnotation(names = { "url", "forceImport", "forceWorld" })
	void loadROIURL(String url, boolean forceImport, boolean forceWorld);



	/**
	 * Loads a surface.
	 * 
	 * @param openLocation the location of the file to load
	 * @return the SurfaceManager
	 */
	@ParameterNameAnnotation(names = { "openLocation" })
	SurfaceManager loadSurface(String openLocation);



	/**
	 * Logs a point in the All Results table.
	 * 
	 * @param coordinate the coordinate to log
	 * @param description a label for the point
	 */
	@ParameterNameAnnotation(names = { "coordinate", "description" })
	void logPoint(Coordinate coordinate, String description);



	/**
	 * Builds a filename based on the base volume with the specified extension.
	 * 
	 * @param ext the extension
	 * @return a filename
	 */
	String makeFilename(String ext);



	/**
	 * Creates a new ImageVolume.
	 * 
	 * @return a new ImageVolume
	 */
	ImageVolume makeNewVolume();



	/**
	 * Makes a static slice overlay for painting by a plugin.
	 * 
	 * @param sliceDirection slice direction of the main slice view
	 * @return the overlay data as an ARGB BufferedImage
	 */
	BufferedImage makePluginOverlay(int sliceDirection);



	/**
	 * Creates a new ProgressMeter.
	 * 
	 * @return a new ProgressMeter
	 */
	ProgressMeter makeProgressMeter();



	/**
	 * Pauses script execution.
	 */
	void pauseScriptForUserInput();



	/**
	 * Records a scripted plugin action.
	 * 
	 * @param pluginName the plugin name
	 * @param args the arguments
	 */
	@ParameterNameAnnotation(names = { "pluginName", "args" })
	void recordPluginAction(String pluginName, String... args);



	/**
	 * Reload this VolumeManager.
	 */
	void reload();



	/**
	 * Remove all overlays from this VolumeManager.
	 */
	void removeAllOverlays();



	/**
	 * Remove overlay from this VolumeManager.
	 * 
	 * @param overlay the volume to remove
	 */
	@ParameterNameAnnotation(names = { "overlay" })
	void removeOverlay(ImageVolume overlay);



	/**
	 * Removes all plugin overlays.
	 * 
	 * @deprecated
	 */
	@Deprecated
	void removePluginOverlays();



	/**
	 * Remove a plugin overlay.
	 * 
	 * @param pluginOverlay the BufferedImage to remove
	 * @param sliceDirection the slice direction
	 */
	@ParameterNameAnnotation(names = { "pluginOverlay", "sliceDirection" })
	void removePluginOverlay(final BufferedImage pluginOverlay, final int sliceDirection);



	/**
	 * Removes a slice listener.
	 * 
	 * @param listener the SliceListener to remove
	 * @param sliceDirection the slice direction
	 */
	@ParameterNameAnnotation(names = { "listener", "sliceDirection" })
	void removeSliceListener(SliceListener listener, int sliceDirection);



	/**
	 * Adds ROI mask data in the currently selected ROI color at the specified location.
	 * 
	 * @param coordinate the center of the new ROI mask data
	 * @param size the size in mm
	 * @param sphere true to add a sphere, false to add a cube
	 */
	@ParameterNameAnnotation(names = { "coordinate", "size", "cube/sphere" })
	void runAddROI(final Coordinate coordinate, final double size, final boolean sphere);



	/**
	 * Adds ROI mask data in the currently selected ROI color at the specified location.
	 * 
	 * @param sliceDirection the slice direction on which to add the new ROI data
	 * @param coordinate the center of the new ROI mask data
	 * @param size the size in mm
	 * @param circle true to add a circle, false to add a square
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "coordinate", "size", "square/circle" })
	void runAddROISlice(int sliceDirection, final Coordinate coordinate, final double size, final boolean circle);



	/**
	 * Runs slice statistics on all slices of the currently selected ImageVolume in the specified slice direction.
	 * 
	 * @param sliceDirection the slice direction
	 * @return a list of Analyses
	 */
	@ParameterNameAnnotation(names = { "sliceDirection" })
	List<Analysis> runAllSliceStats(int sliceDirection);



	/**
	 * Runs slice statistics on all ROI slices of the currently selected ImageVolume in the specified slice direction.
	 * 
	 * @param sliceDirection the slice direction
	 * @param roiMask the mask of the ROIs to include
	 * @param includePointsAndLines true to also analyze points and lines
	 * @param onlySelectedPoints true to only analyze selected points
	 * @param onlySelectedLines true to only analyze selected lines
	 * @return a list of Analyses
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "roiMask", "includePointsAndLines", "onlySelectedPoints", "onlySelectedLines" })
	List<Analysis> runAllSliceStatsROI(int sliceDirection, long roiMask, boolean includePointsAndLines, boolean onlySelectedPoints, boolean onlySelectedLines);



	/**
	 * Runs volume statistics on all volumes in a series of the currently selected ImageVolume.
	 * 
	 * @return a list of Analyses
	 */
	List<Analysis> runAllVolumeStats();



	/**
	 * Runs volume statistics on all ROI volumes in a series of the currently selected ImageVolume.
	 * 
	 * @param roiMask the mask of ROIs to include
	 * @param includePointsAndLines true to also analyze points and lines
	 * @param onlySelectedPoints true to only analyze selected points
	 * @param onlySelectedLines true to only analyze selected lines
	 * @return a list of Analyses
	 */
	@ParameterNameAnnotation(names = { "roiMask", "includePointsAndLines", "onlySelectedPoints", "onlySelectedLines" })
	List<Analysis> runAllVolumeStatsROI(long roiMask, boolean includePointsAndLines, boolean onlySelectedPoints, boolean onlySelectedLines);



	/**
	 * Runs a close ROI operation.
	 * 
	 * @param roiMask the mask of ROIs to include
	 * @param kernelSize the kernel size
	 */
	@ParameterNameAnnotation(names = { "roiMask", "kernelSize" })
	void runCloseROI(final long roiMask, final int kernelSize);



	/**
	 * Runs a close ROI slice operation.
	 * 
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param roiMask the mask of the ROIs to include
	 * @param kernelSize the kernel size
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "roiMask", "kernelSize" })
	void runCloseROISlice(final int sliceDirection, final int sliceNumber, final long roiMask, final int kernelSize);



	/**
	 * Runs a cluster analysis.
	 * 
	 * @param overlay the ImageVolume to analyze
	 * @param threshold the threshold
	 * @return a list of Clusters
	 */
	@ParameterNameAnnotation(names = { "overlay", "threshold" })
	List<Cluster> runClusterAnalysis(ImageVolume overlay, double threshold);



	/**
	 * Runs a ROI component anlaysis.
	 * 
	 * @param roiColorIndex the ROI color index
	 * @return a list of Clusters
	 */
	@ParameterNameAnnotation(names = { "roiColorIndex" })
	List<Cluster> runComponentAnalysisROI(int roiColorIndex);



	/**
	 * Splits ROI components into different colors [runComponentAnalysisROI() must precede this call].
	 * 
	 * @param roiColorIndex the ROI color index
	 * @param voxelCountThreshold the minimum number of voxels a component must contain in order to include in the split
	 */
	@ParameterNameAnnotation(names = { "roiColorIndex", "voxelCountThreshold" })
	void runComponentSplitROI(int roiColorIndex, int voxelCountThreshold);



	/**
	 * Runs a convex hull on the specified ROI.
	 * 
	 * @param roi the ROI
	 */
	@ParameterNameAnnotation(names = { "roi" })
	void runConvexHull(final ROI roi);



	/**
	 * Runs convex hull on the specified ROI mask.
	 * 
	 * @param roiMask the ROI mask to include
	 * @param roiOutputColorIndex the output ROI color index
	 */
	@ParameterNameAnnotation(names = { "roiMask", "roiOutputColorIndex" })
	void runConvexHullROI(long roiMask, int roiOutputColorIndex);



	/**
	 * Runs convex hull on the specified ROI mask.
	 * 
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param roiMask the ROI mask to include
	 * @param roiOutputColorIndex the output ROI color index
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "roiMask", "roiOutputColorIndex" })
	void runConvexHullROISlice(final int sliceDirection, final int sliceNumber, long roiMask, int roiOutputColorIndex);



	/**
	 * Runs a cross-section on the specified Line.
	 * 
	 * @param line the Line
	 * @param step the interpolation interval
	 * @return an array of values at the interpolated points
	 */
	@ParameterNameAnnotation(names = { "line", "step" })
	double[] runCrosssectionMM(Line line, double step);



	/**
	 * Runs a cross-section on the specified line.
	 * 
	 * @param line the Line
	 * @return an array of value of all intersected voxels
	 */
	@ParameterNameAnnotation(names = { "line" })
	double[] runCrosssectionVoxels(Line line);



	/**
	 * Runs a dilate ROI operation.
	 * 
	 * @param roiMask the ROI mask to include
	 * @param kernelSize the kernel size
	 */
	@ParameterNameAnnotation(names = { "roiMask", "kernelSize" })
	void runDilateROI(final long roiMask, final int kernelSize);



	/**
	 * Runs a dilate ROI slice operation.
	 * 
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param roiMask the ROI mask to include
	 * @param kernelSize the kernel size
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "roiMask", "kernelSize" })
	void runDilateROISlice(final int sliceDirection, final int sliceNumber, final long roiMask, final int kernelSize);



	/**
	 * Runs an erode ROI operation.
	 * 
	 * @param roiMask the ROI mask to include
	 * @param kernelSize the kernel size
	 */
	@ParameterNameAnnotation(names = { "roiMask", "kernelSize" })
	void runErodeROI(final long roiMask, final int kernelSize);



	/**
	 * Runs a erode ROI slice operation.
	 * 
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param roiMask the ROI mask to include
	 * @param kernelSize the kernel size
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "roiMask", "kernelSize" })
	void runErodeROISlice(final int sliceDirection, final int sliceNumber, final long roiMask, final int kernelSize);



	/**
	 * Runs a filter.
	 * 
	 * @param name the name of the filter
	 */
	@ParameterNameAnnotation(names = { "name" })
	void runFilter(final String name);



	/**
	 * Generate ROIs from clusters [runClusterAnalysis() must precede this call].
	 * 
	 * @param overlay the overlay ImageVolume
	 * @param voxelCountThreshold the minimum number of voxels a component must contain in order to include in the split
	 */
	@ParameterNameAnnotation(names = { "overlay", "voxelCountThreshold" })
	void runGenerateClusterROI(ImageVolume overlay, int voxelCountThreshold);



	/**
	 * Adds new ROI mask data for voxels included in this histogram.
	 * 
	 * @param hist the Histogram
	 */
	@ParameterNameAnnotation(names = { "histogram" })
	void runGenerateHistogramROI(final Histogram hist);



	/**
	 * Adds new ROI mask data for voxels included in this histogram.
	 * 
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param hist the Histogram
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "histogram" })
	void runGenerateHistogramROISlice(final int sliceDirection, final int sliceNumber, final Histogram hist);



	/**
	 * Adds new ROI mask data for voxels included in this histogram, but only within the specified ROI mask.
	 * 
	 * @param hist the Histogram
	 * @param roiMask the ROI mask to include
	 */
	@ParameterNameAnnotation(names = { "histogram", "roiMask" })
	void runGenerateHistogramWithinROI(final Histogram hist, final long roiMask);



	/**
	 * Adds new ROI mask data for voxels included in this histogram, but only within the specified ROI mask.
	 * 
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param hist the Histogram
	 * @param roiMask the ROI mask to include
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "histogram", "roiMask" })
	void runGenerateHistogramWithinROISlice(final int sliceDirection, final int sliceNumber, final Histogram hist, final long roiMask);



	/**
	 * Creates a ROI based on the intersection of a list of ImageVolumes.
	 * 
	 * @param overlays a list of ImageVolumes
	 */
	@ParameterNameAnnotation(names = { "overlays" })
	void runGenerateLogicalROI(List<ImageVolume> overlays);



	/**
	 * Runs a histogram.
	 * 
	 * @param numBins the number of histogram bins
	 * @param min the histogram bin range min
	 * @param max the histogram bin range max
	 * @param excludeZero true to exclude voxel value zero counts, false to include zeros
	 * @param isIntegerMode true to create equal size bins when working with integer data (may automatically change numBins)
	 * @return the Histogram
	 */
	@ParameterNameAnnotation(names = { "numberOfBins", "thresholdMin", "thresholdMax", "excludeZero", "integerMode" })
	Histogram runHistogram(final int numBins, final double min, final double max, final boolean excludeZero, final boolean isIntegerMode);



	/**
	 * Runs a histogram within a ROI mask.
	 * 
	 * @param numBins the number of histogram bins
	 * @param min the histogram bin range min
	 * @param max the histogram bin range max
	 * @param roiMask the ROI mask to include
	 * @param excludeZero true to exclude voxel value zero counts, false to include zeros
	 * @param integerMode true to create equal size bins when working with integer data (may automatically change numBins)
	 * @return the Histogram
	 */
	@ParameterNameAnnotation(names = { "numberOfBins", "thresholdMin", "thresholdMax", "roiMask", "excludeZero", "integerMode" })
	Histogram runHistogramROI(final int numBins, final double min, final double max, final long roiMask, final boolean excludeZero, final boolean integerMode);



	/**
	 * Runs statistics on the intersection of a list of ImageVolumes.
	 * 
	 * @param overlays a list of ImageVolumes
	 */
	@ParameterNameAnnotation(names = { "overlays" })
	Analysis runLogicalAnalysis(List<ImageVolume> overlays);



	/**
	 * Runs an open ROI operation.
	 * 
	 * @param roiMask the ROI mask to include
	 * @param kernelSize the kernel size
	 */
	@ParameterNameAnnotation(names = { "roiMask", "kernelSize" })
	void runOpenROI(final long roiMask, final int kernelSize);



	/**
	 * Runs an open ROI slice operation.
	 * 
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param roiMask the ROI mask to include
	 * @param kernelSize the kernel size
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "roiMask", "kernelSize" })
	void runOpenROISlice(final int sliceDirection, final int sliceNumber, final long roiMask, final int kernelSize);



	/**
	 * Runs an operation.
	 * 
	 * @param expression the expression, where "this" refers to the base volume (e.g., "this + 1" would add one to every voxel in the image)
	 * @param selectionArea the selection area (see constants)
	 * @param outputNewImage true to output a new image, false to output into the existing image
	 * @param allSeriesPoints true to process all series points, alse to process only the current series point
	 * @param roiSlicesOnly true to process only those slices that contain ROI data, false to process all slices
	 * @return the output VolumeManager
	 */
	@ParameterNameAnnotation(names = { "expression", "selectionArea", "outputNewImage", "allSeriesPoints", "roiSlicesOnly" })
	VolumeManager runOperation(final String expression, final int selectionArea, final boolean outputNewImage, final boolean allSeriesPoints,
			final boolean roiSlicesOnly);



	/**
	 * Runs a ROI logic operation.
	 * 
	 * @param expression the expression
	 * @param roiOutputColorIndex the output ROI color index
	 */
	@ParameterNameAnnotation(names = { "expression", "roiOutputColorIndex" })
	void runOperationROILogic(String expression, int roiOutputColorIndex);



	/**
	 * Runs a series operation.
	 * 
	 * @param startingSeriesIndex the starting series index
	 * @param endingSeriesIndex the ending series index
	 * @param seriesGroupSize the series group size (e.g., 10 series points with a group size of 5 would produce a new series with 2 points)
	 * @param statisticType the series statistic type (see constants)
	 * @return the output VolumeManager
	 */
	@ParameterNameAnnotation(names = { "startingSeriesIndex", "endingSeriesIndex", "seriesGroupSize", "statisticType" })
	VolumeManager runOperationSeriesStatistic(int startingSeriesIndex, int endingSeriesIndex, int seriesGroupSize, int statisticType);



	/**
	 * Runs a plugin.
	 * 
	 * @param pluginName the plugin name
	 * @param args the arguments
	 * @return returns an optional data object
	 */
	@ParameterNameAnnotation(names = { "pluginName", "args" })
	Object runPlugin(String pluginName, String[] args);



	/**
	 * Propagates an ROI slice.
	 * 
	 * @param sliceDirection the slice direction of the slice to propagate
	 * @param sliceNumber the slice number of the slice propagate
	 * @param startSlice the starting slice of the propagate range
	 * @param endSlice the ending slice of the propagate range
	 * @param roiMask the ROI mask to include
	 * @param roiOutputColorIndex the output ROI color index
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "startSlice", "endSlice", "roiMask", "roiOutputColorIndex" })
	void runPropagateROISlice(final int sliceDirection, final int sliceNumber, int startSlice, int endSlice, long roiMask, int roiOutputColorIndex);



	/**
	 * Creates new ROI mask data based on a range.
	 * 
	 * @param min the range minimum
	 * @param max the range maximum
	 * @param percentMax true if the min/max values represent percent of image maximum, false if they refer to actual range values
	 * @param excludeZero true to exclude zero from calculation
	 * @param makeSeriesROI true to make a series ROI
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 * @return true if new ROI mask data was added, false otherwise
	 */
	@ParameterNameAnnotation(names = { "thresholdMin", "thresholdMax", "percentMax", "excludeZero", "makeSeriesROI", "seriesUsesDynamicThreshold" })
	boolean runRangeToROI(final double min, final double max, final boolean percentMax, final boolean excludeZero, final boolean makeSeriesROI,
			final boolean seriesUsesDynamicThreshold);



	/**
	 * Creates new ROI mask data based on a range.
	 * 
	 * @param min the range minimum
	 * @param max the range maximum
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param percentMax true if the min/max values represent percent of image maximum, false if they refer to actual range values
	 * @param excludeZero true to exclude zero from calculation
	 * @param makeSeriesROI true to make a series ROI
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 * @return true if new ROI mask data was added, false otherwise
	 */
	@ParameterNameAnnotation(names = { "thresholdMin", "thresholdMax", "sliceDirection", "sliceNumber", "percentMax", "excludeZero", "makeSeriesROI",
			"seriesUsesDynamicThreshold" })
	boolean runRangeToROISlice(final double min, final double max, final int sliceDirection, final int sliceNumber, final boolean percentMax,
			final boolean excludeZero, final boolean makeSeriesROI, final boolean seriesUsesDynamicThreshold);



	/**
	 * Runs a rank filter.
	 * 
	 * @param kernelSize the kernel size
	 * @param rankType the rank type (see contants)
	 * @param filterType the filter type (see constants)
	 */
	@ParameterNameAnnotation(names = { "kernelSize", "rankType", "filterType" })
	void runRankFilter(final int kernelSize, final int rankType, final int filterType);



	/**
	 * Reflects a ROI.
	 * 
	 * @param sliceDirection the slice direction
	 * @param axis the axis of reflection (slice number)
	 * @param vertical true to reflect vertically, false horizontally
	 * @param includePointsAndLines true to reflect points and lines, false otherwise
	 * @param onlySelectedLines true to reflect only selected lines, false otherwise
	 * @param onlySelectedPoints true to reflect only selected points, false otherwise
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "axis", "horizontal/vertical", "includePointsAndLines", "onlySelectedLines", "onlySelectedPoints" })
	void runReflectROI(final int sliceDirection, final int axis, final boolean vertical, boolean includePointsAndLines, final boolean onlySelectedLines,
			final boolean onlySelectedPoints);



	/**
	 * 
	 * @param sliceDirection
	 * @param sliceNumber
	 * @param axis
	 * @param vertical
	 * @param includePointsAndLines
	 * @param onlySelectedLines
	 * @param onlySelectedPoints
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "axis", "horizontal/vertical", "includePointsAndLines", "onlySelectedLines",
			"onlySelectedPoints" })
	void runReflectROISlice(final int sliceDirection, final int sliceNumber, final int axis, final boolean vertical, boolean includePointsAndLines,
			final boolean onlySelectedLines, final boolean onlySelectedPoints);



	/**
	 * Runs a script.
	 * 
	 * @param scriptName the script name
	 */
	@ParameterNameAnnotation(names = { "scriptName" })
	void runScript(String scriptName);



	/**
	 * Runs statistics on a series ROI.
	 * 
	 * @param roi the ROI
	 * @return a list of Analyses
	 */
	@ParameterNameAnnotation(names = { "roi" })
	List<Analysis> runSeriesROI(ROI roi);



	/**
	 * Runs series statistics on the currently selected ImageVolume.
	 * 
	 * @return a series statistic
	 */
	Analysis runSeriesStats();



	/**
	 * Runs ROI series statistics.
	 * 
	 * @param roiMask the ROI mask to include
	 * @param includePointsAndLines true to also run stats on lines and points, false otherwise
	 * @param onlySelectedPoints true to run stats on only selected points, false otherwise
	 * @param onlySelectedLines true to run stats on only selected lines, false otherwise
	 * @return a list of Analyses
	 */
	@ParameterNameAnnotation(names = { "roiMask", "includePointsAndLines", "onlySelectedPoints", "onlySelectedLines" })
	List<Analysis> runSeriesStatsROI(long roiMask, boolean includePointsAndLines, boolean onlySelectedPoints, boolean onlySelectedLines);



	/**
	 * Shrink wraps (3D) a ROI mask.
	 * 
	 * @param threshold the threshold
	 * @param roiMask the roiMask to include
	 * @param roiOutputColorIndex the output ROI color index
	 * @param percentMax true if the threshold represents percent of image maximum, false if they refer to actual range values
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 */
	@ParameterNameAnnotation(names = { "threshold", "roiMask", "roiOutputColorIndex", "percentMax", "seriesUsesDynamicThreshold" })
	void runShrinkWrapROI(final double threshold, long roiMask, int roiOutputColorIndex, final boolean percentMax, final boolean seriesUsesDynamicThreshold);



	/**
	 * Shrink wraps (3D) a ROI mask.
	 * 
	 * @param roiMask the roiMask to include
	 * @param roiOutputColorIndex the output ROI color index
	 */
	@ParameterNameAnnotation(names = { "roiMask", "roiOutputColorIndex" })
	void runShrinkWrapROI(long roiMask, int roiOutputColorIndex);



	/**
	 * Shrink wraps (2D) a ROI mask in a specified slice direction.
	 * 
	 * @param threshold the threshold
	 * @param sliceDirection the slice direction
	 * @param roiMask the ROI mask to include
	 * @param roiOutputColorIndex the output ROI color index
	 * @param percentMax true if the threshold represents percent of image maximum, false if they refer to actual range values
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 */
	@ParameterNameAnnotation(names = { "threshold", "sliceDirection", "roiMask", "roiOutputColorIndex", "percentMax", "seriesUsesDynamicThreshold" })
	void runShrinkWrapROIAllSlices(final double threshold, final int sliceDirection, long roiMask, int roiOutputColorIndex, final boolean percentMax,
			final boolean seriesUsesDynamicThreshold);



	/**
	 * Shrink wraps (2D) a ROI mask in a specified slice direction.
	 * 
	 * @param sliceDirection the slice direction
	 * @param roiMask the ROI mask to include
	 * @param roiOutputColorIndex the output ROI color index
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "roiMask", "roiOutputColorIndex" })
	void runShrinkWrapROIAllSlices(final int sliceDirection, long roiMask, int roiOutputColorIndex);



	/**
	 * Shrink wraps (2D) a ROI mask in a specified slice direction.
	 * 
	 * @param threshold the threshold
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param roiMask the ROI mask to include
	 * @param roiOutputColorIndex the output ROI color index
	 * @param percentMax true if the threshold represents percent of image maximum, false if they refer to actual range values
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 */
	@ParameterNameAnnotation(names = { "threshold", "sliceDirection", "sliceNumber", "roiMask", "roiOutputColorIndex", "percentMax",
			"seriesUsesDynamicThreshold" })
	void runShrinkWrapROISlice(final double threshold, final int sliceDirection, final int sliceNumber, long roiMask, int roiOutputColorIndex,
			final boolean percentMax, final boolean seriesUsesDynamicThreshold);



	/**
	 * Shrink wraps (2D) a ROI mask in a specified slice direction.
	 * 
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param roiMask the ROI mask to include
	 * @param roiOutputColorIndex the output ROI color index
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "roiMask", "roiOutputColorIndex" })
	void runShrinkWrapROISlice(final int sliceDirection, final int sliceNumber, long roiMask, int roiOutputColorIndex);



	/**
	 * Run statistics on a slice.
	 * 
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @return the slice stat
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber" })
	Analysis runSliceStats(int sliceDirection, int sliceNumber);



	/**
	 * Runs statistics on a ROI slice.
	 * 
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param roiMask the ROI mask to include
	 * @param includePointsAndLines true to include points and lines, false otherwise
	 * @param onlySelectedPoints true to include only selected points
	 * @param onlySelectedLines true to include only selected lines
	 * @return a list of Analyses
	 */
	@ParameterNameAnnotation(names = { "sliceDirection", "sliceNumber", "roiMask", "includePointsAndLines", "onlySelectedPoints", "onlySelectedLines" })
	List<Analysis> runSliceStatsROI(final int sliceDirection, final int sliceNumber, long roiMask, boolean includePointsAndLines, boolean onlySelectedPoints,
			boolean onlySelectedLines);



	/**
	 * Smooths a ROI (only applies to Lines).
	 * 
	 * @param roi the ROI to smooth
	 */
	@ParameterNameAnnotation(names = { "roi" })
	void runSmooth(ROI roi);



	/**
	 * Runs statistics on a specified ROI.
	 * 
	 * @param roi the ROI
	 * @return the stat
	 */
	@ParameterNameAnnotation(names = { "roi" })
	Analysis runStat(final ROI roi);



	/**
	 * Adds new ROI mask data based on a threshold and in a convex hull.
	 * 
	 * @param threshold the threshold
	 * @param percentMax true if the threshold represents percent of image maximum, false if they refer to actual range values
	 * @param excludeZero true to exclude zero from calculation
	 * @param makeSeriesROI true to make a series ROI
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 */
	@ParameterNameAnnotation(names = { "threshold", "percentMax", "excludeZero", "makeSeriesROI", "seriesUsesDynamicThreshold" })
	void runThresholdToConvexHullROI(final double threshold, final boolean percentMax, final boolean excludeZero, final boolean makeSeriesROI,
			final boolean seriesUsesDynamicThreshold);



	/**
	 * Adds new ROI mask data based on a threshold and in a convex hull.
	 * 
	 * @param threshold the threshold
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param percentMax true if the threshold represents percent of image maximum, false if they refer to actual range values
	 * @param excludeZero true to exclude zero from calculation
	 * @param makeSeriesROI true to make a series ROI
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 */
	@ParameterNameAnnotation(names = { "threshold", "sliceDirection", "sliceNumber", "percentMax", "excludeZero", "makeSeriesROI",
			"seriesUsesDynamicThreshold" })
	void runThresholdToConvexHullROISlice(final double threshold, final int sliceDirection, final int sliceNumber, final boolean percentMax,
			final boolean excludeZero, final boolean makeSeriesROI, final boolean seriesUsesDynamicThreshold);



	/**
	 * Adds new ROI mask data based on a threshold.
	 * 
	 * @param threshold the threshold
	 * @param percentMax true if the threshold represents percent of image maximum, false if they refer to actual range values
	 * @param excludeZero true to exclude zero from calculation
	 * @param makeSeriesROI true to make a series ROI
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 * @return true if new ROI mask data was added, false otherwise
	 */
	@ParameterNameAnnotation(names = { "threshold", "percentMax", "excludeZero", "makeSeriesROI", "seriesUsesDynamicThreshold" })
	boolean runThresholdToROI(final double threshold, final boolean percentMax, final boolean excludeZero, final boolean makeSeriesROI,
			final boolean seriesUsesDynamicThreshold);



	/**
	 * Adds new ROI mask data based on a threshold.
	 * 
	 * @param threshold the threshold
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param percentMax true if the threshold represents percent of image maximum, false if they refer to actual range values
	 * @param excludeZero true to exclude zero from calculation
	 * @param makeSeriesROI true to make a series ROI
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 * @return true if new ROI mask data was added, false otherwise
	 */
	@ParameterNameAnnotation(names = { "threshold", "sliceDirection", "sliceNumber", "percentMax", "excludeZero", "makeSeriesROI",
			"seriesUsesDynamicThreshold" })
	boolean runThresholdToROISlice(final double threshold, final int sliceDirection, final int sliceNumber, final boolean percentMax, final boolean excludeZero,
			final boolean makeSeriesROI, final boolean seriesUsesDynamicThreshold);



	/**
	 * Adds new ROI mask data based on a threshold and shrink wrap (3D).
	 * 
	 * @param threshold the threshold
	 * @param percentMax true if the threshold represents percent of image maximum, false if they refer to actual range values
	 * @param excludeZero true to exclude zero from calculation
	 * @param makeSeriesROI true to make a series ROI
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 * @return true if new ROI mask data was added, false otherwise
	 */
	@ParameterNameAnnotation(names = { "threshold", "percentMax", "excludeZero", "makeSeriesROI", "seriesUsesDynamicThreshold" })
	boolean runThresholdToShrinkWrapROI(final double threshold, final boolean percentMax, final boolean excludeZero, final boolean makeSeriesROI,
			final boolean seriesUsesDynamicThreshold);



	/**
	 * Adds new ROI mask data based on a threshold and shrink wrap (2D).
	 * 
	 * @param threshold the threshold
	 * @param sliceDirection the slice direction
	 * @param percentMax true if the threshold represents percent of image maximum, false if they refer to actual range values
	 * @param excludeZero true to exclude zero from calculation
	 * @param makeSeriesROI true to make a series ROI
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 * @return true if new ROI mask data was added, false otherwise
	 */
	@ParameterNameAnnotation(names = { "threshold", "sliceDirection", "percentMax", "excludeZero", "makeSeriesROI", "seriesUsesDynamicThreshold" })
	boolean runThresholdToShrinkWrapROIAllSlices(final double threshold, final int sliceDirection, final boolean percentMax, final boolean excludeZero,
			final boolean makeSeriesROI, final boolean seriesUsesDynamicThreshold);



	/**
	 * Adds new ROI mask data based on a threshold and shrink wrap (2D).
	 * 
	 * @param threshold the threshold
	 * @param sliceDirection the slice direction
	 * @param sliceNumber the slice number
	 * @param percentMax true if the threshold represents percent of image maximum, false if they refer to actual range values
	 * @param excludeZero true to exclude zero from calculation
	 * @param makeSeriesROI true to make a series ROI
	 * @param seriesUsesDynamicThreshold true only if percentMax is also true, in which case percent max is determined for each series point individually
	 * @return true if new ROI mask data was added, false otherwise
	 */
	@ParameterNameAnnotation(names = { "threshold", "sliceDirection", "sliceNumber", "percentMax", "excludeZero", "makeSeriesROI",
			"seriesUsesDynamicThreshold" })
	boolean runThresholdToShrinkWrapROISlice(final double threshold, final int sliceDirection, final int sliceNumber, final boolean percentMax,
			final boolean excludeZero, final boolean makeSeriesROI, final boolean seriesUsesDynamicThreshold);



	/**
	 * Runs volume statistics on the currently selected series point of the currently selected ImageVolume.
	 * 
	 * @return the volume stat
	 */
	Analysis runVolumeStats();



	/**
	 * Runs volume ROI statistics.
	 * 
	 * @param roiMask the ROI mask to include
	 * @param includePointsAndLines true to include points and lines, false otherwise
	 * @param onlySelectedPoints true to include selected points
	 * @param onlySelectedLines true to include selected lines
	 * @return a list of Analyses
	 */
	@ParameterNameAnnotation(names = { "roiMask", "includePointsAndLines", "onlySelectedPoints", "onlySelectedLines" })
	List<Analysis> runVolumeStatsROI(long roiMask, boolean includePointsAndLines, boolean onlySelectedPoints, boolean onlySelectedLines);



	/**
	 * Saves this VolumeManager.
	 */
	void save();



	/**
	 * Saves this VolumeManager
	 * 
	 * @param saveLocation the location of the file to save to
	 */
	@ParameterNameAnnotation(names = { "saveLocation" })
	void saveWithName(String saveLocation);



	/**
	 * Saves this VolumeManager.
	 * 
	 * @param saveLocation the location of the file to save to
	 * @param headerFormat the header format (see ImageVolume constants)
	 * @param datatype (see ImageType constants)
	 * @param numBytes the number of bytes per voxel (1, 2, or 4 are supported)
	 * @param littleEndian true to save multi-byte data in little endian byte order, false to use big endian byte order
	 * @param compress true to compress (not applicable to all header types)
	 * @param xMin x minimum bounds
	 * @param xMax x maximum bounds
	 * @param yMin y minimum bounds
	 * @param yMax y maximum bounds
	 * @param zMin z minimum bounds
	 * @param zMax z maximum bounds
	 * @param tMin t minimum bounds
	 * @param tMax t maximum bounds
	 * @param xSize x voxel size
	 * @param ySize y voxel size
	 * @param zSize z voxel size
	 * @param tSize t size ratio (if old size is 2 and new size is .5, then this value should be .25)
	 * @param orientation the orientation (e.g., "XYZ+--")
	 * @param applyTransform true to apply the transform when saving, false otherwise
	 * @param applyOverlay true to save the currently selected overlay ImageVoluem, false to save the base volume
	 * @param overlayThreshold true to apply a threshold to the saved overlay
	 * @param overlayMix true to save both the base volume and overlay volumes in the same image (i.e., needs a split color table)
	 * @param interpolation the interpolation code (see constants)
	 * @param applyDataScales true to apply data scales
	 * @param fitPrecision true to fit the image data to the bounds of the datatype
	 * @param maintainZero true to use data scale slope, but not data scale intercept (i.e., intercept = 0)
	 */
	@ParameterNameAnnotation(names = { "saveLocation", "headerFormat", "datatype", "numBytes", "littleEndian", "compress", "xMin", "xMax", "yMin", "yMax",
			"zMin", "zMax", "tMin", "tMax", "xSize", "ySize", "zSize", "tSize", "orientation", "applyTransform", "applyOverlay", "overlayThreshold",
			"overlayMix", "interpolation", "applyDataScales", "fitPrecision", "maintainZero" })
	void saveAs(final String saveLocation, final String headerFormat, int datatype, int numBytes, boolean littleEndian, final boolean compress, final int xMin,
			final int xMax, final int yMin, final int yMax, final int zMin, final int zMax, final int tMin, final int tMax, final double xSize,
			final double ySize, final double zSize, final double tSize, final String orientation, boolean applyTransform, final boolean applyOverlay,
			final boolean overlayThreshold, final boolean overlayMix, int interpolation, final boolean applyDataScales, final boolean fitPrecision,
			final boolean maintainZero);



	/**
	 * Saves overlays.
	 */
	void saveOverlays();



	/**
	 * Saves ROI.
	 * 
	 * @param saveLocation the file location to save
	 */
	@ParameterNameAnnotation(names = { "saveLocation" })
	void saveROI(String saveLocation);



	/**
	 * Saves a single ROI color.
	 * 
	 * @param saveLocation the file location to save
	 * @param index the roi color index
	 */
	@ParameterNameAnnotation(names = { "saveLocation", "roiColorIndex" })
	void saveSingleROI(String saveLocation, int index);



	/**
	 * Selects or unselects a specified ROI.
	 * 
	 * @param roi the ROI
	 * @param select true to select, false to unselect
	 */
	@ParameterNameAnnotation(names = { "roi", "select" })
	void select(ROI roi, boolean select);



	/**
	 * Selects all ROI masks, lines, and points.
	 */
	void selectAll();



	/**
	 * Selects a menu option.
	 * 
	 * @param name the menu option name
	 */
	@ParameterNameAnnotation(names = { "name" })
	void selectMenuOption(String name);



	/**
	 * Sets the coordinate space of the currently selected ImageVolume.
	 * 
	 * @param index the coordinate space index
	 * @deprecated
	 */
	@Deprecated
	@ParameterNameAnnotation(names = { "index" })
	void setCoordinateSpace(int index);



	/**
	 * Sets the orientation option (e.g., NIFTI s-form or q-form) of the currently selected ImageVolume.
	 * 
	 * @param volume the image volume
	 * @param index the orientation option index
	 */
	@ParameterNameAnnotation(names = { "volume", "index" })
	void setOrientationOption(ImageVolume volume, int index);



	/**
	 * Sets the current position.
	 * 
	 * @param coordinate the new position
	 */
	@ParameterNameAnnotation(names = { "coordinate" })
	void setCurrentPosition(Coordinate coordinate);



	/**
	 * Sets the current series point.
	 * 
	 * @param seriesIndex the new series point
	 */
	@ParameterNameAnnotation(names = { "seriesIndex" })
	void setCurrentSeriesPoint(int seriesIndex);



	/**
	 * Sets the current volume by index.
	 * 
	 * @param index the volume index
	 */
	@ParameterNameAnnotation(names = { "index" })
	void setCurrentVolumeIndex(int index);



	/**
	 * Sets the display range of the specified ImageVolume to its image min/max.
	 * 
	 * @param volume the ImageVolume
	 * @param entireSeries true to use the min/max of the entire series, false to use the current series point
	 */
	@ParameterNameAnnotation(names = { "volume", "entireSeries" })
	void setDisplayRangeToImageRange(ImageVolume volume, boolean entireSeries);



	/**
	 * Sets the label of the specified ROI.
	 * 
	 * @param roi the ROI
	 * @param label the label
	 */
	@ParameterNameAnnotation(names = { "roi", "label" })
	void setLabel(final ROI roi, final String label);



	/**
	 * Sets a menu option.
	 * 
	 * @param name the menu option
	 * @param value the menu option value
	 */
	@ParameterNameAnnotation(names = { "name", "value" })
	void setMenuOption(String name, boolean value);



	/**
	 * Sets the note of the base volume.
	 * 
	 * @param text the note text
	 */
	@ParameterNameAnnotation(names = { "text" })
	void setNotes(String text);



	/**
	 * Sets the origin of the base volume.
	 * 
	 * @param coordinate the new origin
	 */
	@ParameterNameAnnotation(names = { "coordinate" })
	void setOrigin(Coordinate coordinate);



	/**
	 * Sets the current ROI color.
	 * 
	 * @param color the ROI color index
	 */
	@ParameterNameAnnotation(names = { "roiColorIndex" })
	void setRoiColor(int color);



	/**
	 * Sets the current slice direction.
	 * 
	 * @param sliceDirection the slice direction
	 */
	@ParameterNameAnnotation(names = { "sliceDirection" })
	void setSliceDirection(int sliceDirection);



	/**
	 * Sets the current slice number.
	 * 
	 * @param sliceNumber the slice number
	 * @return true if the change in slice number was successful, false otherwise
	 */
	@ParameterNameAnnotation(names = { "sliceNumber" })
	boolean setSliceNumber(int sliceNumber);



	/**
	 * Sets the transform mode.
	 * 
	 * @param transform true to turn on transform, false to turn off transform
	 */
	@ParameterNameAnnotation(names = { "transform" })
	void setTransformMode(boolean transform);



	/**
	 * Sets the alpha value of the specified ImageVolume.
	 * 
	 * @param volume the ImageVolume
	 * @param alpha the alpha value (0 to 1)
	 */
	@ParameterNameAnnotation(names = { "volume", "alpha" })
	void setVolumeDisplayAlpha(ImageVolume volume, double alpha);



	/**
	 * Sets the color table of the specified ImageVolume.
	 * 
	 * @param volume the ImageVolume
	 * @param colorTable the color table name
	 */
	@ParameterNameAnnotation(names = { "volume", "colorTable" })
	void setVolumeDisplayColorTable(ImageVolume volume, String colorTable);



	/**
	 * Sets the hidden state of the specified ImageVolume.
	 * 
	 * @param volume the ImageVolume
	 * @param hidden true to hide the volume, false to display it
	 */
	@ParameterNameAnnotation(names = { "volume", "hidden" })
	void setVolumeDisplayHidden(ImageVolume volume, boolean hidden);



	/**
	 * Sets the display index of the specified ImageVolume (cannot change base volume index).
	 * 
	 * @param volume the ImageVolume
	 * @param index the new index
	 */
	@ParameterNameAnnotation(names = { "volume", "index" })
	void setVolumeDisplayIndex(ImageVolume volume, int index);



	/**
	 * Sets the display range of the specified ImageVolume.
	 * 
	 * @param volume the ImageVolume
	 * @param screenMin the new display range minimum
	 * @param screenMax the new display range maximum
	 */
	@ParameterNameAnnotation(names = { "volume", "screenMin", "screenMax" })
	void setVolumeDisplayRange(ImageVolume volume, double screenMin, double screenMax);



	/**
	 * Sets a voxel value at the specified coordinate.
	 * 
	 * @param coordinate the coordinate
	 * @param seriesIndex the series point
	 * @param value the new value
	 */
	@ParameterNameAnnotation(names = { "coordinate", "seriesIndex", "value" })
	void setVoxelValue(final Coordinate coordinate, final int seriesIndex, final double value);



	/**
	 * Shows an error message.
	 * 
	 * @param message the message text
	 * @param title the message title
	 */
	@ParameterNameAnnotation(names = { "message", "title" })
	void showErrorMessage(String message, String title);



	/**
	 * Shows a warning message.
	 * 
	 * @param message the message text
	 * @param title the message title
	 */
	@ParameterNameAnnotation(names = { "message", "title" })
	void showWarningMessage(String message, String title);



	/**
	 * Returns a string representation.
	 * 
	 * @return a string representation
	 */
	@Override
	String toString();



	/**
	 * Transform the specified ImageVolume.
	 * 
	 * @param volume the ImageVolume
	 * @param m00 m00
	 * @param m01 m01
	 * @param m02 m02
	 * @param m03 m03
	 * @param m10 m10
	 * @param m11 m11
	 * @param m12 m12
	 * @param m13 m13
	 * @param m20 m20
	 * @param m21 m21
	 * @param m22 m22
	 * @param m23 m23
	 * @param m30 m30
	 * @param m31 m31
	 * @param m32 m32
	 * @param m33 m33
	 */
	@ParameterNameAnnotation(names = { "volume", "m00", "m01", "m02", "m03", "m10", "m11", "m12", "m13", "m20", "m21", "m22", "m23", "m30", "m31", "m32",
			"m33" })
	void transform(ImageVolume volume, double m00, double m01, double m02, double m03, double m10, double m11, double m12, double m13, double m20, double m21,
			double m22, double m23, double m30, double m31, double m32, double m33);



	/**
	 * Redraws this VolumeManager.
	 */
	void updateViewer();
}
