function evlab17_run_model(varargin)
% EVLAB17_RUN_MODEL runs model estimation pipeline
%   evlab17_run_model(cfgfile1,cfgfile2,...); 
%      loads model info configuration from cfgfile's 
%      and runs model definition/estimation and contrast definition/estimation pipeline
%
%   evlab17_run_model(..., [], fieldname1, fieldvalue1, ...)
%      configuration fields may be directly directly defined from fieldname/fieldvalue pairs
%
%  OPTIONS (entered in .cfg file with fieldnames preceeded by #, or entered as argument pairs to evlab17_run_model; see for example "example_modelfiles.cfg")
%      dataset         : file generated by preprocessing step evlab17_*.mat (note: explicitly specifying a dataset may be skipped when evlab17_run_model is run immediately after evlab17_run_preproc)
%      design          : file containing experimental design model/condition definitions (e.g. *.cat file). 
%                        note: if this field is not defined then model definition/estimation is skipped (e.g. to run only contrast definition/estimation; same as setting the field contrastonly to true)
%                        This file should contain the following fields:
%          runs        : list of runs/sessions to include in the model (default: all sessions identified as functional data during preprocessing step)
%                        alternatively, use fieldname 'abs_runs' or 'dicom_runs' to specify absolute dicom run/session numbers
%                                       use fieldname 'rel_runs' or 'runs' to specify relative run/session numbers (among those identified as functional data during preprocessing step)
%          files       : list of condition-information files (enter one *.para file per run). Each of these files should contain the following fields:
%             onsets   : Nx2 array with onset times for each event/block in the first column and condition number (1-M) for each event/block in the second column   
%             durations: Nx1 array with duration of each event/block OR 1xM array with duration of each condition
%             names    : 1xM cell array with condition names (note: condition names cannot contain whitespace charaters)
%             units    : 'scans' / 'secs; : units for onset/duration specification of condition information  (default: scans)
%             (optional fields for sparse sampling acquisitions)
%             scan_times: Nscans x 1 array with times of each scan acquisition onset (in seconds, starting at 0s) event-effects will be sampled at time RT*fMRI_T0/fMRI_T after each scan onset
%             scan     : 1/0 value indicating whether an explicit scanner-noise regressor should be included [0] (note: this is modeled as an additional condition and named 'scannernoise')
%             (optional fields only applicable when multiple within-condition effects are estimated)
%             orth     :  1/0 value indicating whether within-condition regressors should be GS orthogonalized [1]
%             (optional fields for temporal modulation)
%             tmod     : 1xM 1/0 array indicating whether condition M is modulated by time (if values >1 are entered they are interpreted as polynomial order of temporal modulation effects)
%             (optional fields for parametric modulation)
%             pmod     : KxM 1/0 array indicating whether condition M is modulated by covariate K
%             pmod_names  : 1xK cell array of covariate names (note: covariate names cannot contain whitespace characters)
%             pmod_values : NxK matrix of covariate values (e.g. reaction time)
%             pmod_interaction : 1xK 1/0 array indicating whether to include condition-by-covariate interactions for each covariate [1]
%             (optional fields for non-parametric modulation; estimation of individual trial-level effects)
%             npmod    : 1xM 1/0 array indicating whether condition M is broken down into individual-trial effects
%         (optional fields) 
%          path        : path to files listed in 'files' (default: expects full filepaths in #files field)
%         (alternative fields) 
%          files       : alternatively, 'files' may point instead to an individual SPM.mat file containing the entire first-level model specification
%      model_name      : model name
%      contrasts       : contrast definition strings. Cell array (with one line per contrast, fields are separated by whitespaces):
%                          CONTRAST_NAME CONDITION_NAME1 CONDITION_WEIGHT1 CONDITION_NAME2 CONDITION_WEIGHT2 ...
%                          e.g. S-N S 1 N -1
%                          note: valid condition names are:
%                           1) (in all cases) those defined in the files.names field above
%                           2) (for temporal modulations) [conditionname]xtime, where [conditionname] is a condition name (in files.names)
%                           3) (for parametric modulations including condition-by-covariate interactions) [conditionname]x[covariatename], where [conditionname] is a condition name (in files.names) and [covariatename] is a parametric covariate name (in files.pmod_names)
%                           4) (for parametric modulations not including condition-by-covariate interactions) [covariatename], where [covariatename] is a parametric covariate name (in files.pmod_names)
%                           5) (for non-parametric modulations) [conditionname]_EVENT## where [conditionname] is a condition name (in files.names) and ## is the event number
%
%  (additional data- and model- definition options)
%      RT              : repetition time (in seconds) (typically this is already defined during evlab17_run_preproc step)
%      units           : scans / secs : units for onset/duration specification of condition information  (default: scans; note: #units field may alternatively be defined inside .para files)
%      functional_label: choose version of functional data to enter in first-level analysis: only when used in combination with "functional_label" preprocessing steps: enter Secondary Dataset label identifying the desired functional dataset (default: Primary Dataset, i.e. fully preprocessed functional data)
%      functional_smoothinglevel: choose version of functional data to enter in first-level analysis: only when functional_label is not specified; only when used in combination with preprocessing pipelines which implement multiple smoothing steps: enter 0/1/2 (0 unsmoothed data; 1 minimally-smoothed data; 2 fully-smoothed data) (default: 1)
%      model_basis     : hrf / hrf+deriv / hrf+derivs / none : response function: enter hrf for hemodynamic response function only; hrf+deriv to add temporal derivative; hrf+derivs to add temporal and dispersion derivatives; none for no hrf convolution (default: hrf+deriv)
%      model_covariates: list of additional covariates; possible values: motion, motion+deriv, motion+deriv+square, art (for scrubbing), linear (for detrending), denoise (for any functional_regression step), any 1st-level covariate name, or <filename>.mat (one file per run) (default: motion / art)
%      model_serial    : none / AR(1) : serial correlation modeling (default: AR(1))
%      model_session   : 1/0 : estimates session-specific task effects (default value: 1; when set to 0 SPM estimates session-invariant task effects -it assumes same effect across all sessions-; alternatively set to cell array of task names in order to specify individual task effects that will be model as session-invariant)
%      model_folder    : folder where model_name subfolder will be created (default '../'; relative to dataset file) 
%      mthresh         : implicit masking threshold as proportion of global signal (default value: 0.8)
%      explicitmask    : optional explicit mask (filename)
%      hpf             : high-pass filter threhsold -in seconds- (default value: 128s)
%      lpf             : low-pass filter threshold -in seconds- (default value: 0s = no low-pass filter)
%      contrast_addsession: 0/1 for each contrast defined in #contrasts field above, create additional SESSION##_[contrastname] including only the n-th run data (default 0)
%      contrast_addcv     : 0/1 for each contrast defined in #contrasts field above, create additional SESSION##_[contrastname] and ORTH_TO_SESSION##_[contrastname] including only the n-th run and all but the n-th run data, respectively (default 0)
%      contrast_addoddeven: 0/1 for each contrast defined in #contrasts field above, create additional ODD_[contrastname] and EVEN_[contrastname] including only odd and even numbered runs, respectively (default 0)
%      contrast_removenonestimablecols: 0/1 When defining contrast vectors skip design-matrix columns that are not fully estimable individually (default 0)
%      contrast_removenonestimablecontrasts: 0/1 Skip contrasts that are not fully estimable (default 1)
%      contrast_removeexistingcontrasts: 0/1 removes/deletes any older/existing contrasts in this first-level analysis (default 1)
%      contrast_addevent: (for non-parametric modulations) 0/1 for each contrast defined in #contrasts field above, automatically create additional [contrastname]_EVENT## contrasts for each individual condition (default 0)
%      contrastonly    : 1/0 : skips first-level model definition/estimation step and performs only contrast definition/estimation [0]
%      subjects        : index to subject(s) to process (for multi-subjet datasets) (default value: 1)
%      qa_plots        : (model review displays) 1/0 to create quality assurance plots [0]
%      qa_plist        : (model review displays) list of quality assurance plots to create (default value: [] for all QA displays)
%      qa_folder       : (model review displays) output directory for QA plots

%      qa_parallel     : (model review displays) 1/0 to create plots in parallel (using background or cluster computing resources) [1]
%      qa_profile      : (model review displays) name of parallelization profile (see "conn_jobmanager profiles" to see a list of available parallelization profiles) [background process]
%

evlab17_module init;

% loads .cfg files
options=struct;
for n=1:nargin
    if isempty(varargin{n}), break; end
    filename=varargin{n};
    if ischar(filename)
        if isempty(dir(filename)),
            if ~isempty(dir(fullfile(fileparts(which(mfilename)),filename))), filename=fullfile(fileparts(which(mfilename)),filename);
            else filename=which(filename);
            end
        end
        if isempty(dir(filename)), error('file %s not found',varargin{n}); end
        fprintf('loading file %s\n',filename);
    end
    options=conn_loadcfgfile(filename,options);
end
for n=n+1:2:nargin-1
    fieldname=regexp(varargin{n},'\.','split');
    fieldvalue=varargin{n+1};
    options=setfield(options,fieldname{:},fieldvalue);
end
if isfield(options,'design'),
    if ~isempty(options.design), options=conn_loadcfgfile(options.design,options); end
    options=rmfield(options,'design');
end
fprintf('%s options:\n',mfilename);
disp(options);
options0=options;
options0.arguments=varargin;

skipmodeldefinition=false;
skipmodelestimation=false;
pwd0=pwd;
clear matlabbatch;
debugskip=false;
tag=datestr(now,'yyyy_mm_dd_HHMMSSFFF');

if isfield(options,'dataset')
    filename=char(options.dataset);
    ok=evlab17_module('load',filename);
    if ~ok, error('problem loading preprocessed dataset info from %s',filename); end
    options=rmfield(options,'dataset');
elseif ~isfield(options,'oknodataset')
    filename=evlab17_module('filename');
    if isempty(filename), error('#dataset field not specified'); end
    if 0, fprintf('Warning: #dataset field not specified; using current dataset (%s)\n',filename);
    elseif ~isequal(conn_questdlg({'#dataset field not specified','Do you want to continue processing dataset',[filename,'?']},'','Yes','No','Yes'),'Yes'), return;
    end
end

if isfield(options,'subjects')
    nsubject=options.subjects;
    options=rmfield(options,'subjects');
else nsubject=1;
end
prependmodelname=true;
if isfield(options,'model_folder'), 
    model_folder=char(options.model_folder);
    if isempty(model_folder)||isequal(model_folder,'root'), % back-compatibility: input <root>/nii; output <root>/firstlevel_*/
        model_folder=fullfile(fileparts(evlab17_module('filename'))); 
        [tpath,tname]=fileparts(model_folder); if isequal(tname,'nii')||isequal(tname,'func'), model_folder=tpath; end
    elseif isequal(model_folder,'preproc') % separate folder for each preprocessing pipeline: <preproc>/results/firstlevel/*
        model_folder=fullfile(conn_prepend('',evlab17_module('filename'),''),'results','firstlevel');
        prependmodelname=false;
    end
    if numel(model_folder)>=1&&model_folder(1)=='.',model_folder=fullfile(fileparts(evlab17_module('filename')),model_folder); end
    options=rmfield(options,'model_folder');
elseif evlab17_module('inconnfolders'), 
    model_folder=fullfile(conn_prepend('',evlab17_module('filename'),''),'results','firstlevel'); 
    prependmodelname=false;
else model_folder=fullfile(fileparts(fileparts(evlab17_module('filename')))); 
end
if isfield(options,'model_name'), 
    model_name=fullfile(model_folder,char(options.model_name));
    if prependmodelname, model_name=conn_prepend('firstlevel_',model_name); end
    [ok,nill]=mkdir(model_name);
    if evlab17_module('get','Setup.nsubjects')>1, model_name=fullfile(model_name,sprintf('sub-%04d',nsubject)); [ok,nill]=mkdir(model_name); end
    matlabbatch{1}.spm.stats.fmri_spec.dir={model_name};
    options=rmfield(options,'model_name');
    pwd1=char(matlabbatch{1}.spm.stats.fmri_spec.dir);
else
    error('field #model_name not found');
end
% qafolder=fullfile(pwd1,['QA_',tag]);
% if isfield(options,'qa_folder'), qafolder=options.qa_folder;
% elseif evlab17_module('inconnfolders'), 
%     if evlab17_module('get','Setup.nsubjects')>1, qafolder=fullfile(evlab17_module('get','folders.qa'),[sprintf('QA_MODEL_sub-%04d_',nsubject),tag]); 
%     else qafolder=fullfile(evlab17_module('get','folders.qa'),['QA_MODEL_',tag]); 
%     end 
% end

if isfield(options,'contrastonly')&&~isempty(options.contrastonly)&&options.contrastonly~=0
    skipmodelestimation=true;
    options=rmfield(options,'contrastonly');
    if isfield(options,'files'), options=rmfield(options,'files'); end
elseif isfield(options,'files'),
    if isempty(options.files)||(numel(options.files)==1&&isempty(options.files{1})), options.files=evlab17_module('get','spm'); options.files=options.files(nsubject); end % current SPM.mat file
    if ~iscell(options.files), options.files={options.files}; end
    if numel(options.files)==1&&ischar(options.files{1})&&~isempty(regexp(options.files{1},'\.mat$')), skipmodeldefinition=true; end % explicit SPM.mat file
else skipmodelestimation=true; 
end

if skipmodelestimation, conn_disp('fprintf','Skipping model estimation\n');
else
    if isfield(options,'functional_smoothinglevel')
        smoothinglevel=options.functional_smoothinglevel;
        options=rmfield(options,'functional_smoothinglevel');
    elseif isfield(options,'smoothinglevel')
        smoothinglevel=options.smoothinglevel;
        options=rmfield(options,'smoothinglevel');
    else smoothinglevel=1;
    end
    if isfield(options,'functional_label')
        functional_label=options.functional_label;
        options=rmfield(options,'functional_label');
        smoothinglevel=2;
    else functional_label='';
    end
        
    NSESSIONS=evlab17_module('get','Setup.nsessions');
    NSESSIONS=NSESSIONS(nsubject);
    if isfield(options,'dicom_runs')||isfield(options,'abs_runs')
        if isfield(options,'dicom_runs'), abs_runs=options.dicom_runs; options=rmfield(options,'dicom_runs');
        else abs_runs=options.abs_runs; options=rmfield(options,'abs_runs');
        end
        files=evlab17_module('get','functionals');
        files=files(nsubject);
        RUNS=zeros(1,numel(abs_runs));
        for nses=1:NSESSIONS
            filename=char(files{1}{nses});
            [filepath,filename,fileext]=fileparts(filename);
            sesstr=regexp(filename,'\d+$','match','once');
            if ~isempty(sesstr), sesstr=str2num(sesstr); end
            if numel(sesstr)~=1, error('unable to interpret dicom session number from filename %s',filename); end
            RUNS(abs_runs==sessstr)=nses;
        end
        if any(RUNS==0), error('unable to find dicom run(s) %s in dataset',mat2str(abs_runs(RUNS==0))); end
    elseif isfield(options,'runs')||isfield(options,'rel_runs')
        if isfield(options,'runs'), runs=options.runs; options=rmfield(options,'runs');
        else runs=options.rel_runs; options=rmfield(options,'rel_runs');
        end
        RUNS=reshape(runs,1,[]);
        assert(all(ismember(RUNS,1:NSESSIONS)),'Invalid run numbers (run numbers cannot be higher than the number of runs/sessions in this dataset = %d)',NSESSIONS);
    else
        RUNS=1:NSESSIONS;
    end
    NSESSIONS=numel(RUNS);
    pmod_interaction={};    
    scan_times={};

    if skipmodeldefinition % re-estimates model from existing SPM.mat file
        if isequal(options.files{1},fullfile(pwd1,'SPM.mat'))
            conn_disp('fprintf','Re-computing model estimation %s\n',fullfile(pwd1,'SPM.mat'));
        else
            if ispc, [ok,msg]=system(sprintf('copy "%s" "%s"',options.files{1},fullfile(pwd1,'SPM.mat')));
            else [ok,msg]=system(sprintf('cp ''%s'' ''%s''',options.files{1},fullfile(pwd1,'SPM.mat')));
            end
            conn_disp('fprintf','Re-computing model estimation %s (original source: %s)\n',fullfile(pwd1,'SPM.mat'),options.files{1});
        end
    else % creates and estimates new model
        nscans=conn_module('get','Setup.nscans');
        nscans=nscans(nsubject);
        if isfield(options,'units')
            paraunits=char(options.units);
            options=rmfield(options,'units');
        else paraunits=[];
        end
        
        if isfield(options,'RT'),
            if iscell(options.RT), options.RT=options.RT{1}; end
            if ischar(options.RT), options.RT=str2num(options.RT); end
            if isnan(options.RT), options=rmfield(options,'RT'); end
        end
        if isfield(options,'RT'),
            matlabbatch{1}.spm.stats.fmri_spec.timing.RT=options.RT;
            options=rmfield(options,'RT');
        else
            RT=conn_get_rt(nsubject,RUNS); 
            assert(numel(RT)==1||all(diff(RT)==0), 'Different RT values across sessions (%s). All sessions in first-level analysis must have the same scanning Repetition Time',mat2str(RT));
            matlabbatch{1}.spm.stats.fmri_spec.timing.RT=RT;
            conn_disp('fprintf','Warning: first-level analysis #RT field not defined. Using data-derived value = %s \n',mat2str(matlabbatch{1}.spm.stats.fmri_spec.timing.RT));
        end
        
        if isfield(options,'model_basis')
            switch(lower(char(options.model_basis)))
                case 'none',
                    matlabbatch{1}.spm.stats.fmri_spec.bases.none=1;
                case 'hrf',
                    matlabbatch{1}.spm.stats.fmri_spec.bases.hrf.derivs=[0 0];
                case 'hrf+deriv',
                    matlabbatch{1}.spm.stats.fmri_spec.bases.hrf.derivs=[1 0];
                case 'hrf+derivs',
                    matlabbatch{1}.spm.stats.fmri_spec.bases.hrf.derivs=[1 1];
                otherwise
                    error('unrecognized model_basis option %s',lower(char(options.model_basis)));
            end
            options=rmfield(options,'model_basis');
        else
            matlabbatch{1}.spm.stats.fmri_spec.bases.hrf.derivs=[1 0];
        end
        
        if isfield(options,'model_serial')
            switch(lower(char(options.model_serial)))
                case 'none',
                    matlabbatch{1}.spm.stats.fmri_spec.cvi='none';
                case 'ar(1)',
                    matlabbatch{1}.spm.stats.fmri_spec.cvi='AR(1)';
                case 'fast',
                    matlabbatch{1}.spm.stats.fmri_spec.cvi='FAST';
                otherwise
                    error('unrecognized model_basis option %s',lower(char(options.model_serial)));
            end
            options=rmfield(options,'model_serial');
        else
            matlabbatch{1}.spm.stats.fmri_spec.cvi='AR(1)';
        end
        
        if isfield(options,'mthresh')
            if iscell(options.mthresh), options.mthresh=options.mthresh{1}; end
            if ischar(options.mthresh), options.mthresh=str2num(options.mthresh); end
            matlabbatch{1}.spm.stats.fmri_spec.mthresh=options.mthresh;
            options=rmfield(options,'mthresh');
        end
        
        if isfield(options,'explicitmask')
            matlabbatch{1}.spm.stats.fmri_spec.mask=options.explicitmask;
            options=rmfield(options,'explicitmask');
        end
                
        if isfield(options,'hpf')
            if iscell(options.hpf), options.hpf=options.hpf{1}; end
            if ischar(options.hpf), options.hpf=str2num(options.hpf); end
            for nses=1:NSESSIONS
                matlabbatch{1}.spm.stats.fmri_spec.sess(nses).hpf=options.hpf;
            end
            options=rmfield(options,'hpf');
        end
        if isfield(options,'lpf')
            if iscell(options.lpf), options.lpf=options.lpf{1}; end
            if ischar(options.lpf), options.lpf=str2num(options.lpf); end
            for nses=1:NSESSIONS
                isnewregress=false; if ~isfield(matlabbatch{1}.spm.stats.fmri_spec,'sess')||numel(matlabbatch{1}.spm.stats.fmri_spec.sess)<nses||~isfield(matlabbatch{1}.spm.stats.fmri_spec.sess(nses),'regress')||isempty(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).regress), isnewregress=true; end
                N=nscans{1}{RUNS(nses)};
                n=(0:N-1)';
                k=fix(2*N*matlabbatch{1}.spm.stats.fmri_spec.timing.RT/options.lpf):N-1;
                for nk=1:numel(k)
                    c=sqrt(2/N)*cos(pi*(n+1/2)*k(nk)/N);
                    newregress=struct('name',sprintf('LPF%d',nk),'val',c);
                    if isnewregress, matlabbatch{1}.spm.stats.fmri_spec.sess(nses).regress=newregress; isnewregress=false;
                    else matlabbatch{1}.spm.stats.fmri_spec.sess(nses).regress(end+1)=newregress;
                    end
                end
            end
            options=rmfield(options,'lpf');
        end
        
        if ~isfield(options,'model_covariates')
            options.model_covariates={'motion','art'};
        end
        
        if isfield(options,'model_covariates')
            if ~iscell(options.model_covariates), options.model_covariates={options.model_covariates}; end
            if any(strcmp(lower(options.model_covariates),'denoise'))
                options.model_covariates=options.model_covariates(~strcmp(lower(options.model_covariates),'denoise'));
                files=evlab17_module('get','l1covariates','QC_regressors');
                if numel(files)<nsubject||isempty(files(nsubject))
                    conn_disp('fprintf','WARNING: unable to load %s information. Skipping this covariate and using ''motion'' and ''art'' instead\n','denoise');
                    options.model_covariates=unique([options.model_covariates(:)',{'motion','art'}]);
                else
                    files=files(nsubject);
                    for nses=1:NSESSIONS
                        if ~isfield(matlabbatch{1}.spm.stats.fmri_spec,'sess')||numel(matlabbatch{1}.spm.stats.fmri_spec.sess)<nses||~isfield(matlabbatch{1}.spm.stats.fmri_spec.sess(nses),'multi_reg')||isempty(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg),
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg={};
                        end
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg{end+1,1}=files{1}{RUNS(nses)};
                    end
                end
            end
            if any(ismember(lower(options.model_covariates),{'motion','motion+deriv','motion+deriv+square','motion+square+deriv'}))
                do12=any(ismember(lower(options.model_covariates),{'motion+deriv'}));
                do24=any(ismember(lower(options.model_covariates),{'motion+deriv+square','motion+square+deriv'}));
                options.model_covariates=options.model_covariates(~ismember(lower(options.model_covariates),{'motion','motion+deriv','motion+deriv+square','motion+square+deriv'}));
                files=evlab17_module('get','l1covariates','realignment');
                if numel(files)<nsubject||isempty(files(nsubject))
                    conn_disp('fprintf','WARNING: unable to load %s information. Skipping this covariate\n','motion');
                else
                    files=files(nsubject);
                    for nses=1:NSESSIONS
                        cfile=files{1}{RUNS(nses)};
                        if do24||do12,
                            R=spm_load(cfile);
                            R=[R convn([R(1,:);R],[1;-1],'valid')];
                            if do24, R=[R R.*(R./repmat(max(eps,max(abs(R),[],1)),size(R,1),1))]; end
                            cfile=conn_prepend('',cfile,sprintf('_%d.mat',12+12*do24));
                            save(cfile,'R');
                        end
                        if ~isfield(matlabbatch{1}.spm.stats.fmri_spec,'sess')||numel(matlabbatch{1}.spm.stats.fmri_spec.sess)<nses||~isfield(matlabbatch{1}.spm.stats.fmri_spec.sess(nses),'multi_reg')||isempty(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg),
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg={};
                        end
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg{end+1,1}=cfile;
                    end
                end
            end
            if any(strcmp(lower(options.model_covariates),'art'))
                options.model_covariates=options.model_covariates(~strcmp(lower(options.model_covariates),'art'));
                files=evlab17_module('get','l1covariates','scrubbing');
                if numel(files)<nsubject||isempty(files(nsubject))
                    conn_disp('fprintf','WARNING: unable to load %s information. Skipping this covariate\n','art');
                else
                    files=files(nsubject);
                    for nses=1:NSESSIONS
                        if ~isfield(matlabbatch{1}.spm.stats.fmri_spec,'sess')||numel(matlabbatch{1}.spm.stats.fmri_spec.sess)<nses||~isfield(matlabbatch{1}.spm.stats.fmri_spec.sess(nses),'multi_reg')||isempty(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg),
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg={};
                        end
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg{end+1,1}=files{1}{RUNS(nses)};
                    end
                end
            end
            if any(strcmp(lower(options.model_covariates),'linear'))
                options.model_covariates=options.model_covariates(~strcmp(lower(options.model_covariates),'linear'));
                for nses=1:NSESSIONS
                    newregress=struct('name','LinearTrend','val',linspace(-1,1,nscans{1}{RUNS(nses)})');
                    if ~isfield(matlabbatch{1}.spm.stats.fmri_spec,'sess')||numel(matlabbatch{1}.spm.stats.fmri_spec.sess)<nses||~isfield(matlabbatch{1}.spm.stats.fmri_spec.sess(nses),'regress')||isempty(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).regress),
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).regress=newregress;
                    else matlabbatch{1}.spm.stats.fmri_spec.sess(nses).regress(end+1)=newregress;
                    end
                end
            end
            while ~isempty(options.model_covariates) % 1st-level covariate names
                cname=options.model_covariates{1};
                files={}; try, files=evlab17_module('get','l1covariates',cname); end
                if isempty(files), break; end
                options.model_covariates=options.model_covariates(~strcmp(options.model_covariates,cname));
                if numel(files)<nsubject||isempty(files(nsubject))
                    conn_disp('fprintf','WARNING: unable to load %s information. Skipping this covariate\n',cname);
                else
                    files=files(nsubject);
                    for nses=1:NSESSIONS
                        if ~isfield(matlabbatch{1}.spm.stats.fmri_spec,'sess')||numel(matlabbatch{1}.spm.stats.fmri_spec.sess)<nses||~isfield(matlabbatch{1}.spm.stats.fmri_spec.sess(nses),'multi_reg')||isempty(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg),
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg={};
                        end
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg{end+1,1}=files{1}{RUNS(nses)};
                    end
                end
            end
            assert(mod(numel(options.model_covariates),NSESSIONS)==0,'unexpected model_covariate values %s',sprintf('%s ',options.model_covariates{:}));
            while ~isempty(options.model_covariates)
                for nses=1:NSESSIONS
                    files=options.model_covariates{nses};
                    if ~isfield(matlabbatch{1}.spm.stats.fmri_spec,'sess')||numel(matlabbatch{1}.spm.stats.fmri_spec.sess)<nses||~isfield(matlabbatch{1}.spm.stats.fmri_spec.sess(nses),'multi_reg')||isempty(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg),
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg={};
                    end
                    matlabbatch{1}.spm.stats.fmri_spec.sess(nses).multi_reg{end+1,1}=files;
                end
                options.model_covariates=options.model_covariates(NSESSIONS+1:end);
            end
            options=rmfield(options,'model_covariates');
        end
        
        if isfield(options,'path')
            pathtopara=char(options.path);
            options=rmfield(options,'path');
        else pathtopara='';
        end
        
        if isfield(options,'files')
            if ~iscell(options.files), options.files={options.files}; end
            assert(numel(options.files)==NSESSIONS, 'number of entries in design.files does not match number of runs (expected %d, found %d)',NSESSIONS,numel(options.files));
            
            for nses=1:NSESSIONS
                para_file=options.files{nses};
                if isempty(regexp(para_file,'^[\\\/]')), para_file=fullfile(pathtopara,para_file); end
                para=conn_loadcfgfile(para_file);
                if isfield(para,'scan_times'), matlabbatch{1}.spm.stats.fmri_spec.timing.units='secs'; break; end % note: if any session has #scan_times field transform all condition onset/duration units to secs
            end
            for nses=1:NSESSIONS
                para_file=options.files{nses};
                if isempty(regexp(para_file,'^[\\\/]')), para_file=fullfile(pathtopara,para_file); end
                para=conn_loadcfgfile(para_file);
                for overwriteopts={'orth','pmod','pmod_interaction','tmod','npmod'}, if isfield(options,overwriteopts{1}), para.(overwriteopts{1})=options.(overwriteopts{1}); end; end
                assert(isfield(para,'names'), 'Para file %s must contain the field ''names''!', para_file);
                assert(isfield(para, 'onsets'), 'Para file %s must contain the field ''onsets''!', para_file);
                factor=1;
                if ~isfield(para,'units')&&~isempty(paraunits), para.units=paraunits; end
                if isfield(para,'units'),
                    para.units=char(para.units);
                    if isfield(matlabbatch{1}.spm.stats.fmri_spec.timing,'units')&&~isequal(matlabbatch{1}.spm.stats.fmri_spec.timing.units,para.units), 
                        if strcmp(para.units,'secs')&&strcmp(matlabbatch{1}.spm.stats.fmri_spec.timing.units,'scans'), assert(~isfield(para,'scan_times'),'incorrect units combination'); factor=1/matlabbatch{1}.spm.stats.fmri_spec.timing.RT;
                        elseif strcmp(para.units,'scans')&&strcmp(matlabbatch{1}.spm.stats.fmri_spec.timing.units,'secs'), factor=matlabbatch{1}.spm.stats.fmri_spec.timing.RT; if isfield(para,'scan_times'), factor='scantosecs'; end
                        else error('unrecognized units %s / %s',para.units,matlabbatch{1}.spm.stats.fmri_spec.timing.units);
                        end
                        %error('Inconsistent #units field in %s',para_file);
                    else matlabbatch{1}.spm.stats.fmri_spec.timing.units=para.units;
                    end
                end
                if numel(para.names)==1&&size(para.onsets,2)>1&&max(para.onsets(:,2))>1,
                    para.names=regexp(char(para.names),'\s+','split');
                    para.names=para.names(cellfun('length',para.names)>0);
                end
                switch(size(para.onsets,2))
                    case 1, %onset contains (onset) single-condition case?
                        if numel(para.names)~=1, error('Para file %s: incorrect format, expected multiple columns in #onset field',para_file); end
                        para.onsets(:,2)=1;
                    case 2, %onset contains (onset,condition#) default case
                    otherwise, %onset contains (onset,condition#,param_mod)
                        %onset contains (onset,condition#,duration)
                        if ~isfield(para,'durations')&&size(para.onsets,2)==3,
                            para.durations=para.onsets(:,3);
                            para.onsets=para.onsets(:,1:2);
                            %onset contains (onset,condition#,pmod_values)
                        else
                            if ~isfield(para,'pmod'), para.pmod=zeros(size(para.onsets,2)-2,numel(para.names)); end
                            if ~isfield(para,'pmod_values'), para.pmod_values=para.onsets(:,3:end); end
                            if ~isfield(para,'pmod_names'), para.pmod_names=arrayfun(@(n)sprintf('pmod%d',n),1:size(para.onsets,2)-2,'uni',0); end
                            para.onsets=para.onsets(:,1:2);
                            conn_disp('fprintf','warning: Para file %s: extra columns in #onset field treated as parametric modulators pmod_values\n',para_file);
                        end
                        %otherwise, error('Para file %s: too many columns in #onset field',para_file);
                end
                assert(all(para.onsets(:,2)>0)&all(rem(para.onsets(:,2),1)==0),'Para file %s: condition numbers (second column in the #onsets field) should be positive integer values',para_file);
                assert(isfield(para, 'durations'), 'Para file %s must contain the field ''durations''!', para_file);
                if size(para.durations,1)==1&&size(para.durations,2)==numel(para.names), % para.durations contains one value per condition
                    assert(numel(para.durations)>=max(para.onsets(:,2)),'Para file %s: expected #durations field to contain one value per condition (found %d values, expected at least %d -maximum condition number indicated in #onsets field-)', para_file,numel(para.durations),max(para.onsets(:,2)));
                    para.durations=reshape(para.durations(para.onsets(:,2)),[],1);
                end
                assert(size(para.durations,2)==1, 'Para file %s: unable to interpret #durations field',para_file);
                assert(size(para.durations,1)==size(para.onsets,1), 'Para file %s: incorrect number of onset/duration pairs (%d onset lines, %d duration lines)',para_file,size(para.onsets,1),size(para.durations,1)); % para.durations contains one value per block/event
                if ~isfield(para,'orth'), para.orth=ones(size(para.names)); end
                assert(numel(para.orth)==numel(para.names),'Para file %s: incorrect number of orth values (%d conditions, %d values in orth field)',para_file,numel(para.names),numel(para.orth));
                if isfield(para,'pmod_names')
                    para.pmod_names=regexp(char(para.pmod_names),'\s+','split');
                    para.pmod_names=para.pmod_names(cellfun('length',para.pmod_names)>0);
                end
                [conditions,nill,iconditions]=unique(para.onsets(:,2));
                if isfield(para,'scan_times'), 
                    scan_times{nses}=para.scan_times; 
                    assert(numel(para.scan_times)==nscans{1}{RUNS(nses)},'Para file %s: number of elements in #scan_times field (%d) must be equal to number of functional scans/acquisitions (%d)',para_file,numel(para.scan_times),nscans{1}{RUNS(nses)});
                    if isequal(factor,'scantosecs')
                        assert(max(para.onsets(:,1))+1<=numel(para.scan_times),'Para file %s: condition onset times in ''scan'' units exceeds number of #scan_times values',para_file);
                        para.onsets(:,1)=para.scan_times(round(1+para.onsets(:,1)));
                        factor=1;
                    end
                end
                if isfield(matlabbatch{1}.spm.stats.fmri_spec.timing,'units'), conn_disp('fprintf','Run #%d (%d scans; %d secs): %d conditions (onsets [%d-%d %s])\n',nses,round(nscans{1}{RUNS(nses)}),round(nscans{1}{RUNS(nses)}*matlabbatch{1}.spm.stats.fmri_spec.timing.RT),length(conditions),floor(factor*min(para.onsets(:,1))),ceil(factor*max(para.onsets(:,1))),matlabbatch{1}.spm.stats.fmri_spec.timing.units);
                else conn_disp('fprintf','Run #%d (%d scans): %d conditions (onsets [%d-%d])\n',nses,round(nscans{1}{RUNS(nses)}),length(conditions),floor(factor*min(para.onsets(:,1))),ceil(factor*max(para.onsets(:,1))));
                end
                newcond=0;
                for n1=1:length(conditions),
                    if isfield(para,'npmod')&&para.npmod(conditions(n1))
                        assert(numel(para.npmod)==numel(para.names),'Para file %s: incorrect number of values in "npmod" field (%d conditions, %d values in npmod field)',para_file,numel(para.names),numel(para.npmod));
                        idx=find(iconditions==n1);
                        for n2=1:numel(idx)
                            newcond=newcond+1;
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).name=sprintf('%s_EVENT%02d',para.names{conditions(n1)},n2);
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset=factor*para.onsets(idx(n2),1);
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).duration=factor*para.durations(idx(n2));
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).orth=para.orth(conditions(n1));
                        end
                        conn_disp('fprintf','   condition %s (%d blocks/events modeled individually)\n',para.names{conditions(n1)},numel(idx));
                    else
                        newcond=newcond+1;
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).name=para.names{conditions(n1)};
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset=factor*para.onsets(iconditions==n1,1);
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).duration=factor*para.durations(iconditions==n1);
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).orth=para.orth(conditions(n1));
                        conn_disp('fprintf','   condition %s (%d blocks/events)\n',matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).name,numel(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset));
                        if isfield(para,'tmod'),
                            assert(numel(para.tmod)==numel(para.names),'Para file %s: incorrect number of values in "tmod" field (%d conditions, %d values in tmod field)',para_file,numel(para.names),numel(para.tmod));
                            if para.tmod(conditions(n1))
                                matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).tmod=para.tmod(conditions(n1));
                                conn_disp('fprintf','      temporal modulation order %d\n',para.tmod(conditions(n1)));
                            end
                        end
                        if isfield(para,'pmod'),
                            assert(isfield(para,'pmod_names'),'Para file %s: field "pmod_names" not found',para_file);
                            assert(isfield(para,'pmod_values'),'Para file %s: field "pmod_values" not found',para_file);
                            if ~isfield(para,'pmod_interaction'), para.pmod_interaction=ones(size(para.pmod_names)); end %assert(isfield(para,'pmod_interaction'),'Para file %s: field "pmod_interaction" not found',para_file);
                            assert(numel(para.pmod_names)==size(para.pmod_values,2),'Para file %s: mismatch number of parametric modulators (%d names in "pmod_names", %d columns in "pmod_values")',para_file,numel(para.pmod_names),size(para.pmod_values,2));
                            assert(size(para.onsets,1)==size(para.pmod_values,1),'Para file %s: mismatch number of parametric modulator values (%d events in "onsets", %d rows in "pmod_values")',para_file,size(para.onsets,1),size(para.pmod_values,1));
                            assert(isequal(size(para.pmod),[numel(para.pmod_names),numel(para.names)]),'Para file %s: incorrect number of field "pmod" values (expected %d rows and %d columns, found %d rows and %d columns)',para_file,numel(para.pmod_names),numel(para.names),size(para.pmod,1),size(para.pmod,2));
                            assert(numel(para.pmod_interaction)==numel(para.pmod_names),'Para file %s: mismatch number of parametric modulators (%d names in "pmod_names", %d values in "pmod_interaction")',para_file,numel(para.pmod_names),numel(para.pmod_interaction));
                            newpmod=0;
                            for n2=1:numel(para.pmod_names)
                                if para.pmod(n2,conditions(n1))>0 && (para.pmod_interaction(n2) || nnz(para.pmod(n2,conditions)>0)==1) % condition*covariate interaction
                                    newpmod=newpmod+1;
                                    matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).pmod(newpmod).name=para.pmod_names{n2};
                                    matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).pmod(newpmod).param=para.pmod_values(iconditions==n1,n2);
                                    matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).pmod(newpmod).poly=para.pmod(n2,conditions(n1));
                                    conn_disp('fprintf','      parametric modulation covariate %s order %d\n',para.pmod_names{n2},para.pmod(n2,conditions(n1)));
                                end
                            end
                        end
                    end
                end
                if isfield(para,'pmod_interaction')&&any(~para.pmod_interaction),
                    for n2=reshape(find(~para.pmod_interaction),1,[])
                        if nnz(para.pmod(n2,conditions)>0)>1 % main covariate effect only
                            newcond=newcond+1;
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).name=sprintf('pmodtemp%d',n2);
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset=[];
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).duration=[];
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).orth=max(para.orth(conditions(para.pmod(n2,conditions)>0)));
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).pmod.name=para.pmod_names{n2};
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).pmod.param=[];
                            matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).pmod.poly=max(para.pmod(n2,conditions));
                            for n1=reshape(find(para.pmod(n2,conditions)),1,[])
                                matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset=cat(1,matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset,factor*para.onsets(iconditions==n1,1));
                                matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).duration=cat(1,matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).duration,factor*para.durations(iconditions==n1));
                                matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).pmod.param=cat(1,matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).pmod.param,para.pmod_values(iconditions==n1,n2));
                            end
                            conn_disp('fprintf','   condition %s (%d blocks/events)\n',matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).name,numel(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset));
                            pmod_interaction{end+1}=sprintf('pmodtemp%d',n2);
                        end
                    end
                end
                if isfield(para,'scan')&&para.scan
                    newcond=newcond+1;
                    matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).name='scannernoise';
                    if numel(scan_times)<nses||isempty(scan_times{nses}),
                        if isfield(matlabbatch{1}.spm.stats.fmri_spec.timing,'units')&&strcmp(matlabbatch{1}.spm.stats.fmri_spec.timing.units,'secs'), matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset=(0:nscans{1}{RUNS(nses)}-1)*matlabbatch{1}.spm.stats.fmri_spec.timing.RT;
                        else matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset=(0:nscans{1}{RUNS(nses)}-1);
                        end
                    else
                        matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset=scan_times{nses};
                    end
                    matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).duration=matlabbatch{1}.spm.stats.fmri_spec.timing.RT+zeros(size(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset));
                    matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).orth=1;
                    conn_disp('fprintf','   condition %s (%d blocks/events)\n',matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).name,numel(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(newcond).onset));
                end
            end
            options=rmfield(options,'files');
        else error('field #files or #design.files not found');
        end
        if ~isfield(matlabbatch{1}.spm.stats.fmri_spec.timing,'units'), matlabbatch{1}.spm.stats.fmri_spec.timing.units='scans'; end
        isunitsscan=strcmp(matlabbatch{1}.spm.stats.fmri_spec.timing.units,'scans');
        for nses=1:NSESSIONS % warnings for onsets beyond session boundaries
            for ncond=1:numel(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond)
                if numel(scan_times)>=nses&&~isempty(scan_times{nses})&&isunitsscan, exceededonset=[]; % this should never happen
                elseif numel(scan_times)>=nses&&~isempty(scan_times{nses})&&~isunitsscan, exceededonset=find(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(ncond).onset>max(scan_times{nses})+matlabbatch{1}.spm.stats.fmri_spec.timing.RT);
                elseif isunitsscan, exceededonset=find(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(ncond).onset>nscans{1}{RUNS(nses)});
                else exceededonset=find(matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(ncond).onset>nscans{1}{RUNS(nses)}*matlabbatch{1}.spm.stats.fmri_spec.timing.RT);
                end
                for nexceededonset=exceededonset(:)', conn_disp('fprintf','Warning: condition %s onset %d %s exceeds total session #%d length (%d scans)\n',matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(ncond).name,matlabbatch{1}.spm.stats.fmri_spec.sess(nses).cond(ncond).onset,matlabbatch{1}.spm.stats.fmri_spec.timing.units,nses,round(nscans{1}{RUNS(nses)})); end
            end
        end
    end
    
    if skipmodeldefinition, load(fullfile(pwd1,'SPM.mat'),'SPM'); SPM.xY.Y={}; SPM.xY=rmfield(SPM.xY,'VY'); end
    if isempty(functional_label), files=evlab17_module('get','functionals');
    elseif isempty(conn_datasetlabel(char(functional_label))), conn_disp('fprintf','Warning: functional label %s not found in dataset. Using last preprocessed files\n',char(functional_label)); files=evlab17_module('get','functionals');
    else files=evlab17_module('get','functionals',char(functional_label));
    end
    files=files(nsubject);
    for nses=1:NSESSIONS
        filename=cellstr(char(files{1}{RUNS(nses)}));
        [filepath,filename,fileext]=cellfun(@fileparts,filename,'uni',0);
        switch(smoothinglevel)
            case 0, filename=regexprep(filename,'^s+',''); % remove all leading s
            case 1, filename=regexprep(filename,'^s+','s');% change one or several leading s to single leading s
            case 2,                                        % no change
            otherwise, error('Incorrect smoothinglevel value %d',smoothinglevel);
        end
        filename=char(cellfun(@(a,b,c)fullfile(a,[b,c]),filepath,filename,fileext,'uni',0));
        tfilename=cellstr(conn_expandframe(filename));
        if skipmodeldefinition, SPM.xY.Y=cat(1,SPM.xY.Y,reshape(tfilename,[],1));
        else matlabbatch{1}.spm.stats.fmri_spec.sess(nses).scans=tfilename;
        end
        if size(filename,1)>1, conn_disp('fprintf','Run #%d: %s to %s\n',nses,filename(1,:),filename(end,:));
        else conn_disp('fprintf','Run #%d: %s\n',nses,filename);
        end
        if nses==1&&conn_surf_dimscheck(tfilename{1}), % set implicit threhsold to -inf for surface-based data
            if skipmodeldefinition, options.mthresh=-inf;
            else matlabbatch{1}.spm.stats.fmri_spec.mthresh=-inf; 
            end
        end
    end
    if skipmodeldefinition, 
        SPM.xY.P=char(SPM.xY.Y);
        SPM.xY.VY=spm_data_hdr_read(SPM.xY.P);
        assert(numel(SPM.xY.VY)==size(SPM.xX.X,1),'Mismatched number of functional files. Found %d functionals, expected %d (from %s design)',numel(SPM.xY.VY),size(SPM.xX.X,1),fullfile(pwd1,'SPM.mat'));
        SPM.swd=pwd1;
        if isfield(SPM,'xM')&&isstruct(SPM.xM), 
            if isfield(options,'explicitmask'), SPM.xM.VM=spm_vol(char(options.explicitmask)); options=rmfield(options,'explicitmask');
            elseif isfield(SPM.xM,'VM'), SPM.xM.VM=[]; % note: disregards previous explicit mask (possibly in incorrect space now that functional data has changed)
            end 
        end
        if isfield(options,'mthresh'), SPM.xM.gMT=options.mthresh; options=rmfield(options,'mthresh'); end
        if isfield(SPM,'xVi')&&isfield(SPM.xVi,'V')&&numel(SPM.xVi.Vi)>1, SPM.xVi=rmfield(SPM.xVi,'V'); end
        if isfield(SPM,'xX')&&isfield(SPM.xX,'W'), SPM.xX=rmfield(SPM.xX,'W'); end
        SPM=evlab17_run_model_configuredesign(SPM);
        save(fullfile(pwd1,'SPM.mat'),'SPM',evlab17_module('default','mat_format')); 
    end
        
    if isfield(options,'contrasts')
        for ncon=1:numel(options.contrasts)
            str=regexp(options.contrasts{ncon},'\s+','split');
            str=str(cellfun('length',str)>0);
            m=numel(str);
            if ~(rem(m,2)&&m>1), error('incorrect number of tokens in contrast string "%s". Expected format: CONTRAST_NAME CONDITION1_NAME CONDITION1_WEIGHT CONDITION2_NAME CONDITION2_WEIGHT ...',sprintf('%s ',str{:})); end
            contname=str{1};
            condnames=str(2:2:end-1);
            try, condweights=cellfun(@str2num,str(3:2:end));
            catch, error('unable to interpret weights values %s',sprintf('%s ',str{3:2:end}));
            end
        end
    end
        
    tnames=fieldnames(options);
    if ~isempty(tnames),
        tnames=tnames(~ismember(tnames,{'model_session','contrasts','contrast_addsession','contrast_addoddeven','contrast_addcv','contrast_addevent','qa_plots','qa_plist','qa_folder','qa_parallel','qa_profile','orth','pmod','pmod_interaction','tmod','npmod'}));
        if ~isempty(tnames), conn_disp('fprintf','note: the following information fields are not used by run_model: %s\n',sprintf('%s ',tnames{:})); end
    end
       
    if ~skipmodeldefinition
        % model definition
        spm_jobman('initcfg');
        cd(pwd1);
        if ~debugskip
            save(fullfile(pwd1,'modelspecification.mat'),'matlabbatch');
            conn_disp('fprintf','Running model specification %s\n',pwd1);
            conn_setup_preproc_disp(matlabbatch);
            spm_unlink(fullfile(pwd1,'SPM.mat'));
            spm_unlink(fullfile(pwd1,'mask.nii'));
            spm_unlink(fullfile(pwd1,'mask.img'));
            job_id=spm_jobman('run',matlabbatch);
        end
        cd(pwd0);
    end
    if ~isempty(scan_times), 
        conn_disp('fprintf','Resampling task effects at specified scan-times\n');
        load(fullfile(pwd1,'SPM.mat'),'SPM');
        RT=SPM.xY.RT;
        T=SPM.xBF.T;
        T0=SPM.xBF.T0;
        for nses=1:NSESSIONS, 
            if numel(scan_times)<nses||isempty(scan_times{nses}), 
                scan_times{nses}=(0:SPM.nscan(nses)-1)*SPM.xY.RT;
            end
        end
        maxrt=RT;
        conn_disp('fprintf','Sampling offset = %.2fs\n',T0/T*RT);
        for nses=1:NSESSIONS, 
            conn_disp('fprintf','Scan-times session #%d (s) = %s\n',nses,mat2str(scan_times{nses}));
            maxrt=max(maxrt, (max(scan_times{nses})+SPM.xY.RT)/SPM.nscan(nses)); % RT if scans were uniformly sampled
        end
        SPM.xY.RT=maxrt;
        SPM.xBF.T=max(16,2*ceil(8*SPM.xY.RT)); % number of bins per "uniform-scan"
        SPM.xBF.T0=round(T0/T*RT/SPM.xY.RT*SPM.xBF.T); % sample bin
        SPM = evlab17_run_model_resample(SPM,scan_times);
        SPM.xY.RT=RT;
        SPM.xBF.T=T;
        SPM.xBF.T0=T0;
        SPM.xX.sparsesampling=true;
        save(fullfile(pwd1,'SPM.mat'),'SPM',evlab17_module('default','mat_format'));
    end
    if isfield(options,'model_session')&&(iscell(options.model_session)||~options.model_session), 
        conn_disp('fprintf','Collapsing session-specific task effects into session-invariant task effects\n');
        if isfield(options,'contrast_addsession')&&options.contrast_addsession, conn_disp('fprintf','Warning: contrast_addsession field not valid in the context of non-session-specific task effects\n'); options.contrast_addsession=0; end
        if isfield(options,'contrast_addcv')&&options.contrast_addcv, conn_disp('fprintf','Warning: contrast_addcv field not valid in the context of non-session-specific task effects\n'); options.contrast_addcv=0; end
        if isfield(options,'contrast_addoddeven')&&options.contrast_addoddeven, conn_disp('fprintf','Warning: contrast_addoddeven field not valid in the context of non-session-specific task effects\n'); options.contrast_addoddeven=0; end
        load(fullfile(pwd1,'SPM.mat'),'SPM');
        SPMcolnames=SPM.xX.name;
        S=regexp(SPMcolnames,'^Sn\((\d+)\)\s+(.*?)(\*bf\(1\))?$','tokens','once');
        SPMvalid=cellfun(@(s)isequal(size(s),[1,3]),S)';
        tS=cat(1,S{SPMvalid});
        if ~isempty(pmod_interaction), maskforce=cellfun('length',regexp(tS(:,2),'^pmodtemp\d+'))>0; else maskforce=0; end
        if iscell(options.model_session), SPMvalid(SPMvalid)=ismember(tS(:,2),options.model_session)|maskforce;
        else SPMvalid(SPMvalid)=cellfun('length',regexp(tS(:,3),'^\*bf'))>0|maskforce;
        end
        %S=regexp(SPMcolnames,'^Sn\((\d+)\)\s+(.*)\*bf\(1\)$','tokens','once');
        %SPMvalid=cellfun(@(s)isequal(size(s),[1,2]),S);
        SPMidxvalid=find(SPMvalid);
        assert(~isempty(SPMidxvalid),'No match for expected condition names. Please use #model_session condition1 condition2 ... syntax (SPM design names = %s)',sprintf('%s ',SPMcolnames{:}));
        S=cat(1,S{SPMidxvalid});
        SPMsessions=str2double(S(:,1));
        SPMconditions=S(:,2);
        SPMconditions=regexprep(SPMconditions,'\^1$','');
        SPMbasis=S(:,3);
        try, [uC,jC,iC]=unique(SPMconditions,'stable'); catch, [uC,jC,iC]=unique(SPMconditions); end
        conn_disp('fprintf','Session-invariant conditions: %s \n',sprintf('%s ',uC{:}));
        Xnew=[];
        for neff=1:numel(uC),
            idx=find(iC==neff);
            Xnew=[Xnew,sum(SPM.xX.X(:,SPMidxvalid(idx)),2)];
        end
        Xnamenew=cellfun(@(x,y)sprintf('Sn(0) %s%s',x,y),uC,SPMbasis(jC),'uni',0);
        newidx=zeros(1,size(SPM.xX.X,2)); 
        SPM.xX.X=[Xnew SPM.xX.X(:,~SPMvalid)];
        SPM.xX.name=[Xnamenew(:)',SPM.xX.name(~SPMvalid)];
        newidx(SPMvalid)=iC; newidx(~SPMvalid)=numel(uC)+(1:nnz(~SPMvalid));
        for nval={'iB','iC','iG','iH'}
            if isfield(SPM.xX,nval{1})&&~isempty(SPM.xX.(nval{1})),
                SPM.xX.(nval{1})=unique(newidx(SPM.xX.(nval{1})));
            end
        end
        if isfield(SPM.xX,'K')&&isstruct(SPM.xX.K)
            SPM.xX.K=spm_filter(SPM.xX.K,speye(size(SPM.xX.X,1)));
        end
        save(fullfile(pwd1,'SPM.mat'),'SPM',evlab17_module('default','mat_format'));
    end
    if ~isempty(pmod_interaction), 
        pmod_interaction=unique(pmod_interaction);
        conn_disp('fprintf','Removing redundant terms in parametric modulator effects\n');
        load(fullfile(pwd1,'SPM.mat'),'SPM');
        SPMcolnames=SPM.xX.name;
        S=regexp(SPMcolnames,'^Sn\((\d+)\)\s+(.*?)(\*bf\(\d+\))?$','tokens','once');
        SPMvalid=cellfun(@(s)isequal(size(s),[1,3]),S)';
        tS=cat(1,S{SPMvalid});
        SPMvalid(SPMvalid)=ismember(regexprep(tS(:,2),'\^\d+$',''),pmod_interaction);
        newidx=zeros(1,size(SPM.xX.X,2)); 
        SPM.xX.X=SPM.xX.X(:,~SPMvalid);
        SPM.xX.name=regexprep(SPM.xX.name(~SPMvalid),'pmodtemp\d+x','');
        newidx(~SPMvalid)=(1:nnz(~SPMvalid));
        for nval={'iB','iC','iG','iH'}
            if isfield(SPM.xX,nval{1})&&~isempty(SPM.xX.(nval{1})),
                temp=newidx(SPM.xX.(nval{1}));
                SPM.xX.(nval{1})=unique(temp(temp>0));
            end
        end
        if isfield(SPM.xX,'K')&&isstruct(SPM.xX.K)
            SPM.xX.K=spm_filter(SPM.xX.K,speye(size(SPM.xX.X,1)));
        end
        save(fullfile(pwd1,'SPM.mat'),'SPM',evlab17_module('default','mat_format'));
    end
    files=evlab17_module('get','spm');
    files(nsubject)={fullfile(pwd1,'SPM.mat')};
    evlab17_module('set','spm',files);
    evlab17_module('setinfo','run_model_input',options0);
    evlab17_module('save');
    cd(pwd0);
    
    % model estimation
    clear matlabbatch;
    matlabbatch{1}.spm.stats.fmri_est.spmmat={fullfile(pwd1,'SPM.mat')};
    matlabbatch{1}.spm.stats.fmri_est.method.Classical=1;
    spm_jobman('initcfg');
    cd(pwd1);
    if ~debugskip
        save(fullfile(pwd1,'modelestimation.mat'),'matlabbatch');
        conn_disp('fprintf','Running model estimation %s\n',pwd1);
        spm_unlink(fullfile(pwd1,'mask.nii'));
        spm_unlink(fullfile(pwd1,'mask.img'));
        job_id=spm_jobman('run',matlabbatch);
    end
    conn_fixpermissions(pwd1);
    cd(pwd0);
end

% contrast definition
if isfield(options,'contrasts')
    clear matlabbatch;
    load(fullfile(pwd1,'SPM.mat'),'SPM');
    SPMcolnames=SPM.xX.name;
    S=regexp(SPMcolnames,'^Sn\((\d+)\)\s+(.*?)(\*bf\(1\))?$','tokens','once');
    SPMidxvalid=find(cellfun(@(s)isequal(size(s),[1,3]),S));
    S=cat(1,S{SPMidxvalid});
    SPMsessions=str2double(S(:,1));
    SPMconditions=S(:,2);
    SPMconditions=regexprep(SPMconditions,'\^1$','');
    if ~iscell(options.contrasts), options.contrasts={options.contrasts}; end
    matlabbatch{1}.spm.stats.con.spmmat={fullfile(pwd1,'SPM.mat')};
    if isfield(options,'contrast_removeexistingcontrasts')&&~options.contrast_removeexistingcontrasts, matlabbatch{1}.spm.stats.con.delete=0;
    else matlabbatch{1}.spm.stats.con.delete=1;
    end
    matlabbatch{1}.spm.stats.con.consess={};
    order=[];
    sX=spm_sp('Set',SPM.xX.X);
    if sX.rk>0, opp=sX.v(:,[1:sX.rk])*sX.v(:,[1:sX.rk])';
    else opp=zeros( size(sX.X,2) );
    end
    estimablecols=max(abs(opp-speye(size(opp,1))),[],1)<= sX.tol;
    for ncon=1:numel(options.contrasts)
        str=regexp(options.contrasts{ncon},'\s+','split');
        str=str(cellfun('length',str)>0);
        m=numel(str);
        if ~(rem(m,2)&&m>1), error('incorrect number of tokens in contrast string "%s". Expected format: CONTRAST_NAME CONDITION1_NAME CONDITION1_WEIGHT CONDITION2_NAME CONDITION2_WEIGHT ...',sprintf('%s ',str{:})); end
        contname=str{1};
        condnames=str(2:2:end-1);
        condnames=regexprep(condnames,'\^1$','');
        try, condweights=cellfun(@str2num,str(3:2:end));
        catch, error('unable to interpret weights values %s',sprintf('%s ',str{3:2:end}));
        end
        if any(isnan(condweights)), error('unable to interpret weights values %s',sprintf('%s ',str{3:2:end})); end
        if isfield(options,'contrast_removenonestimablecols')&&options.contrast_removenonestimablecols, contrast_removenonestimablecols=true; else contrast_removenonestimablecols=false; end
        matchcol=cell(1,numel(condnames));
        contrast=zeros(1,numel(SPMcolnames));
        for ncondition=1:numel(condnames)
            i=find(strcmp(condnames{ncondition},SPMconditions));
            if isempty(i), i=find(cellfun('length',regexp(SPMconditions,['^',condnames{ncondition},'_EVENT\d+$']))); end
            if isempty(i), error('unable to match condition name %s. Valid condition names: %s',condnames{ncondition},sprintf('%s ',SPMconditions{:})); end
            if contrast_removenonestimablecols, i(~estimablecols(SPMidxvalid(i)))=[]; end
            if isempty(i), %error('no estimable effects for condition name %s',condnames{ncondition});
            else contrast(SPMidxvalid(i))=condweights(ncondition)/numel(i); % note: divides condition weight equally into all sessions/runs where this condition is present
            end
            matchcol{ncondition}=i;
        end
        matlabbatch{1}.spm.stats.con.consess{end+1}.tcon.name=contname;
        matlabbatch{1}.spm.stats.con.consess{end}.tcon.weights=contrast;
        order(end+1)=0;
        
        fh=[]; try, fh=fopen(fullfile(SPM.swd,'contrastdefinitions.stderr'),'wt'); end
        if isfield(options,'contrast_addsession')&&options.contrast_addsession, addsession=true; else addsession=false; end
        if isfield(options,'contrast_addcv')&&options.contrast_addcv, addcv=true; else addcv=false; end
        if isfield(options,'contrast_addoddeven')&&options.contrast_addoddeven, addoddeven=true; else addoddeven=false; end
        if isfield(options,'contrast_addevent')&&options.contrast_addevent, addevent=true; else addevent=false; end
        if addevent
            nevent=1; remevent=true;
            while remevent
                contrast=zeros(1,numel(SPMcolnames));
                for ncondition=1:numel(condnames)
                    i=find(strcmp(sprintf('%s_EVENT%02d',condnames{ncondition},nevent),SPMconditions));
                    if isempty(i), remevent=false; break; 
                    else contrast(1,SPMidxvalid(i))=condweights(ncondition)/numel(i); 
                    end
                end
                if ~remevent, break; end
                matlabbatch{1}.spm.stats.con.consess{end+1}.tcon.name=sprintf('%s_EVENT%02d',contname,nevent);
                matlabbatch{1}.spm.stats.con.consess{end}.tcon.weights=contrast(1,:);
                order(end+1)=1;
                if addsession
                    for nses=1:max(SPMsessions)
                        contrast=zeros(1,numel(SPMcolnames));
                        ok=true;
                        for ncondition=1:numel(condnames)
                            i=find(strcmp(sprintf('%s_EVENT%02d',condnames{ncondition},nevent),SPMconditions));
                            j=find(SPMsessions(i)==nses);
                            if ~isempty(j), contrast(1,SPMidxvalid(i(j)))=condweights(ncondition)/numel(j);
                            else ok=false;
                            end
                        end
                        if ok
                            matlabbatch{1}.spm.stats.con.consess{end+1}.tcon.name=sprintf('SESSION%02d_%s_EVENT%02d',nses,contname,nevent);
                            matlabbatch{1}.spm.stats.con.consess{end}.tcon.weights=contrast(1,:);
                            order(end+1)=2;
                        end
                    end
                end
                nevent=nevent+1;
            end
        end
        if addoddeven
            contrast=zeros(2,numel(SPMcolnames));
            for ncondition=1:numel(condnames)
                i=matchcol{ncondition};
                j=find(rem(SPMsessions(i),2)==1);
                if ~isempty(j), contrast(1,SPMidxvalid(i(j)))=condweights(ncondition)/numel(j); end
                j=find(rem(SPMsessions(i),2)==0);
                if ~isempty(j), contrast(2,SPMidxvalid(i(j)))=condweights(ncondition)/numel(j); end
            end
            if any(contrast(1,:))
                matlabbatch{1}.spm.stats.con.consess{end+1}.tcon.name=sprintf('ODD_%s',contname);
                matlabbatch{1}.spm.stats.con.consess{end}.tcon.weights=contrast(1,:);
                order(end+1)=1;
            else conn_disp('fprintf','Warning: contrast %s not estimable. skipping\n',sprintf('ODD_%s',contname)); try, fprintf(fh,'Contrast %s not estimable. skipping\n',sprintf('ODD_%s',contname)); end
            end
            if any(contrast(2,:))
                matlabbatch{1}.spm.stats.con.consess{end+1}.tcon.name=sprintf('EVEN_%s',contname);
                matlabbatch{1}.spm.stats.con.consess{end}.tcon.weights=contrast(2,:);
                order(end+1)=1;
            else conn_disp('fprintf','Warning: contrast %s not estimable. skipping\n',sprintf('EVEN_%s',contname)); try, fprintf(fh,'Contrast %s not estimable. skipping\n',sprintf('EVEN_%s',contname)); end
            end
        end
        if addcv||addsession
            for nses=1:max(SPMsessions)
                contrast=zeros(2,numel(SPMcolnames));
                for ncondition=1:numel(condnames)
                    i=matchcol{ncondition};
                    j=find(SPMsessions(i)==nses);
                    if ~isempty(j), contrast(1,SPMidxvalid(i(j)))=condweights(ncondition)/numel(j); end
                    j=find(SPMsessions(i)~=nses);
                    if ~isempty(j), contrast(2,SPMidxvalid(i(j)))=condweights(ncondition)/numel(j); end
                end
                if any(contrast(1,:))
                    matlabbatch{1}.spm.stats.con.consess{end+1}.tcon.name=sprintf('SESSION%02d_%s',nses,contname);
                    matlabbatch{1}.spm.stats.con.consess{end}.tcon.weights=contrast(1,:);
                    order(end+1)=2;
                else conn_disp('fprintf','Warning: contrast %s not estimable. skipping\n',sprintf('SESSION%02d_%s',nses,contname)); try, fprintf(fh,'Contrast %s not estimable. skipping\n',sprintf('SESSION%02d_%s',nses,contname)); end
                end
                if addcv
                    if any(contrast(2,:))
                        matlabbatch{1}.spm.stats.con.consess{end+1}.tcon.name=sprintf('ORTH_TO_SESSION%02d_%s',nses,contname);
                        matlabbatch{1}.spm.stats.con.consess{end}.tcon.weights=contrast(2,:);
                        order(end+1)=2;
                    else conn_disp('fprintf','Warning: contrast %s not estimable. skipping\n',sprintf('ORTH_TO_SESSION%02d_%s',nses,contname)); try, fprintf(fh,'Contrast %s not estimable. skipping\n',sprintf('ORTH_TO_SESSION%02d_%s',nses,contname)); end
                    end
                end
            end
        end
    end
    [nill,idx]=sort(order);
    matlabbatch{1}.spm.stats.con.consess=matlabbatch{1}.spm.stats.con.consess(idx);
    estimablecons=true(1,numel(matlabbatch{1}.spm.stats.con.consess));
    for n=1:numel(matlabbatch{1}.spm.stats.con.consess),
        c=matlabbatch{1}.spm.stats.con.consess{n}.tcon.weights';
        if ~all(all( abs(opp*c - c) <= sX.tol ))  %if ~spm_sp('isinspp',sX,c)
            conn_disp('fprintf','Warning: contrast %s cannot be uniquely estimated with the current design. skipping\n',matlabbatch{1}.spm.stats.con.consess{n}.tcon.name);
            try, fprintf(fh,'Contrast %s cannot be uniquely estimated with the current design. skipping\n',matlabbatch{1}.spm.stats.con.consess{n}.tcon.name); end
            estimablecons(n)=false;
%         else fprintf('Contrast %s ok\n',matlabbatch{1}.spm.stats.con.consess{n}.tcon.name); %,mat2str(matlabbatch{1}.spm.stats.con.consess{n}.tcon.weights));
        end
    end
    try, fclose(fh); end
    try
        fh=fopen(fullfile(SPM.swd,'contrastdefinitions.stdout'),'wt');
        for n=1:numel(matlabbatch{1}.spm.stats.con.consess),
            fprintf(fh,'Contrast %s. Contrast vector = %s @ %s\n',matlabbatch{1}.spm.stats.con.consess{n}.tcon.name,mat2str(nonzeros(matlabbatch{1}.spm.stats.con.consess{n}.tcon.weights)),mat2str(find(matlabbatch{1}.spm.stats.con.consess{n}.tcon.weights)));
        end
        fclose(fh);
    end
    fh=fopen(fullfile(pwd1,'EstimableContrasts.log'),'wt');
    if all(estimablecons), fprintf(fh,'All Contrasts are estimable\n');
    else fprintf(fh,'List of non-estimable contrasts:\n'); for nc=find(~estimablecons), fprintf(fh,'%s\n',matlabbatch{1}.spm.stats.con.consess{nc}.tcon.name); end
    end
    if all(estimablecols), fprintf(fh,'All Design Matrix columns are estimable\n');
    else fprintf(fh,'List of non-estimable effects:\n'); for nc=find(~estimablecols), fprintf(fh,'%s\n',SPM.xX.name{nc}); end
    end
    fclose(fh);
    qa_contrasts_original=matlabbatch{1}.spm.stats.con.consess;
    if ~isfield(options,'contrast_removenonestimablecontrasts')||options.contrast_removenonestimablecontrasts, addasmissingcons=[]; 
    else addasmissingcons=matlabbatch{1}.spm.stats.con.consess(~estimablecons); ;
    end
    matlabbatch{1}.spm.stats.con.consess=matlabbatch{1}.spm.stats.con.consess(estimablecons);
    if ~isempty(matlabbatch{1}.spm.stats.con.consess)
        spm_jobman('initcfg');
        cd(pwd1);
        if ~debugskip
            save(fullfile(pwd1,'contrastestimation.mat'),'matlabbatch');
            conn_disp('fprintf','Running contrast estimation %s\n',pwd1);
            if 0
                load(matlabbatch{1}.spm.stats.con.spmmat{1},'SPM');
                if matlabbatch{1}.spm.stats.con.delete, nc0=numel(SPM.xCon); else nc0=0; end
                for nc=1:numel(matlabbatch{1}.spm.stats.con.consess)
                    cname=matlabbatch{1}.spm.stats.con.consess{nc}.tcon.name;
                    c=matlabbatch{1}.spm.stats.con.consess{nc}.tcon.weights;
                    if nc==1&&matlabbatch{1}.spm.stats.con.delete, SPM.xCon = spm_FcUtil('Set',cname,'T','c',c',SPM.xX.xKXs);
                    else SPM.xCon(end+1) = spm_FcUtil('Set',cname,'T','c',c',SPM.xX.xKXs);
                    end
                end
                SPM=spm_conman(SPM,nc0+1:numel(SPM.xCon));
                SPM.xsDes='';
                save(matlabbatch{1}.spm.stats.con.spmmat{1},'SPM',evlab17_module('default','mat_format'));
            else
                job_id=spm_jobman('run',matlabbatch);
            end
        end
        conn_fixpermissions(pwd1);
        cd(pwd0);
    end
    if ~isempty(addasmissingcons), % adds NaN non-estimable-contrast placeholders
        load(fullfile(pwd1,'SPM.mat'),'SPM');
        try, state=warning; end
        warning off; % stops SPM non-estimable contrast warning messages
        for nc=1:numel(addasmissingcons)
            cname=addasmissingcons{nc}.tcon.name;
            c=addasmissingcons{nc}.tcon.weights;
            if size(c,1)>1, statsname='F'; else statsname='T'; end
            if ~isfield(SPM,'xCon')||isempty(SPM.xCon),
                idxmatch=1;
                SPM.xCon = spm_FcUtil('Set',cname,statsname,'c',c',SPM.xX.xKXs);
            else
                idxmatch=find(strcmp(cname,{SPM.xCon.name}),1);
                if isempty(idxmatch), idxmatch=numel(SPM.xCon)+1; end
                SPM.xCon(idxmatch) = spm_FcUtil('Set',cname,statsname,'c',c',SPM.xX.xKXs);
            end
            V=struct('mat',SPM.xY.VY(1).mat,'dim',SPM.xY.VY(1).dim,'pinfo',[1;0;0],'dt',[spm_type('float32') spm_platform('bigend')],'fname',fullfile(pwd1,sprintf('con_%04d.nii',idxmatch)),'descrip','CONNlabel:MissingData');
            SPM.xCon(idxmatch).Vcon=spm_write_vol(V,nan(V.dim));
            V=struct('mat',SPM.xY.VY(1).mat,'dim',SPM.xY.VY(1).dim,'pinfo',[1;0;0],'dt',[spm_type('float32') spm_platform('bigend')],'fname',fullfile(pwd1,sprintf('spm%s_%04d.nii',statsname,idxmatch)),'descrip','CONNlabel:MissingData');
            SPM.xCon(idxmatch).Vspm=spm_write_vol(V,nan(V.dim));
        end
        warning on;
        try, warning(state); end
        save(fullfile(pwd1,'SPM.mat'),'SPM',evlab17_module('default','mat_format'));
   end
    if ~isempty(qa_contrasts_original) % keesp track of original contrasts (before removal of non-estimable contrasts) for QA plots
        load(fullfile(pwd1,'SPM.mat'),'SPM');
        for nc=1:numel(qa_contrasts_original)
            xcon=struct('name',qa_contrasts_original{nc}.tcon.name,'c',qa_contrasts_original{nc}.tcon.weights');
            if nc==1, SPM.xConOriginal=xcon;
            else SPM.xConOriginal(nc)=xcon;
            end
        end
        save(fullfile(pwd1,'SPM.mat'),'SPM',evlab17_module('default','mat_format'));
    end
end

% design QA plots
if isfield(options,'qa_plots'), qa_plots=options.qa_plots;
else qa_plots=evlab17_module('default','qa_plots');
end
if all(qa_plots>0)
    fileout=evlab17_module('filename');
    if skipmodelestimation
        files=evlab17_module('get','spm');
        files(nsubject)={fullfile(pwd1,'SPM.mat')};
        evlab17_module('set','spm',files);
        evlab17_module('save',fileout);
    end
    namesQA=names(cellfun('length',regexp(names,'^qa_'))>0);
    optionsQA=struct('qa_plist','firstlevel','subjects',nsubject);
    for n=1:numel(namesQA)
        optionsQA.(namesQA{n})=options.(namesQA{n});
    end
    if isfield(options,'parallel'),qaoptions.parallel=options.parallel;end
    evlab17_run_qa(optionsQA,[],'dataset',fileout); 
%     if isfield(options,'qa_parallel'), qa_parallel=options.qa_parallel;
%     else qa_parallel=evlab17_module('default','qa_parallel');
%     end
%     if isfield(options,'qa_profile'), qa_profile=options.qa_profile;
%     else qa_profile=evlab17_module('default','qa_profile');
%     end
%     if qa_parallel, conn_batch('QA.foldername',qafolder,'QA.plots',[21,22,23],'subjects',nsubject,'filename',evlab17_module('filename'),'parallel.N',qa_parallel,'parallel.profile',qa_profile);
%     else conn_batch('QA.foldername',qafolder,'QA.plots',[21,22,23],'subjects',nsubject);
%     end
    %conn_qaplots(qafolder,[21,22,23],nsubject);
    conn_fixpermissions(qafolder);
end

end



function SPM=evlab17_run_model_configuredesign(SPM) % scaling & threshold configuration (from SPM12 spm_fmri_spm_ui.m)
VY    = SPM.xY.VY;
nscan = SPM.nscan;
nsess = length(nscan);
%-Compute Global variate
%==========================================================================
GM    = 100;
q     = length(VY);
g     = zeros(q,1);
if spm_mesh_detect(VY)
    for i = 1:q
        dat = spm_data_read(VY(i));
        g(i) = mean(dat(~isnan(dat)));
    end
else
    for i = 1:q
        g(i) = spm_global(VY(i));
    end
end
%-Scale if specified (otherwise session specific grand mean scaling)
%--------------------------------------------------------------------------
gSF   = GM./g;
if strcmpi(SPM.xGX.iGXcalc,'none')
    for i = 1:nsess
        gSF(SPM.Sess(i).row) = GM./mean(g(SPM.Sess(i).row));
    end
end
%-Apply gSF to memory-mapped scalefactors to implement scaling
%--------------------------------------------------------------------------
for i = 1:q
    SPM.xY.VY(i).pinfo(1:2,:) = SPM.xY.VY(i).pinfo(1:2,:) * gSF(i);
    if spm_mesh_detect(VY)
        SPM.xY.VY(i).private.private.data{1}.data.scl_slope = ...
            SPM.xY.VY(i).private.private.data{1}.data.scl_slope * gSF(i);
        SPM.xY.VY(i).private.private.data{1}.data.scl_inter = ...
            SPM.xY.VY(i).private.private.data{1}.data.scl_inter * gSF(i);
    else
        SPM.xY.VY(i).private.dat.scl_slope = ...
            SPM.xY.VY(i).private.dat.scl_slope * gSF(i);
        SPM.xY.VY(i).private.dat.scl_inter = ...
            SPM.xY.VY(i).private.dat.scl_inter * gSF(i);
    end
end
%-Place global variates in xGX
%--------------------------------------------------------------------------
SPM.xGX.sGXcalc = 'mean voxel value';
SPM.xGX.sGMsca  = 'session specific';
SPM.xGX.rg      = g;
SPM.xGX.GM      = GM;
SPM.xGX.gSF     = gSF;
%-Masking
%==========================================================================
%-Masking threshold, as proportion of globals
%--------------------------------------------------------------------------
try
    gMT = SPM.xM.gMT;
catch
    gMT = spm_get_defaults('mask.thresh');
end
TH = g.*gSF*gMT;
try
    VM=SPM.xM.VM;
catch
	VM=[];
end
%-Place masking structure in xM
%--------------------------------------------------------------------------
SPM.xM = struct(...
    'T',   ones(q,1),...
    'TH',  TH,...
    'gMT', gMT,...
    'I',   0,...
    'VM',  VM,...
    'xs',  struct('Masking','analysis threshold'));
end


function [SPM] = evlab17_run_model_resample(SPM,scantimes)  % resample design at non-uniform sample times (from SPM12 spm_fMRI_design.m)
if nargin<2||isempty(scantimes), scantimes={}; end
if ~iscell(scantimes), scantimes={scantimes}; end % scantimes: cell array, scan times (in seconds, starting at 0) for each session

SVNid = '$Rev: 7739 $';


%-Construct Design matrix {X}
%==========================================================================

%-Microtime onset and microtime resolution
%--------------------------------------------------------------------------
try
    fMRI_T     = SPM.xBF.T;
    fMRI_T0    = SPM.xBF.T0;
catch
    fMRI_T     = spm_get_defaults('stats.fmri.t');
    fMRI_T0    = spm_get_defaults('stats.fmri.t0');
    SPM.xBF.T  = fMRI_T;
    SPM.xBF.T0 = fMRI_T0;
end

%-Time units, dt = time bin {secs}
%--------------------------------------------------------------------------
SPM.xBF.dt     = SPM.xY.RT/SPM.xBF.T;

%-Get basis functions
%--------------------------------------------------------------------------
SPM.xBF        = spm_get_bf(SPM.xBF);


%-Get session specific design parameters
%==========================================================================
Xx    = [];
Xb    = [];
Xname = {};
Bname = {};
for s = 1:length(SPM.nscan)
    %-Number of scans for this session
    %----------------------------------------------------------------------
    k = SPM.nscan(s);
    
    
    %-Create convolved stimulus functions or inputs
    %======================================================================
    
    %-Get inputs, neuronal causes or stimulus functions U
    %----------------------------------------------------------------------
    U = spm_get_ons(SPM,s);
    
    %-Convolve stimulus functions with basis functions
    %----------------------------------------------------------------------
    [X,Xn,Fc] = spm_Volterra(U, SPM.xBF.bf, SPM.xBF.Volterra);
    
    %-Resample regressors at acquisition times (32 bin offset)
    %----------------------------------------------------------------------
    if ~isempty(X)
        if numel(scantimes)<s||isempty(scantimes{s}), scantimes{s}=SPM.xY.RT*(0:k - 1); end
        X = X(round(scantimes{s}/SPM.xY.RT*fMRI_T) + fMRI_T0 + 32,:);
        %X = X((0:(k - 1))*fMRI_T + fMRI_T0 + 32,:);
    end
    
    %-Orthogonalise (within trial type)
    %----------------------------------------------------------------------
    for i = 1:length(Fc)
        if i<= numel(U) && ... % for Volterra kernels
                (~isfield(U(i),'orth') || U(i).orth)
            p = ones(size(Fc(i).i));
        else
            p = Fc(i).p;
        end
        for j = 1:max(p)
            X(:,Fc(i).i(p==j)) = spm_orth(X(:,Fc(i).i(p==j)));
        end
    end
    
    
    %-Get user specified regressors
    %======================================================================
    C     = SPM.Sess(s).C.C;
    Cname = SPM.Sess(s).C.name;
    
    %-Append mean-corrected regressors and names
    %----------------------------------------------------------------------
    X     = [X spm_detrend(C)];
    Xn    = {Xn{:} Cname{:}};
    
    
    %-Confounds: Session effects
    %======================================================================
    B     = ones(k,1);
    Bn    = {'constant'};
    
    
    %-Session structure array
    %======================================================================
    SPM.Sess(s).U      = U;
    SPM.Sess(s).C.C    = C;
    SPM.Sess(s).C.name = Cname;
    SPM.Sess(s).row    = size(Xb,1) + (1:k);
    SPM.Sess(s).col    = size(Xx,2) + (1:size(X,2));
    SPM.Sess(s).Fc     = Fc;
    
    
    %-Append into Xx and Xb
    %======================================================================
    Xx      = blkdiag(Xx,X);
    Xb      = blkdiag(Xb,B);
    
    %-Append names
    %----------------------------------------------------------------------
    for i = 1:length(Xn)
        Xname{end + 1} = [sprintf('Sn(%i) ',s) Xn{i}];
    end
    for i = 1:length(Bn)
        Bname{end + 1} = [sprintf('Sn(%i) ',s) Bn{i}];
    end
    
end


%-Place design matrix structure in xX
%==========================================================================
SPM.xX.X    = [Xx Xb];
SPM.xX.iH   = [];
SPM.xX.iC   = 1:size(Xx,2);
SPM.xX.iB   = (1:size(Xb,2)) + size(Xx,2);
SPM.xX.iG   = [];
SPM.xX.name = {Xname{:} Bname{:}};

end
